"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.requestNewAuthToken = exports.getAuthToken = void 0;
const opengroups_1 = require("../../../../data/opengroups");
const util_worker_interface_1 = require("../../../../webworker/workers/util_worker_interface");
const Promise_1 = require("../../../utils/Promise");
const String_1 = require("../../../utils/String");
const User_1 = require("../../../utils/User");
const OpenGroupAPIV2_1 = require("./OpenGroupAPIV2");
const OpenGroupAPIV2Parser_1 = require("./OpenGroupAPIV2Parser");
async function claimAuthToken(authToken, serverUrl, roomId) {
    const headers = { Authorization: authToken };
    const request = {
        method: 'POST',
        headers,
        room: roomId,
        server: serverUrl,
        queryParams: { public_key: (0, User_1.getOurPubKeyStrFromCache)() },
        isAuthRequired: false,
        endpoint: 'claim_auth_token',
    };
    const result = await (0, OpenGroupAPIV2_1.sendApiV2Request)(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        window?.log?.warn(`Could not claim token, status code: ${statusCode}`);
        return null;
    }
    return authToken;
}
async function oneAtATimeGetAuth({ serverUrl, roomId }) {
    return (0, Promise_1.allowOnlyOneAtATime)(`getAuthToken${serverUrl}:${roomId}`, async () => {
        try {
            const roomDetails = await (0, opengroups_1.getV2OpenGroupRoomByRoomId)({ serverUrl, roomId });
            if (!roomDetails) {
                window?.log?.warn('getAuthToken Room does not exist.');
                return null;
            }
            if (roomDetails?.token) {
                return roomDetails.token;
            }
            window?.log?.info(`Triggering getAuthToken with serverUrl:'${serverUrl}'; roomId: '${roomId}'`);
            const token = await requestNewAuthToken({ serverUrl, roomId });
            if (!token) {
                window?.log?.warn('invalid new auth token', token);
                return;
            }
            window?.log?.info(`Got AuthToken for serverUrl:'${serverUrl}'; roomId: '${roomId}'`);
            const claimedToken = await claimAuthToken(token, serverUrl, roomId);
            if (!claimedToken) {
                window?.log?.warn('Failed to claim token', claimedToken);
            }
            else {
                window?.log?.info(`Claimed AuthToken for serverUrl:'${serverUrl}'; roomId: '${roomId}'`);
            }
            roomDetails.token = claimedToken || '';
            await (0, opengroups_1.saveV2OpenGroupRoom)(roomDetails);
            window?.log?.info(`AuthToken saved to DB for serverUrl:'${serverUrl}'; roomId: '${roomId}'`);
            return claimedToken;
        }
        catch (e) {
            window?.log?.error('Failed to getAuthToken', e);
            throw e;
        }
    });
}
async function getAuthToken({ serverUrl, roomId, }) {
    return oneAtATimeGetAuth({ roomId, serverUrl });
}
exports.getAuthToken = getAuthToken;
async function requestNewAuthToken({ serverUrl, roomId, }) {
    const userKeyPair = await (0, User_1.getIdentityKeyPair)();
    if (!userKeyPair) {
        throw new Error('Failed to fetch user keypair');
    }
    const ourPubkey = (0, User_1.getOurPubKeyStrFromCache)();
    const parameters = {};
    parameters.public_key = ourPubkey;
    const request = {
        method: 'GET',
        room: roomId,
        server: serverUrl,
        queryParams: parameters,
        isAuthRequired: false,
        endpoint: 'auth_token_challenge',
    };
    const json = (await (0, OpenGroupAPIV2_1.sendApiV2Request)(request));
    if (!json || !json?.result?.challenge) {
        window?.log?.warn('Parsing failed');
        return null;
    }
    const { ciphertext: base64EncodedCiphertext, ephemeral_public_key: base64EncodedEphemeralPublicKey, } = json?.result?.challenge;
    if (!base64EncodedCiphertext || !base64EncodedEphemeralPublicKey) {
        window?.log?.warn('Parsing failed');
        return null;
    }
    const ciphertext = (await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', base64EncodedCiphertext));
    const ephemeralPublicKey = (await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', base64EncodedEphemeralPublicKey));
    try {
        const symmetricKey = (await (0, util_worker_interface_1.callUtilsWorker)('deriveSymmetricKey', new Uint8Array(ephemeralPublicKey), new Uint8Array(userKeyPair.privKey)));
        const plaintextBuffer = await (0, util_worker_interface_1.callUtilsWorker)('DecryptAESGCM', new Uint8Array(symmetricKey), new Uint8Array(ciphertext));
        const token = (0, String_1.toHex)(plaintextBuffer);
        return token;
    }
    catch (e) {
        window?.log?.error('Failed to decrypt token Social group v2');
        return null;
    }
}
exports.requestNewAuthToken = requestNewAuthToken;
