"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadDefaultRooms = exports.parseMessages = void 0;
const lodash_1 = __importDefault(require("lodash"));
const types_1 = require("../../../types");
const Promise_1 = require("../../../utils/Promise");
const defaultRooms_1 = require("../../../../state/ducks/defaultRooms");
const util_1 = require("../../../../util");
const OpenGroupUtils_1 = require("../utils/OpenGroupUtils");
const JoinOpenGroupV2_1 = require("./JoinOpenGroupV2");
const OpenGroupAPIV2_1 = require("./OpenGroupAPIV2");
const OpenGroupMessageV2_1 = require("./OpenGroupMessageV2");
const util_worker_interface_1 = require("../../../../webworker/workers/util_worker_interface");
const parseMessages = async (rawMessages) => {
    if (!rawMessages || rawMessages.length === 0) {
        return [];
    }
    const startParse = Date.now();
    const opengroupMessagesSignatureUnchecked = lodash_1.default.compact(rawMessages.map(rawMessage => {
        try {
            const opengroupv2Message = OpenGroupMessageV2_1.OpenGroupMessageV2.fromJson(rawMessage);
            if (!opengroupv2Message?.serverId ||
                !opengroupv2Message.sentTimestamp ||
                !opengroupv2Message.base64EncodedData ||
                !opengroupv2Message.base64EncodedSignature) {
                window?.log?.warn('invalid Social group message received');
                return null;
            }
            const sender = types_1.PubKey.cast(opengroupv2Message.sender).withoutPrefix();
            return { opengroupv2Message, sender };
        }
        catch (e) {
            window.log.warn('an error happened with opengroup message', e);
            return null;
        }
    }));
    window.log.debug(`[perf] parseMessage took ${Date.now() - startParse}ms`);
    const sentToWorker = opengroupMessagesSignatureUnchecked.map(m => {
        return {
            sender: m.sender,
            base64EncodedSignature: m.opengroupv2Message.base64EncodedSignature,
            base64EncodedData: m.opengroupv2Message.base64EncodedData,
        };
    });
    const startVerify = Date.now();
    const signatureValidEncodedData = (await (0, util_worker_interface_1.callUtilsWorker)('verifyAllSignatures', sentToWorker));
    window.log.info(`[perf] verifyAllSignatures took ${Date.now() - startVerify}ms.`);
    const parsedMessages = opengroupMessagesSignatureUnchecked
        .filter(m => signatureValidEncodedData.includes(m.opengroupv2Message.base64EncodedData))
        .map(m => m.opengroupv2Message);
    return lodash_1.default.compact(parsedMessages.map(m => m && m.sender && !util_1.BlockedNumberController.isBlocked(m.sender) ? m : null)).sort((a, b) => (a.serverId || 0) - (b.serverId || 0));
};
exports.parseMessages = parseMessages;
const defaultServerUrl = 'http://social.beldex.io';
const defaultServerPublicKey = '0cfdbcc8bba5989a6787019c6635c08415c103174609360f9c3e4e764ef48073';
const defaultRoom = `${defaultServerUrl}/main?public_key=${defaultServerPublicKey}`;
const loadDefaultRoomsSingle = () => (0, Promise_1.allowOnlyOneAtATime)('loadDefaultRoomsSingle', async () => {
    const roomInfos = (0, JoinOpenGroupV2_1.parseOpenGroupV2)(defaultRoom);
    if (roomInfos) {
        try {
            const roomsGot = await (0, OpenGroupAPIV2_1.getAllRoomInfos)(roomInfos);
            if (!roomsGot) {
                return [];
            }
            return roomsGot.map(room => {
                return {
                    ...room,
                    completeUrl: (0, OpenGroupUtils_1.getCompleteUrlFromRoom)({
                        serverUrl: roomInfos.serverUrl,
                        serverPublicKey: roomInfos.serverPublicKey,
                        roomId: room.id,
                    }),
                };
            });
        }
        catch (e) {
            window?.log?.warn('loadDefaultRoomloadDefaultRoomssIfNeeded failed', e);
        }
        return [];
    }
    return [];
});
const loadDefaultRooms = async () => {
    window.inboxStore?.dispatch((0, defaultRooms_1.updateDefaultRoomsInProgress)(true));
    const allRooms = await loadDefaultRoomsSingle();
    window.inboxStore?.dispatch((0, defaultRooms_1.updateDefaultRoomsInProgress)(false));
    if (allRooms !== undefined) {
        window.inboxStore?.dispatch((0, defaultRooms_1.updateDefaultRooms)(allRooms));
    }
};
exports.loadDefaultRooms = loadDefaultRooms;
