"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenGroupManagerV2 = exports.getOpenGroupManager = void 0;
const opengroups_1 = require("../../../../data/opengroups");
const conversation_1 = require("../../../../models/conversation");
const conversations_1 = require("../../../conversations");
const Promise_1 = require("../../../utils/Promise");
const OpenGroupUtils_1 = require("../utils/OpenGroupUtils");
const OpenGroupAPIV2_1 = require("./OpenGroupAPIV2");
const OpenGroupServerPoller_1 = require("./OpenGroupServerPoller");
const lodash_1 = __importDefault(require("lodash"));
const auto_bind_1 = __importDefault(require("auto-bind"));
let instance;
const getOpenGroupManager = () => {
    if (!instance) {
        instance = new OpenGroupManagerV2();
    }
    return instance;
};
exports.getOpenGroupManager = getOpenGroupManager;
class OpenGroupManagerV2 {
    static useV2OpenGroups = false;
    pollers = new Map();
    isPolling = false;
    constructor() {
        (0, auto_bind_1.default)(this);
    }
    async attemptConnectionV2OneAtATime(serverUrl, roomId, publicKey) {
        const oneAtaTimeStr = `oneAtaTimeOpenGroupV2Join:${serverUrl}${roomId}`;
        return (0, Promise_1.allowOnlyOneAtATime)(oneAtaTimeStr, async () => {
            return this.attemptConnectionV2(serverUrl, roomId, publicKey);
        });
    }
    async startPolling() {
        await (0, Promise_1.allowOnlyOneAtATime)('V2ManagerStartPolling', this.startPollingBouncy);
    }
    stopPolling() {
        if (!this.isPolling) {
            return;
        }
        this.pollers.forEach(poller => {
            poller.stop();
        });
        this.pollers.clear();
        this.isPolling = false;
    }
    addRoomToPolledRooms(roomInfos) {
        const grouped = lodash_1.default.groupBy(roomInfos, r => r.serverUrl);
        const groupedArray = Object.values(grouped);
        for (const groupedRooms of groupedArray) {
            const groupedRoomsServerUrl = groupedRooms[0].serverUrl;
            const poller = this.pollers.get(groupedRoomsServerUrl);
            if (!poller) {
                const uniqGroupedRooms = lodash_1.default.uniqBy(groupedRooms, r => r.roomId);
                this.pollers.set(groupedRoomsServerUrl, new OpenGroupServerPoller_1.OpenGroupServerPoller(uniqGroupedRooms));
            }
            else {
                roomInfos.forEach(poller.addRoomToPoll);
            }
        }
    }
    removeRoomFromPolledRooms(roomInfos) {
        const poller = this.pollers.get(roomInfos.serverUrl);
        if (!poller) {
            return;
        }
        poller.removeRoomFromPoll(roomInfos);
        if (poller.getPolledRoomsCount() === 0) {
            this.pollers.delete(roomInfos.serverUrl);
            poller.stop();
        }
    }
    async startPollingBouncy() {
        if (this.isPolling) {
            return;
        }
        const allConvos = await (0, opengroups_1.getAllOpenGroupV2Conversations)();
        let allRoomInfos = await (0, opengroups_1.getAllV2OpenGroupRooms)();
        if (allRoomInfos) {
            await Promise.all([...allRoomInfos.values()].map(async (infos) => {
                try {
                    const roomConvoId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(infos.serverUrl, infos.roomId);
                    if (!allConvos.get(roomConvoId)) {
                        await (0, opengroups_1.removeV2OpenGroupRoom)(roomConvoId);
                        (0, exports.getOpenGroupManager)().removeRoomFromPolledRooms(infos);
                    }
                }
                catch (e) {
                    window?.log?.warn('cleanup roomInfos error', e);
                }
            }));
        }
        allRoomInfos = await (0, opengroups_1.getAllV2OpenGroupRooms)();
        if (allRoomInfos) {
            this.addRoomToPolledRooms([...allRoomInfos.values()]);
        }
        this.isPolling = true;
    }
    async attemptConnectionV2(serverUrl, roomId, serverPublicKey) {
        const conversationId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(serverUrl, roomId);
        if ((0, conversations_1.getConversationController)().get(conversationId)) {
            throw new Error(window.i18n('publicChatExists'));
        }
        await (0, opengroups_1.removeV2OpenGroupRoom)(conversationId);
        const room = {
            serverUrl,
            roomId,
            conversationId,
            serverPublicKey,
        };
        try {
            await (0, opengroups_1.saveV2OpenGroupRoom)(room);
            const roomInfos = await (0, OpenGroupAPIV2_1.openGroupV2GetRoomInfo)({ roomId, serverUrl });
            if (!roomInfos) {
                throw new Error('Invalid Social group roomInfo result');
            }
            const conversation = await (0, conversations_1.getConversationController)().getOrCreateAndWait(conversationId, conversation_1.ConversationTypeEnum.GROUP);
            room.imageID = roomInfos.imageId || undefined;
            room.roomName = roomInfos.name || undefined;
            await (0, opengroups_1.saveV2OpenGroupRoom)(room);
            conversation.set({
                active_at: Date.now(),
                name: room.roomName,
                avatarPath: room.roomName,
                isApproved: true,
            });
            await conversation.commit();
            this.addRoomToPolledRooms([room]);
            return conversation;
        }
        catch (e) {
            window?.log?.warn('Failed to join Social group v2', e.message);
            await (0, opengroups_1.removeV2OpenGroupRoom)(conversationId);
            return undefined;
        }
    }
}
exports.OpenGroupManagerV2 = OpenGroupManagerV2;
