"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TEST_fetchSnodePoolFromSeedNodeRetryable = exports.getMinTimeout = exports.fetchSnodePoolFromSeedNodeWithRetries = void 0;
const __1 = require("../..");
const node_fetch_1 = __importDefault(require("node-fetch"));
const https_1 = __importDefault(require("https"));
const lodash_1 = __importDefault(require("lodash"));
const tls_1 = __importDefault(require("tls"));
const crypto_1 = require("../../crypto");
const p_retry_1 = __importDefault(require("p-retry"));
const _1 = require(".");
const Promise_1 = require("../../utils/Promise");
async function fetchSnodePoolFromSeedNodeWithRetries(seedNodes) {
    try {
        window?.log?.info(`fetchSnodePoolFromSeedNode with seedNodes.length ${seedNodes.length}`);
        let snodes = await getSnodeListFromSeednodeOneAtAtime(seedNodes);
        snodes = lodash_1.default.shuffle(snodes);
        const fetchSnodePool = snodes.map(snode => ({
            ip: snode.public_ip,
            port: snode.storage_port,
            pubkey_x25519: snode.pubkey_x25519,
            pubkey_ed25519: snode.pubkey_ed25519,
        }));
        window?.log?.info('SeedNodeAPI::fetchSnodePoolFromSeedNodeWithRetries - Refreshed random snode pool with', snodes.length, 'mnodes');
        return fetchSnodePool;
    }
    catch (e) {
        window?.log?.warn('BchatSnodeAPI::fetchSnodePoolFromSeedNodeWithRetries - error', e.code, e.message);
        throw new Error('Failed to contact seed node');
    }
}
exports.fetchSnodePoolFromSeedNodeWithRetries = fetchSnodePoolFromSeedNodeWithRetries;
const getSslAgentForSeedNode = async (seedNodeHost, isSsl = false) => {
    let certContent = '';
    let pubkey256 = '';
    let cert256 = '';
    if (!isSsl) {
        return undefined;
    }
    switch (seedNodeHost) {
        case 'publicnode1.rpcnode.stream':
            certContent = Buffer.from(storageSeed1Crt, 'utf-8').toString();
            pubkey256 =
                'C5+lpZ7tcVwmwQIMcRtPbsQtWLABXhQzejna0wHFr8M=';
            cert256 =
                '6D:99:FB:26:5E:B1:C5:B3:74:47:65:FC:BC:64:8F:3C:D8:E1:BF:FA:FD:C4:C2:F9:9B:9D:47:CF:7F:F1:C2:4F';
            break;
        case 'publicnode2.rpcnode.stream':
            certContent = Buffer.from(storageSeed3Crt, 'utf-8').toString();
            pubkey256 = 'C5+lpZ7tcVwmwQIMcRtPbsQtWLABXhQzejna0wHFr8M=';
            cert256 =
                '6D:99:FB:26:5E:B1:C5:B3:74:47:65:FC:BC:64:8F:3C:D8:E1:BF:FA:FD:C4:C2:F9:9B:9D:47:CF:7F:F1:C2:4F';
            break;
        case 'publicnode3.rpcnode.stream':
            certContent = Buffer.from(publicBeldexFoundationCtr, 'utf-8').toString();
            pubkey256 = 'C5+lpZ7tcVwmwQIMcRtPbsQtWLABXhQzejna0wHFr8M=';
            cert256 =
                '6D:99:FB:26:5E:B1:C5:B3:74:47:65:FC:BC:64:8F:3C:D8:E1:BF:FA:FD:C4:C2:F9:9B:9D:47:CF:7F:F1:C2:4F';
            break;
        case 'publicnode4.rpcnode.stream':
            certContent = Buffer.from(publicBeldexFoundationCtr, 'utf-8').toString();
            pubkey256 = 'C5+lpZ7tcVwmwQIMcRtPbsQtWLABXhQzejna0wHFr8M=';
            cert256 =
                '6D:99:FB:26:5E:B1:C5:B3:74:47:65:FC:BC:64:8F:3C:D8:E1:BF:FA:FD:C4:C2:F9:9B:9D:47:CF:7F:F1:C2:4F';
            break;
        case 'publicnode5.rpcnode.stream':
            console.log(storageSeed3Crt, storageSeed1Crt);
            certContent = Buffer.from(publicBeldexFoundationCtr, 'utf-8').toString();
            pubkey256 = 'C5+lpZ7tcVwmwQIMcRtPbsQtWLABXhQzejna0wHFr8M=';
            cert256 =
                '6D:99:FB:26:5E:B1:C5:B3:74:47:65:FC:BC:64:8F:3C:D8:E1:BF:FA:FD:C4:C2:F9:9B:9D:47:CF:7F:F1:C2:4F';
            break;
        default:
            throw new Error(`Unknown seed node: ${seedNodeHost}`);
    }
    const sslOptions = {
        ca: certContent,
        rejectUnauthorized: false,
        keepAlive: false,
        checkServerIdentity: (host, cert) => {
            const err = tls_1.default.checkServerIdentity(host, cert);
            if (err) {
                return err;
            }
            if ((0, crypto_1.sha256)(cert.pubkey) !== pubkey256) {
                const msg = 'Certificate verification error: ' +
                    `The public key of '${cert.subject.CN}' ` +
                    'does not match our pinned fingerprint';
                return new Error(msg);
            }
            if (cert.fingerprint256 !== cert256) {
                const msg = 'Certificate verification error: ' +
                    `The certificate of '${cert.subject.CN}' ` +
                    'does not match our pinned fingerprint';
                return new Error(msg);
            }
            return undefined;
        },
    };
    return new https_1.default.Agent(sslOptions);
};
const getSnodeListFromSeednodeOneAtAtime = async (seedNodes) => (0, Promise_1.allowOnlyOneAtATime)('getSnodeListFromSeednode', () => getSnodeListFromSeednode(seedNodes));
async function getSnodeListFromSeednode(seedNodes) {
    const SEED_NODE_RETRIES = 4;
    return (0, p_retry_1.default)(async () => {
        window?.log?.info('getSnodeListFromSeednode starting...');
        if (!seedNodes.length) {
            window?.log?.info('beldex_mnode_api::getSnodeListFromSeednode - seedNodes are empty');
            throw new Error('getSnodeListFromSeednode - seedNodes are empty');
        }
        const snodes = await _1.SeedNodeAPI.TEST_fetchSnodePoolFromSeedNodeRetryable(seedNodes);
        return snodes;
    }, {
        retries: SEED_NODE_RETRIES - 1,
        factor: 2,
        minTimeout: _1.SeedNodeAPI.getMinTimeout(),
        onFailedAttempt: e => {
            window?.log?.warn(`fetchSnodePoolFromSeedNodeRetryable attempt #${e.attemptNumber} failed. ${e.retriesLeft} retries left... Error: ${e.message}`);
        },
    });
}
function getMinTimeout() {
    return 1000;
}
exports.getMinTimeout = getMinTimeout;
async function TEST_fetchSnodePoolFromSeedNodeRetryable(seedNodes) {
    window?.log?.info('fetchSnodePoolFromSeedNodeRetryable starting...');
    if (!seedNodes.length) {
        window?.log?.info('beldex_mnode_api::fetchSnodePoolFromSeedNodeRetryable - seedNodes are empty');
        throw new Error('fetchSnodePoolFromSeedNodeRetryable: Seed nodes are empty');
    }
    const seedNodeUrl = lodash_1.default.sample(seedNodes);
    if (!seedNodeUrl) {
        window?.log?.warn('beldex_mnode_api::fetchSnodePoolFromSeedNodeRetryable - Could not select random snodes from', seedNodes);
        throw new Error('fetchSnodePoolFromSeedNodeRetryable: Seed nodes are empty #2');
    }
    const tryUrl = new URL(seedNodeUrl);
    const snodes = await getSnodesFromSeedUrl(tryUrl);
    if (snodes.length === 0) {
        window?.log?.warn(`beldex_mnode_api::fetchSnodePoolFromSeedNodeRetryable - ${seedNodeUrl} did not return any snodes`);
        throw new Error(`Failed to contact seed node: ${seedNodeUrl}`);
    }
    return snodes;
}
exports.TEST_fetchSnodePoolFromSeedNodeRetryable = TEST_fetchSnodePoolFromSeedNodeRetryable;
async function getSnodesFromSeedUrl(urlObj) {
    window?.log?.info(`getSnodesFromSeedUrl starting with ${urlObj.href}`);
    const params = {
        active_only: true,
        ours_only: true,
        fields: {
            public_ip: true,
            storage_port: true,
            pubkey_x25519: true,
            pubkey_ed25519: true,
        },
    };
    const endpoint = 'json_rpc';
    const url = `${urlObj.href}${endpoint}`;
    const body = {
        jsonrpc: '2.0',
        id: '0',
        method: 'get_n_master_nodes',
        params,
    };
    const sslAgent = await getSslAgentForSeedNode(urlObj.hostname, urlObj.protocol !== __1.Constants.PROTOCOLS.HTTP);
    const fetchOptions = {
        method: 'POST',
        timeout: 5000,
        body: JSON.stringify(body),
        headers: {
            'User-Agent': 'WhatsApp',
            'Accept-Language': 'en-us',
        },
        agent: sslAgent,
    };
    window?.log?.info('insecureNodeFetch => plaintext for getSnodesFromSeedUrl');
    const response = await (0, node_fetch_1.default)(url, fetchOptions);
    if (response.status !== 200) {
        window?.log?.error(`beldex_mnode_api:::getSnodesFromSeedUrl - invalid response from seed ${urlObj.toString()}:`, response);
        throw new Error(`getSnodesFromSeedUrl: status is not 200 ${response.status} from ${urlObj.href}`);
    }
    if (response.headers.get('Content-Type') !== 'application/json') {
        window?.log?.error('Response is not json');
        throw new Error(`getSnodesFromSeedUrl: response is not json Content-Type from ${urlObj.href}`);
    }
    try {
        const json = await response.json();
        const result = json.result;
        if (!result) {
            window?.log?.error(`beldex_mnode_api:::getSnodesFromSeedUrl - invalid result from seed ${urlObj.toString()}:`, response);
            throw new Error(`getSnodesFromSeedUrl: json.result is empty from ${urlObj.href}`);
        }
        const validNodes = result.master_node_states.filter((snode) => snode.public_ip !== '0.0.0.0');
        if (validNodes.length === 0) {
            throw new Error(`Did not get a single valid snode from ${urlObj.href}`);
        }
        return validNodes;
    }
    catch (e) {
        window?.log?.error('Invalid json response');
        throw new Error(`getSnodesFromSeedUrl: cannot parse content as JSON from ${urlObj.href}`);
    }
}
const storageSeed1Crt = `-----BEGIN CERTIFICATE-----
MIIFYDCCBEigAwIBAgIQQAF3ITfU6UK47naqPGQKtzANBgkqhkiG9w0BAQsFADA/
MSQwIgYDVQQKExtEaWdpdGFsIFNpZ25hdHVyZSBUcnVzdCBDby4xFzAVBgNVBAMT
DkRTVCBSb290IENBIFgzMB4XDTIxMDEyMDE5MTQwM1oXDTI0MDkzMDE4MTQwM1ow
TzELMAkGA1UEBhMCVVMxKTAnBgNVBAoTIEludGVybmV0IFNlY3VyaXR5IFJlc2Vh
cmNoIEdyb3VwMRUwEwYDVQQDEwxJU1JHIFJvb3QgWDEwggIiMA0GCSqGSIb3DQEB
AQUAA4ICDwAwggIKAoICAQCt6CRz9BQ385ueK1coHIe+3LffOJCMbjzmV6B493XC
ov71am72AE8o295ohmxEk7axY/0UEmu/H9LqMZshftEzPLpI9d1537O4/xLxIZpL
wYqGcWlKZmZsj348cL+tKSIG8+TA5oCu4kuPt5l+lAOf00eXfJlII1PoOK5PCm+D
LtFJV4yAdLbaL9A4jXsDcCEbdfIwPPqPrt3aY6vrFk/CjhFLfs8L6P+1dy70sntK
4EwSJQxwjQMpoOFTJOwT2e4ZvxCzSow/iaNhUd6shweU9GNx7C7ib1uYgeGJXDR5
bHbvO5BieebbpJovJsXQEOEO3tkQjhb7t/eo98flAgeYjzYIlefiN5YNNnWe+w5y
sR2bvAP5SQXYgd0FtCrWQemsAXaVCg/Y39W9Eh81LygXbNKYwagJZHduRze6zqxZ
Xmidf3LWicUGQSk+WT7dJvUkyRGnWqNMQB9GoZm1pzpRboY7nn1ypxIFeFntPlF4
FQsDj43QLwWyPntKHEtzBRL8xurgUBN8Q5N0s8p0544fAQjQMNRbcTa0B7rBMDBc
SLeCO5imfWCKoqMpgsy6vYMEG6KDA0Gh1gXxG8K28Kh8hjtGqEgqiNx2mna/H2ql
PRmP6zjzZN7IKw0KKP/32+IVQtQi0Cdd4Xn+GOdwiK1O5tmLOsbdJ1Fu/7xk9TND
TwIDAQABo4IBRjCCAUIwDwYDVR0TAQH/BAUwAwEB/zAOBgNVHQ8BAf8EBAMCAQYw
SwYIKwYBBQUHAQEEPzA9MDsGCCsGAQUFBzAChi9odHRwOi8vYXBwcy5pZGVudHJ1
c3QuY29tL3Jvb3RzL2RzdHJvb3RjYXgzLnA3YzAfBgNVHSMEGDAWgBTEp7Gkeyxx
+tvhS5B1/8QVYIWJEDBUBgNVHSAETTBLMAgGBmeBDAECATA/BgsrBgEEAYLfEwEB
ATAwMC4GCCsGAQUFBwIBFiJodHRwOi8vY3BzLnJvb3QteDEubGV0c2VuY3J5cHQu
b3JnMDwGA1UdHwQ1MDMwMaAvoC2GK2h0dHA6Ly9jcmwuaWRlbnRydXN0LmNvbS9E
U1RST09UQ0FYM0NSTC5jcmwwHQYDVR0OBBYEFHm0WeZ7tuXkAXOACIjIGlj26Ztu
MA0GCSqGSIb3DQEBCwUAA4IBAQAKcwBslm7/DlLQrt2M51oGrS+o44+/yQoDFVDC
5WxCu2+b9LRPwkSICHXM6webFGJueN7sJ7o5XPWioW5WlHAQU7G75K/QosMrAdSW
9MUgNTP52GE24HGNtLi1qoJFlcDyqSMo59ahy2cI2qBDLKobkx/J3vWraV0T9VuG
WCLKTVXkcGdtwlfFRjlBz4pYg1htmf5X6DYO8A4jqv2Il9DjXA6USbW1FzXSLr9O
he8Y4IWS6wY7bCkjCWDcRQJMEhg76fsO3txE+FiYruq9RUWhiF1myv4Q6W+CyBFC
Dfvp7OOGAN6dEOM4+qR9sdjoSYKEBpsr6GtPAQw4dy753ec5
-----END CERTIFICATE-----
`;
const storageSeed3Crt = `-----BEGIN CERTIFICATE-----
MIIFYDCCBEigAwIBAgIQQAF3ITfU6UK47naqPGQKtzANBgkqhkiG9w0BAQsFADA/
MSQwIgYDVQQKExtEaWdpdGFsIFNpZ25hdHVyZSBUcnVzdCBDby4xFzAVBgNVBAMT
DkRTVCBSb290IENBIFgzMB4XDTIxMDEyMDE5MTQwM1oXDTI0MDkzMDE4MTQwM1ow
TzELMAkGA1UEBhMCVVMxKTAnBgNVBAoTIEludGVybmV0IFNlY3VyaXR5IFJlc2Vh
cmNoIEdyb3VwMRUwEwYDVQQDEwxJU1JHIFJvb3QgWDEwggIiMA0GCSqGSIb3DQEB
AQUAA4ICDwAwggIKAoICAQCt6CRz9BQ385ueK1coHIe+3LffOJCMbjzmV6B493XC
ov71am72AE8o295ohmxEk7axY/0UEmu/H9LqMZshftEzPLpI9d1537O4/xLxIZpL
wYqGcWlKZmZsj348cL+tKSIG8+TA5oCu4kuPt5l+lAOf00eXfJlII1PoOK5PCm+D
LtFJV4yAdLbaL9A4jXsDcCEbdfIwPPqPrt3aY6vrFk/CjhFLfs8L6P+1dy70sntK
4EwSJQxwjQMpoOFTJOwT2e4ZvxCzSow/iaNhUd6shweU9GNx7C7ib1uYgeGJXDR5
bHbvO5BieebbpJovJsXQEOEO3tkQjhb7t/eo98flAgeYjzYIlefiN5YNNnWe+w5y
sR2bvAP5SQXYgd0FtCrWQemsAXaVCg/Y39W9Eh81LygXbNKYwagJZHduRze6zqxZ
Xmidf3LWicUGQSk+WT7dJvUkyRGnWqNMQB9GoZm1pzpRboY7nn1ypxIFeFntPlF4
FQsDj43QLwWyPntKHEtzBRL8xurgUBN8Q5N0s8p0544fAQjQMNRbcTa0B7rBMDBc
SLeCO5imfWCKoqMpgsy6vYMEG6KDA0Gh1gXxG8K28Kh8hjtGqEgqiNx2mna/H2ql
PRmP6zjzZN7IKw0KKP/32+IVQtQi0Cdd4Xn+GOdwiK1O5tmLOsbdJ1Fu/7xk9TND
TwIDAQABo4IBRjCCAUIwDwYDVR0TAQH/BAUwAwEB/zAOBgNVHQ8BAf8EBAMCAQYw
SwYIKwYBBQUHAQEEPzA9MDsGCCsGAQUFBzAChi9odHRwOi8vYXBwcy5pZGVudHJ1
c3QuY29tL3Jvb3RzL2RzdHJvb3RjYXgzLnA3YzAfBgNVHSMEGDAWgBTEp7Gkeyxx
+tvhS5B1/8QVYIWJEDBUBgNVHSAETTBLMAgGBmeBDAECATA/BgsrBgEEAYLfEwEB
ATAwMC4GCCsGAQUFBwIBFiJodHRwOi8vY3BzLnJvb3QteDEubGV0c2VuY3J5cHQu
b3JnMDwGA1UdHwQ1MDMwMaAvoC2GK2h0dHA6Ly9jcmwuaWRlbnRydXN0LmNvbS9E
U1RST09UQ0FYM0NSTC5jcmwwHQYDVR0OBBYEFHm0WeZ7tuXkAXOACIjIGlj26Ztu
MA0GCSqGSIb3DQEBCwUAA4IBAQAKcwBslm7/DlLQrt2M51oGrS+o44+/yQoDFVDC
5WxCu2+b9LRPwkSICHXM6webFGJueN7sJ7o5XPWioW5WlHAQU7G75K/QosMrAdSW
9MUgNTP52GE24HGNtLi1qoJFlcDyqSMo59ahy2cI2qBDLKobkx/J3vWraV0T9VuG
WCLKTVXkcGdtwlfFRjlBz4pYg1htmf5X6DYO8A4jqv2Il9DjXA6USbW1FzXSLr9O
he8Y4IWS6wY7bCkjCWDcRQJMEhg76fsO3txE+FiYruq9RUWhiF1myv4Q6W+CyBFC
Dfvp7OOGAN6dEOM4+qR9sdjoSYKEBpsr6GtPAQw4dy753ec5
-----END CERTIFICATE-----
`;
const publicBeldexFoundationCtr = `-----BEGIN CERTIFICATE-----
MIIFYDCCBEigAwIBAgIQQAF3ITfU6UK47naqPGQKtzANBgkqhkiG9w0BAQsFADA/
MSQwIgYDVQQKExtEaWdpdGFsIFNpZ25hdHVyZSBUcnVzdCBDby4xFzAVBgNVBAMT
DkRTVCBSb290IENBIFgzMB4XDTIxMDEyMDE5MTQwM1oXDTI0MDkzMDE4MTQwM1ow
TzELMAkGA1UEBhMCVVMxKTAnBgNVBAoTIEludGVybmV0IFNlY3VyaXR5IFJlc2Vh
cmNoIEdyb3VwMRUwEwYDVQQDEwxJU1JHIFJvb3QgWDEwggIiMA0GCSqGSIb3DQEB
AQUAA4ICDwAwggIKAoICAQCt6CRz9BQ385ueK1coHIe+3LffOJCMbjzmV6B493XC
ov71am72AE8o295ohmxEk7axY/0UEmu/H9LqMZshftEzPLpI9d1537O4/xLxIZpL
wYqGcWlKZmZsj348cL+tKSIG8+TA5oCu4kuPt5l+lAOf00eXfJlII1PoOK5PCm+D
LtFJV4yAdLbaL9A4jXsDcCEbdfIwPPqPrt3aY6vrFk/CjhFLfs8L6P+1dy70sntK
4EwSJQxwjQMpoOFTJOwT2e4ZvxCzSow/iaNhUd6shweU9GNx7C7ib1uYgeGJXDR5
bHbvO5BieebbpJovJsXQEOEO3tkQjhb7t/eo98flAgeYjzYIlefiN5YNNnWe+w5y
sR2bvAP5SQXYgd0FtCrWQemsAXaVCg/Y39W9Eh81LygXbNKYwagJZHduRze6zqxZ
Xmidf3LWicUGQSk+WT7dJvUkyRGnWqNMQB9GoZm1pzpRboY7nn1ypxIFeFntPlF4
FQsDj43QLwWyPntKHEtzBRL8xurgUBN8Q5N0s8p0544fAQjQMNRbcTa0B7rBMDBc
SLeCO5imfWCKoqMpgsy6vYMEG6KDA0Gh1gXxG8K28Kh8hjtGqEgqiNx2mna/H2ql
PRmP6zjzZN7IKw0KKP/32+IVQtQi0Cdd4Xn+GOdwiK1O5tmLOsbdJ1Fu/7xk9TND
TwIDAQABo4IBRjCCAUIwDwYDVR0TAQH/BAUwAwEB/zAOBgNVHQ8BAf8EBAMCAQYw
SwYIKwYBBQUHAQEEPzA9MDsGCCsGAQUFBzAChi9odHRwOi8vYXBwcy5pZGVudHJ1
c3QuY29tL3Jvb3RzL2RzdHJvb3RjYXgzLnA3YzAfBgNVHSMEGDAWgBTEp7Gkeyxx
+tvhS5B1/8QVYIWJEDBUBgNVHSAETTBLMAgGBmeBDAECATA/BgsrBgEEAYLfEwEB
ATAwMC4GCCsGAQUFBwIBFiJodHRwOi8vY3BzLnJvb3QteDEubGV0c2VuY3J5cHQu
b3JnMDwGA1UdHwQ1MDMwMaAvoC2GK2h0dHA6Ly9jcmwuaWRlbnRydXN0LmNvbS9E
U1RST09UQ0FYM0NSTC5jcmwwHQYDVR0OBBYEFHm0WeZ7tuXkAXOACIjIGlj26Ztu
MA0GCSqGSIb3DQEBCwUAA4IBAQAKcwBslm7/DlLQrt2M51oGrS+o44+/yQoDFVDC
5WxCu2+b9LRPwkSICHXM6webFGJueN7sJ7o5XPWioW5WlHAQU7G75K/QosMrAdSW
9MUgNTP52GE24HGNtLi1qoJFlcDyqSMo59ahy2cI2qBDLKobkx/J3vWraV0T9VuG
WCLKTVXkcGdtwlfFRjlBz4pYg1htmf5X6DYO8A4jqv2Il9DjXA6USbW1FzXSLr9O
he8Y4IWS6wY7bCkjCWDcRQJMEhg76fsO3txE+FiYruq9RUWhiF1myv4Q6W+CyBFC
Dfvp7OOGAN6dEOM4+qR9sdjoSYKEBpsr6GtPAQw4dy753ec5
-----END CERTIFICATE-----
 `;
