"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.snodeRpc = void 0;
const node_fetch_1 = __importDefault(require("node-fetch"));
const p_retry_1 = __importDefault(require("p-retry"));
const errors_1 = require("../../utils/errors");
const types_1 = require("../../types");
const onions_1 = require("./onions");
async function bchatFetch({ options, url, associatedWith, targetNode, timeout, }) {
    const method = options.method || 'GET';
    const fetchOptions = {
        ...options,
        timeout,
        method,
    };
    try {
        const useOnionRequests = window.bchatFeatureFlags?.useOnionRequests === undefined
            ? true
            : window.bchatFeatureFlags?.useOnionRequests;
        if (useOnionRequests && targetNode) {
            const fetchResult = await (0, onions_1.bchatOnionFetch)({
                targetNode,
                body: fetchOptions.body,
                associatedWith,
            });
            if (!fetchResult) {
                return undefined;
            }
            return fetchResult;
        }
        if (url.match(/https:\/\//)) {
            fetchOptions.agent = onions_1.snodeHttpsAgent;
        }
        fetchOptions.headers = {
            'User-Agent': 'WhatsApp',
            'Accept-Language': 'en-us',
        };
        window?.log?.warn(`insecureNodeFetch => bchatFetch of ${url}`);
        const response = await (0, node_fetch_1.default)(url, fetchOptions);
        if (!response.ok) {
            throw new errors_1.HTTPError('beldex_rpc error', response);
        }
        const result = await response.text();
        return {
            body: result,
            status: response.status,
        };
    }
    catch (e) {
        if (e.code === 'ENOTFOUND') {
            throw new errors_1.NotFoundError('Failed to resolve address', e);
        }
        if (e.message === onions_1.ERROR_421_HANDLED_RETRY_REQUEST) {
            throw new p_retry_1.default.AbortError(onions_1.ERROR_421_HANDLED_RETRY_REQUEST);
        }
        throw e;
    }
}
async function snodeRpc({ method, params, targetNode, associatedWith, timeout = 10000, }) {
    const url = `https://${targetNode.ip}:${targetNode.port}/storage_rpc/v1`;
    if (params.pubKey) {
        params = {
            ...params,
            pubKey: (0, types_1.getStoragePubKey)(params.pubKey),
        };
    }
    const body = {
        jsonrpc: '2.0',
        id: '0',
        method,
        params,
    };
    const fetchOptions = {
        method: 'POST',
        body: JSON.stringify(body),
        headers: {
            'Content-Type': 'application/json',
        },
    };
    return bchatFetch({
        url,
        options: fetchOptions,
        targetNode,
        associatedWith,
        timeout,
    });
}
exports.snodeRpc = snodeRpc;
