"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.bchatOnionFetch = exports.getPathString = exports.sendOnionRequestHandlingSnodeEject = exports.incrementBadSnodeCountOrDrop = exports.snodeHttpsAgent = exports.processOnionResponse = exports.decodeOnionResult = exports.CLOCK_OUT_OF_SYNC_MESSAGE_ERROR = exports.ERROR_421_HANDLED_RETRY_REQUEST = exports.NEXT_NODE_NOT_FOUND_PREFIX = exports.OXEN_SERVER_ERROR = exports.resetSnodeFailureCount = void 0;
const node_fetch_1 = __importDefault(require("node-fetch"));
const https_1 = __importDefault(require("https"));
const snodePool_1 = require("./snodePool");
const bytebuffer_1 = __importDefault(require("bytebuffer"));
const onions_1 = require("../../onions");
const String_1 = require("../../utils/String");
const p_retry_1 = __importDefault(require("p-retry"));
const onionPath_1 = require("../../onions/onionPath");
const lodash_1 = __importDefault(require("lodash"));
let snodeFailureCount = {};
const SNodeAPI_1 = require("./SNodeAPI");
const _1 = require(".");
const PnServer_1 = require("../push_notification_api/PnServer");
const util_worker_interface_1 = require("../../../webworker/workers/util_worker_interface");
const resetSnodeFailureCount = () => {
    snodeFailureCount = {};
};
exports.resetSnodeFailureCount = resetSnodeFailureCount;
const snodeFailureThreshold = 3;
exports.OXEN_SERVER_ERROR = 'Oxen Server error';
exports.NEXT_NODE_NOT_FOUND_PREFIX = 'Next node not found: ';
exports.ERROR_421_HANDLED_RETRY_REQUEST = '421 handled. Retry this request with a new targetNode';
exports.CLOCK_OUT_OF_SYNC_MESSAGE_ERROR = 'Your clock is out of sync with the network. Check your clock.';
async function encryptForPubKey(pubKeyX25519hex, reqObj) {
    const reqStr = JSON.stringify(reqObj);
    const textEncoder = new TextEncoder();
    const plaintext = textEncoder.encode(reqStr);
    return (0, util_worker_interface_1.callUtilsWorker)('encryptForPubkey', pubKeyX25519hex, plaintext);
}
async function encryptForRelayV2(relayX25519hex, destination, ctx) {
    if (!destination.host && !destination.destination) {
        window?.log?.warn('beldex_rpc::encryptForRelayV2 - no destination', destination);
    }
    const reqObj = {
        ...destination,
        ephemeral_key: (0, String_1.toHex)(ctx.ephemeralKey),
    };
    const plaintext = encodeCiphertextPlusJson(ctx.ciphertext, reqObj);
    return (0, util_worker_interface_1.callUtilsWorker)('encryptForPubkey', relayX25519hex, plaintext);
}
function encodeCiphertextPlusJson(ciphertext, payloadJson) {
    const payloadStr = JSON.stringify(payloadJson);
    const bufferJson = bytebuffer_1.default.wrap(payloadStr, 'utf8');
    const len = ciphertext.length;
    const arrayLen = bufferJson.buffer.length + 4 + len;
    const littleEndian = true;
    const buffer = new bytebuffer_1.default(arrayLen, littleEndian);
    buffer.writeInt32(len);
    buffer.append(ciphertext);
    buffer.append(bufferJson);
    return new Uint8Array(buffer.buffer);
}
async function buildOnionCtxs(nodePath, destCtx, targetED25519Hex, finalRelayOptions) {
    const ctxes = [destCtx];
    if (!nodePath) {
        throw new Error('buildOnionCtxs needs a valid path');
    }
    const firstPos = nodePath.length - 1;
    for (let i = firstPos; i > -1; i -= 1) {
        let dest;
        const relayingToFinalDestination = i === firstPos;
        if (relayingToFinalDestination && finalRelayOptions) {
            let target = '/beldex/v2/lsrpc';
            const isCallToPn = finalRelayOptions?.host === PnServer_1.hrefPnServerProd;
            if (!isCallToPn) {
                target = '/beldex/v3/lsrpc';
            }
            dest = {
                host: finalRelayOptions.host,
                target,
                method: 'POST',
            };
            if (finalRelayOptions?.protocol === 'http') {
                dest.protocol = finalRelayOptions.protocol;
                dest.port = finalRelayOptions.port || 80;
            }
        }
        else {
            let pubkeyHex = targetED25519Hex;
            if (!relayingToFinalDestination) {
                pubkeyHex = nodePath[i + 1].pubkey_ed25519;
                if (!pubkeyHex) {
                    window?.log?.error('beldex_rpc:::buildOnionGuardNodePayload - no ed25519 for', nodePath[i + 1], 'path node', i + 1);
                }
            }
            dest = {
                destination: pubkeyHex,
            };
        }
        try {
            const ctx = await encryptForRelayV2(nodePath[i].pubkey_x25519, dest, ctxes[ctxes.length - 1]);
            ctxes.push(ctx);
        }
        catch (e) {
            window?.log?.error('beldex_rpc:::buildOnionGuardNodePayload - encryptForRelayV2 failure', e.code, e.message);
            throw e;
        }
    }
    return ctxes;
}
async function buildOnionGuardNodePayload(nodePath, destCtx, targetED25519Hex, finalRelayOptions) {
    const ctxes = await buildOnionCtxs(nodePath, destCtx, targetED25519Hex, finalRelayOptions);
    const guardCtx = ctxes[ctxes.length - 1];
    const guardPayloadObj = {
        ephemeral_key: (0, String_1.toHex)(guardCtx.ephemeralKey),
    };
    return encodeCiphertextPlusJson(guardCtx.ciphertext, guardPayloadObj);
}
function process406Error(statusCode) {
    if (statusCode === 406) {
        throw new p_retry_1.default.AbortError(exports.CLOCK_OUT_OF_SYNC_MESSAGE_ERROR);
    }
}
function processOxenServerError(_statusCode, body) {
    if (body === exports.OXEN_SERVER_ERROR) {
        window?.log?.warn('[path] Got Oxen server Error. Not much to do if the server has troubles.');
        throw new p_retry_1.default.AbortError(exports.OXEN_SERVER_ERROR);
    }
}
async function process421Error(statusCode, body, associatedWith, lsrpcEd25519Key) {
    if (statusCode === 421) {
        await handle421InvalidSwarm({
            snodeEd25519: lsrpcEd25519Key,
            body,
            associatedWith,
        });
    }
}
async function processOnionRequestErrorAtDestination({ statusCode, body, destinationEd25519, associatedWith, }) {
    if (statusCode === 200) {
        return;
    }
    window?.log?.info(`processOnionRequestErrorAtDestination. statusCode nok: ${statusCode}: "${body}"`);
    process406Error(statusCode);
    await process421Error(statusCode, body, associatedWith, destinationEd25519);
    processOxenServerError(statusCode, body);
    if (destinationEd25519) {
        await processAnyOtherErrorAtDestination(statusCode, body, destinationEd25519, associatedWith);
    }
}
async function handleNodeNotFound({ ed25519NotFound, associatedWith, }) {
    const shortNodeNotFound = (0, onionPath_1.ed25519Str)(ed25519NotFound);
    window?.log?.warn('Handling NODE NOT FOUND with: ', shortNodeNotFound);
    if (associatedWith) {
        await (0, snodePool_1.dropSnodeFromSwarmIfNeeded)(associatedWith, ed25519NotFound);
    }
    await (0, snodePool_1.dropSnodeFromSnodePool)(ed25519NotFound);
    snodeFailureCount[ed25519NotFound] = 0;
    await onions_1.OnionPaths.dropSnodeFromPath(ed25519NotFound);
}
async function processAnyOtherErrorOnPath(status, guardNodeEd25519, ciphertext, associatedWith) {
    if (status !== 200) {
        window?.log?.warn(`[path] Got status: ${status}`);
        if (ciphertext?.startsWith(exports.NEXT_NODE_NOT_FOUND_PREFIX)) {
            const nodeNotFound = ciphertext.substr(exports.NEXT_NODE_NOT_FOUND_PREFIX.length);
            await handleNodeNotFound({ ed25519NotFound: nodeNotFound, associatedWith });
        }
        else {
            await (0, onionPath_1.incrementBadPathCountOrDrop)(guardNodeEd25519);
        }
        processOxenServerError(status, ciphertext);
        throw new Error(`Bad Path handled. Retry this request. Status: ${status}`);
    }
}
async function processAnyOtherErrorAtDestination(status, body, destinationEd25519, associatedWith) {
    if (status !== 400 &&
        status !== 406 &&
        status !== 421) {
        window?.log?.warn(`[path] Got status at destination: ${status}`);
        if (body?.startsWith(exports.NEXT_NODE_NOT_FOUND_PREFIX)) {
            const nodeNotFound = body.substr(exports.NEXT_NODE_NOT_FOUND_PREFIX.length);
            await handleNodeNotFound({
                ed25519NotFound: nodeNotFound,
                associatedWith,
            });
            throw new p_retry_1.default.AbortError(`Bad Path handled. Retry this request with another targetNode. Status: ${status}`);
        }
        await _1.Onions.incrementBadSnodeCountOrDrop({
            snodeEd25519: destinationEd25519,
            associatedWith,
        });
        throw new Error(`Bad Path handled. Retry this request. Status: ${status}`);
    }
}
async function processOnionRequestErrorOnPath(httpStatusCode, ciphertext, guardNodeEd25519, lsrpcEd25519Key, associatedWith) {
    if (httpStatusCode !== 200) {
        window?.log?.warn('errorONpath:', ciphertext);
    }
    process406Error(httpStatusCode);
    await process421Error(httpStatusCode, ciphertext, associatedWith, lsrpcEd25519Key);
    await processAnyOtherErrorOnPath(httpStatusCode, guardNodeEd25519, ciphertext, associatedWith);
}
function processAbortedRequest(abortSignal) {
    if (abortSignal?.aborted) {
        window?.log?.warn('[path] Call aborted');
        throw new p_retry_1.default.AbortError('Request got aborted');
    }
}
const debug = false;
async function decodeOnionResult(symmetricKey, ciphertext) {
    let parsedCiphertext = ciphertext;
    try {
        const jsonRes = JSON.parse(ciphertext);
        parsedCiphertext = jsonRes.result;
    }
    catch (e) {
    }
    const ciphertextBuffer = await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', parsedCiphertext);
    const plaintextBuffer = (await (0, util_worker_interface_1.callUtilsWorker)('DecryptAESGCM', new Uint8Array(symmetricKey), new Uint8Array(ciphertextBuffer)));
    return { plaintext: new TextDecoder().decode(plaintextBuffer), ciphertextBuffer };
}
exports.decodeOnionResult = decodeOnionResult;
const STATUS_NO_STATUS = 8888;
async function processOnionResponse({ response, symmetricKey, guardNode, abortSignal, associatedWith, lsrpcEd25519Key, }) {
    let ciphertext = '';
    processAbortedRequest(abortSignal);
    try {
        ciphertext = (await response?.text()) || '';
    }
    catch (e) {
        window?.log?.warn(e);
    }
    await processOnionRequestErrorOnPath(response?.status || STATUS_NO_STATUS, ciphertext, guardNode.pubkey_ed25519, lsrpcEd25519Key, associatedWith);
    if (!ciphertext) {
        window?.log?.warn('[path] bchatRpc::processingOnionResponse - Target node return empty ciphertext');
        throw new Error('Target node return empty ciphertext');
    }
    let plaintext;
    let ciphertextBuffer;
    try {
        const decoded = await exports.decodeOnionResult(symmetricKey, ciphertext);
        plaintext = decoded.plaintext;
        ciphertextBuffer = decoded.ciphertextBuffer;
    }
    catch (e) {
        window?.log?.error('[path] bchatRpc::processingOnionResponse - decode error', e);
        if (symmetricKey) {
            window?.log?.error('[path] bchatRpc::processingOnionResponse - symmetricKey', (0, String_1.toHex)(symmetricKey));
        }
        if (ciphertextBuffer) {
            window?.log?.error('[path] bchatRpc::processingOnionResponse - ciphertextBuffer', (0, String_1.toHex)(ciphertextBuffer));
        }
        throw new Error('Ciphertext decode error');
    }
    if (debug) {
        window?.log?.debug('bchatRpc::processingOnionResponse - plaintext', plaintext);
    }
    try {
        const jsonRes = JSON.parse(plaintext, (_key, value) => {
            if (typeof value === 'number' && value > Number.MAX_SAFE_INTEGER) {
                window?.log?.warn('Received an out of bounds js number');
            }
            return value;
        });
        const status = jsonRes.status_code || jsonRes.status;
        await processOnionRequestErrorAtDestination({
            statusCode: status,
            body: jsonRes?.body,
            destinationEd25519: lsrpcEd25519Key,
            associatedWith,
        });
        return jsonRes;
    }
    catch (e) {
        window?.log?.error(`[path] bchatRpc::processingOnionResponse - Rethrowing error ${e.message}'`);
        throw e;
    }
}
exports.processOnionResponse = processOnionResponse;
exports.snodeHttpsAgent = new https_1.default.Agent({
    rejectUnauthorized: false,
});
async function handle421InvalidSwarm({ body, snodeEd25519, associatedWith, }) {
    if (!snodeEd25519 || !associatedWith) {
        throw new Error('status 421 without a final destination or no associatedWith makes no sense');
    }
    window?.log?.info(`Invalidating swarm for ${(0, onionPath_1.ed25519Str)(associatedWith)}`);
    try {
        const parsedBody = JSON.parse(body);
        if (parsedBody?.mnodes?.length) {
            window?.log?.warn('Wrong swarm, now looking at snodes', parsedBody.mnodes.map((s) => (0, onionPath_1.ed25519Str)(s.pubkey_ed25519)));
            await (0, snodePool_1.updateSwarmFor)(associatedWith, parsedBody.mnodes);
            throw new p_retry_1.default.AbortError(exports.ERROR_421_HANDLED_RETRY_REQUEST);
        }
        await (0, snodePool_1.dropSnodeFromSwarmIfNeeded)(associatedWith, snodeEd25519);
    }
    catch (e) {
        if (e.message !== exports.ERROR_421_HANDLED_RETRY_REQUEST) {
            window?.log?.warn('Got error while parsing 421 result. Dropping this snode from the swarm of this pubkey', e);
            await (0, snodePool_1.dropSnodeFromSwarmIfNeeded)(associatedWith, snodeEd25519);
        }
    }
    await _1.Onions.incrementBadSnodeCountOrDrop({ snodeEd25519, associatedWith });
    throw new p_retry_1.default.AbortError(exports.ERROR_421_HANDLED_RETRY_REQUEST);
}
async function incrementBadSnodeCountOrDrop({ snodeEd25519, associatedWith, }) {
    const oldFailureCount = snodeFailureCount[snodeEd25519] || 0;
    const newFailureCount = oldFailureCount + 1;
    snodeFailureCount[snodeEd25519] = newFailureCount;
    if (newFailureCount >= snodeFailureThreshold) {
        window?.log?.warn(`Failure threshold reached for snode: ${(0, onionPath_1.ed25519Str)(snodeEd25519)}; dropping it.`);
        if (associatedWith) {
            await (0, snodePool_1.dropSnodeFromSwarmIfNeeded)(associatedWith, snodeEd25519);
        }
        await (0, snodePool_1.dropSnodeFromSnodePool)(snodeEd25519);
        snodeFailureCount[snodeEd25519] = 0;
        await onions_1.OnionPaths.dropSnodeFromPath(snodeEd25519);
    }
    else {
        window?.log?.warn(`Couldn't reach snode at: ${(0, onionPath_1.ed25519Str)(snodeEd25519)}; setting his failure count to ${newFailureCount}`);
    }
}
exports.incrementBadSnodeCountOrDrop = incrementBadSnodeCountOrDrop;
const sendOnionRequestHandlingSnodeEject = async ({ destX25519Any, finalDestOptions, nodePath, abortSignal, associatedWith, finalRelayOptions, }) => {
    let response;
    let decodingSymmetricKey;
    try {
        const result = await sendOnionRequest({
            nodePath,
            destX25519Any,
            finalDestOptions,
            finalRelayOptions,
            abortSignal,
        });
        response = result.response;
        if (!lodash_1.default.isEmpty(finalRelayOptions) &&
            response.status === 502 &&
            response.statusText === 'Bad Gateway') {
            throw new p_retry_1.default.AbortError('ENETUNREACH');
        }
        decodingSymmetricKey = result.decodingSymmetricKey;
    }
    catch (e) {
        window?.log?.warn('sendOnionRequest error message: ', e.message);
        if (e.code === 'ENETUNREACH' || e.message === 'ENETUNREACH') {
            throw e;
        }
    }
    const processed = await processOnionResponse({
        response,
        symmetricKey: decodingSymmetricKey,
        guardNode: nodePath[0],
        lsrpcEd25519Key: finalDestOptions?.destination_ed25519_hex,
        abortSignal,
        associatedWith,
    });
    return processed;
};
exports.sendOnionRequestHandlingSnodeEject = sendOnionRequestHandlingSnodeEject;
const sendOnionRequest = async ({ nodePath, destX25519Any, finalDestOptions, finalRelayOptions, abortSignal, }) => {
    let destX25519hex = destX25519Any;
    const copyFinalDestOptions = lodash_1.default.cloneDeep(finalDestOptions);
    if (typeof destX25519hex !== 'string') {
        window?.log?.warn('destX25519hex was not a string');
        destX25519hex = (0, String_1.toHex)(destX25519Any);
    }
    let targetEd25519hex;
    if (copyFinalDestOptions.destination_ed25519_hex) {
        targetEd25519hex = copyFinalDestOptions.destination_ed25519_hex;
        delete copyFinalDestOptions.destination_ed25519_hex;
    }
    const options = copyFinalDestOptions;
    options.headers = options.headers || {};
    const isLsrpc = !!finalRelayOptions;
    let destCtx;
    try {
        if (!isLsrpc) {
            const body = options.body || '';
            delete options.body;
            const textEncoder = new TextEncoder();
            const bodyEncoded = textEncoder.encode(body);
            const plaintext = encodeCiphertextPlusJson(bodyEncoded, options);
            destCtx = (await (0, util_worker_interface_1.callUtilsWorker)('encryptForPubkey', destX25519hex, plaintext));
        }
        else {
            destCtx = await encryptForPubKey(destX25519hex, options);
        }
    }
    catch (e) {
        window?.log?.error('beldex_rpc::sendOnionRequest - encryptForPubKey failure [', e.code, e.message, '] destination X25519', destX25519hex.substr(0, 32), '...', destX25519hex.substr(32), 'options', options);
        throw e;
    }
    const payload = await buildOnionGuardNodePayload(nodePath, destCtx, targetEd25519hex, finalRelayOptions);
    const guardNode = nodePath[0];
    const guardFetchOptions = {
        method: 'POST',
        body: payload,
        agent: exports.snodeHttpsAgent,
        headers: {
            'User-Agent': 'WhatsApp',
            'Accept-Language': 'en-us',
        },
        timeout: 25000,
    };
    if (abortSignal) {
        guardFetchOptions.signal = abortSignal;
    }
    const guardUrl = `https://${guardNode.ip}:${guardNode.port}/onion_req/v2`;
    const response = await (0, node_fetch_1.default)(guardUrl, guardFetchOptions);
    return { response, decodingSymmetricKey: destCtx.symmetricKey };
};
async function sendOnionRequestSnodeDest(onionPath, targetNode, plaintext, associatedWith) {
    return (0, exports.sendOnionRequestHandlingSnodeEject)({
        nodePath: onionPath,
        destX25519Any: targetNode.pubkey_x25519,
        finalDestOptions: {
            destination_ed25519_hex: targetNode.pubkey_ed25519,
            body: plaintext,
        },
        associatedWith,
    });
}
function getPathString(pathObjArr) {
    return pathObjArr.map(node => `${node.ip}:${node.port}`).join(', ');
}
exports.getPathString = getPathString;
async function bchatOnionFetch({ targetNode, associatedWith, body, }) {
    try {
        const retriedResult = await (0, p_retry_1.default)(async () => {
            const path = await onions_1.OnionPaths.getOnionPath({ toExclude: targetNode });
            const result = await sendOnionRequestSnodeDest(path, targetNode, body, associatedWith);
            return result;
        }, {
            retries: 3,
            factor: 1,
            minTimeout: 100,
            onFailedAttempt: e => {
                window?.log?.warn(`onionFetchRetryable attempt #${e.attemptNumber} failed. ${e.retriesLeft} retries left...`);
            },
        });
        return retriedResult;
    }
    catch (e) {
        window?.log?.warn('onionFetchRetryable failed ', e.message);
        if (e?.errno === 'ENETUNREACH') {
            throw new Error(SNodeAPI_1.ERROR_CODE_NO_CONNECT);
        }
        if (e?.message === exports.CLOCK_OUT_OF_SYNC_MESSAGE_ERROR) {
            window?.log?.warn('Its a clock out of sync error ');
            throw new p_retry_1.default.AbortError(exports.CLOCK_OUT_OF_SYNC_MESSAGE_ERROR);
        }
        throw e;
    }
}
exports.bchatOnionFetch = bchatOnionFetch;
