"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConversationController = exports.getConversationController = void 0;
const data_1 = require("../../data/data");
const conversation_1 = require("../../models/conversation");
const util_1 = require("../../util");
const snodePool_1 = require("../apis/snode_api/snodePool");
const conversations_1 = require("../../state/ducks/conversations");
const opengroups_1 = require("../../data/opengroups");
const OpenGroupManagerV2_1 = require("../apis/open_group_api/opengroupV2/OpenGroupManagerV2");
const conversationInteractions_1 = require("../../interactions/conversationInteractions");
let instance;
const getConversationController = () => {
    if (instance) {
        return instance;
    }
    instance = new ConversationController();
    return instance;
};
exports.getConversationController = getConversationController;
class ConversationController {
    conversations;
    _initialFetchComplete = false;
    _initialPromise;
    constructor() {
        this.conversations = new conversation_1.ConversationCollection();
    }
    get(id) {
        if (!this._initialFetchComplete) {
            throw new Error('getConversationController().get() needs complete initial fetch');
        }
        return this.conversations.get(id);
    }
    getOrThrow(id) {
        if (!this._initialFetchComplete) {
            throw new Error('getConversationController().get() needs complete initial fetch');
        }
        const convo = this.conversations.get(id);
        if (convo) {
            return convo;
        }
        throw new Error(`Conversation ${id} does not exist on getConversationController().get()`);
    }
    getUnsafe(id) {
        return this.conversations.get(id);
    }
    dangerouslyCreateAndAdd(attributes) {
        return this.conversations.add(attributes);
    }
    getOrCreate(id, type) {
        if (typeof id !== 'string') {
            throw new TypeError("'id' must be a string");
        }
        if (type !== conversation_1.ConversationTypeEnum.PRIVATE && type !== conversation_1.ConversationTypeEnum.GROUP) {
            throw new TypeError(`'type' must be 'private' or 'group' got: '${type}'`);
        }
        if (!this._initialFetchComplete) {
            throw new Error('getConversationController().get() needs complete initial fetch');
        }
        let conversation = this.conversations.get(id);
        if (conversation) {
            return conversation;
        }
        conversation = this.conversations.add({
            id,
            type,
            version: 2,
        });
        const create = async () => {
            try {
                conversation.attributes.walletUserName = localStorage.getItem('walletUserName');
                await (0, data_1.saveConversation)(conversation.attributes);
                localStorage.setItem('walletUserName', '');
            }
            catch (error) {
                window?.log?.error('Conversation save failed! ', id, type, 'Error:', error && error.stack ? error.stack : error);
                throw error;
            }
            return conversation;
        };
        conversation.initialPromise = create();
        conversation.initialPromise.then(() => {
            if (window?.inboxStore) {
                window.inboxStore?.dispatch(conversations_1.actions.conversationAdded({
                    id: conversation.id,
                    data: conversation.getConversationModelProps(),
                }));
            }
            if (!conversation.isPublic() && conversation.isActive()) {
                void (0, snodePool_1.getSwarmFor)(id);
            }
        });
        return conversation;
    }
    getContactProfileNameOrShortenedPubKey(pubKey) {
        const conversation = (0, exports.getConversationController)().get(pubKey);
        if (!conversation) {
            return pubKey;
        }
        return conversation.getContactProfileNameOrShortenedPubKey();
    }
    getContactProfileNameOrFullPubKey(pubKey) {
        const conversation = this.conversations.get(pubKey);
        if (!conversation) {
            return pubKey;
        }
        return conversation.getContactProfileNameOrFullPubKey();
    }
    isMediumGroup(hexEncodedGroupPublicKey) {
        const convo = this.conversations.get(hexEncodedGroupPublicKey);
        if (convo) {
            return !!convo.isMediumGroup();
        }
        return false;
    }
    async getOrCreateAndWait(id, type) {
        const initialPromise = this._initialPromise !== undefined ? this._initialPromise : Promise.resolve();
        return initialPromise.then(() => {
            if (!id) {
                return Promise.reject(new Error('getOrCreateAndWait: invalid id passed.'));
            }
            const pubkey = id && id.key ? id.key : id;
            const conversation = this.getOrCreate(pubkey, type);
            if (conversation) {
                return conversation.initialPromise.then(() => conversation);
            }
            return Promise.reject(new Error('getOrCreateAndWait: did not get conversation'));
        });
    }
    async deleteContact(id) {
        if (!this._initialFetchComplete) {
            throw new Error('getConversationController().get() needs complete initial fetch');
        }
        window.log.info(`deleteContact with ${id}`);
        const conversation = this.conversations.get(id);
        if (!conversation) {
            window.log.warn(`deleteContact no such convo ${id}`);
            return;
        }
        if (conversation.isClosedGroup()) {
            window.log.info(`deleteContact SecretGroup case: ${id}`);
            await conversation.leaveClosedGroup();
        }
        else if (conversation.isOpenGroupV2()) {
            window?.log?.info('leaving Social group v2', conversation.id);
            const roomInfos = await (0, opengroups_1.getV2OpenGroupRoom)(conversation.id);
            if (roomInfos) {
                (0, OpenGroupManagerV2_1.getOpenGroupManager)().removeRoomFromPolledRooms(roomInfos);
                try {
                    await (0, opengroups_1.removeV2OpenGroupRoom)(conversation.id);
                }
                catch (e) {
                    window?.log?.info('removeV2OpenGroupRoom failed:', e);
                }
            }
        }
        window.log.info(`deleteContact destroyingMessages: ${id}`);
        await (0, conversationInteractions_1.deleteAllMessagesByConvoIdNoConfirmation)(conversation.id);
        window.log.info(`deleteContact message destroyed: ${id}`);
        if (conversation.isPrivate()) {
            window.log.info(`deleteContact isPrivate, marking as inactive: ${id}`);
            await (0, data_1.removeConversation)(id);
            this.conversations.remove(conversation);
            window.inboxStore?.dispatch(conversations_1.actions.conversationRemoved(conversation.id));
        }
        else {
            window.log.info(`deleteContact !isPrivate, removing convo from DB: ${id}`);
            await (0, data_1.removeConversation)(id);
            window.log.info(`deleteContact !isPrivate, convo removed from DB: ${id}`);
            this.conversations.remove(conversation);
            if (window?.inboxStore) {
                window.inboxStore?.dispatch(conversations_1.actions.conversationChanged({
                    id: conversation.id,
                    data: conversation.getConversationModelProps(),
                }));
                window.inboxStore?.dispatch(conversations_1.actions.conversationRemoved(conversation.id));
            }
            window.log.info(`deleteContact !isPrivate, convo removed from store: ${id}`);
        }
    }
    getConversations() {
        return Array.from(this.conversations.models);
    }
    unsafeDelete(convo) {
        this.conversations.remove(convo);
    }
    async load() {
        window?.log?.info('ConversationController: starting initial fetch');
        if (this.conversations.length) {
            throw new Error('ConversationController: Already loaded!');
        }
        const load = async () => {
            try {
                const start = Date.now();
                const collection = await (0, data_1.getAllConversations)();
                this.conversations.add(collection.models);
                this._initialFetchComplete = true;
                const promises = [];
                this.conversations.forEach((conversation) => {
                    if (!conversation.get('lastMessage')) {
                        promises.push(conversation.updateLastMessage());
                    }
                    promises.concat([conversation.updateProfileName()]);
                });
                await Promise.all(promises);
                window?.log?.info(`ConversationController: done with initial fetch in ${Date.now() - start}ms.`);
            }
            catch (error) {
                window?.log?.error('ConversationController: initial fetch failed', error && error.stack ? error.stack : error);
                throw error;
            }
        };
        await util_1.BlockedNumberController.load();
        this._initialPromise = load();
        return this._initialPromise;
    }
    loadPromise() {
        return this._initialPromise;
    }
    reset() {
        this._initialPromise = Promise.resolve();
        this._initialFetchComplete = false;
        if (window?.inboxStore) {
            window.inboxStore?.dispatch(conversations_1.actions.removeAllConversations());
        }
        this.conversations.reset([]);
    }
}
exports.ConversationController = ConversationController;
