"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.initiateOpenGroupUpdate = void 0;
const opengroups_1 = require("../../data/opengroups");
const attachments_1 = require("../../receiver/attachments");
const types_1 = require("../../types");
const VisualAttachment_1 = require("../../types/attachments/VisualAttachment");
const MessageAttachment_1 = require("../../types/MessageAttachment");
const opengroupV2_1 = require("../apis/open_group_api/opengroupV2");
const conversations_1 = require("../conversations");
const crypto_1 = require("../crypto");
const String_1 = require("../utils/String");
async function initiateOpenGroupUpdate(groupId, groupName, avatar) {
    const convo = (0, conversations_1.getConversationController)().get(groupId);
    if (!convo || !convo.isPublic() || !convo.isOpenGroupV2()) {
        throw new Error('Only opengroupv2 are supported');
    }
    if (avatar && avatar.objectUrl) {
        const blobAvatarAlreadyScaled = await (0, VisualAttachment_1.urlToBlob)(avatar.objectUrl);
        const dataResized = await blobAvatarAlreadyScaled.arrayBuffer();
        const roomInfos = await (0, opengroups_1.getV2OpenGroupRoom)(convo.id);
        if (!roomInfos || !dataResized.byteLength) {
            return false;
        }
        const uploadedFileDetails = await opengroupV2_1.ApiV2.uploadImageForRoomOpenGroupV2(new Uint8Array(dataResized), roomInfos);
        if (!uploadedFileDetails || !uploadedFileDetails.fileUrl) {
            window?.log?.warn('File opengroupv2 upload failed');
            return false;
        }
        let url;
        try {
            url = new URL(uploadedFileDetails.fileUrl);
            const pathname = url.pathname;
            const downloaded = await (0, attachments_1.downloadDataFromOpenGroupV2)(pathname, roomInfos);
            if (!(downloaded instanceof Uint8Array)) {
                const typeFound = typeof downloaded;
                throw new Error(`Expected a plain Uint8Array but got ${typeFound}`);
            }
            const upgraded = await (0, MessageAttachment_1.processNewAttachment)({
                data: downloaded.buffer,
                isRaw: true,
                contentType: types_1.MIME.IMAGE_UNKNOWN,
            });
            const newHash = (0, crypto_1.sha256)((0, String_1.fromArrayBufferToBase64)(downloaded.buffer));
            await convo.setBchatProfile({
                displayName: groupName || convo.get('name') || 'Unknown',
                avatar: upgraded.path,
                avatarHash: newHash,
            });
        }
        catch (e) {
            window?.log?.error(`Could not decrypt profile image: ${e}`);
            return false;
        }
    }
    return true;
}
exports.initiateOpenGroupUpdate = initiateOpenGroupUpdate;
