"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PubKey = exports.KeyPrefixType = exports.getStoragePubKey = void 0;
const String_1 = require("../utils/String");
const getStoragePubKey = (key) => window.bchatFeatureFlags.useTestNet ? key.substring(2) : key;
exports.getStoragePubKey = getStoragePubKey;
var KeyPrefixType;
(function (KeyPrefixType) {
    KeyPrefixType["unblinded"] = "00";
    KeyPrefixType["standard"] = "05";
    KeyPrefixType["blinded15"] = "15";
    KeyPrefixType["blinded25"] = "25";
    KeyPrefixType["groupV3"] = "03";
})(KeyPrefixType = exports.KeyPrefixType || (exports.KeyPrefixType = {}));
class PubKey {
    static PUBKEY_LEN = 66;
    static HEX = '[0-9a-fA-F]';
    static regexForPubkeys = `((bd)?${PubKey.HEX}{64})`;
    static PREFIX_GROUP_TEXTSECURE = '__textsecure_group__!';
    static regex = new RegExp(`^(${PubKey.PREFIX_GROUP_TEXTSECURE})?(bd)?(${PubKey.HEX}{64}|${PubKey.HEX}{32})$`);
    key;
    constructor(pubkeyString) {
        if (!PubKey.validate(pubkeyString)) {
            throw new Error(`Invalid pubkey string passed: ${pubkeyString}`);
        }
        this.key = pubkeyString.toLowerCase();
    }
    static cast(value) {
        if (!value) {
            throw new Error(`Invalid pubkey string passed: ${value}`);
        }
        return typeof value === 'string' ? new PubKey(value) : value;
    }
    static shorten(value) {
        const valAny = value;
        const pk = value instanceof PubKey ? valAny.key : value;
        if (!pk) {
            throw new Error('PubkKey.shorten was given an invalid PubKey to shorten.');
        }
        return `(...${pk.substring(pk.length - 6)})`;
    }
    static from(pubkeyString) {
        if (PubKey.validate(pubkeyString)) {
            return new PubKey(pubkeyString);
        }
        return undefined;
    }
    static normalize(pubkeyString) {
        if (PubKey.validate(pubkeyString)) {
            return pubkeyString;
        }
        return undefined;
    }
    static validate(pubkeyString) {
        return this.regex.test(pubkeyString);
    }
    static validateWithError(pubkey) {
        const isHex = pubkey.replace(/[\s]*/g, '').match(/^[0-9a-fA-F]+$/);
        if (!isHex) {
            return window.i18n('invalidBchatId');
        }
        const len = pubkey.length;
        if ((len !== 33 * 2 || !/^bd/.test(pubkey)) && len !== 32 * 2) {
            return window.i18n('invalidPubkeyFormat');
        }
        return undefined;
    }
    static remove05PrefixIfNeeded(keyWithOrWithoutPrefix) {
        if (keyWithOrWithoutPrefix.length === 66 && keyWithOrWithoutPrefix.startsWith('bd')) {
            return keyWithOrWithoutPrefix.substr(2);
        }
        return keyWithOrWithoutPrefix;
    }
    static addTextSecurePrefixIfNeeded(keyWithOrWithoutPrefix) {
        const key = keyWithOrWithoutPrefix instanceof PubKey
            ? keyWithOrWithoutPrefix.key
            : keyWithOrWithoutPrefix;
        if (!key.startsWith(PubKey.PREFIX_GROUP_TEXTSECURE)) {
            return PubKey.PREFIX_GROUP_TEXTSECURE + key;
        }
        return key;
    }
    static removeTextSecurePrefixIfNeeded(keyWithOrWithoutPrefix) {
        const key = keyWithOrWithoutPrefix instanceof PubKey
            ? keyWithOrWithoutPrefix.key
            : keyWithOrWithoutPrefix;
        return key.replace(PubKey.PREFIX_GROUP_TEXTSECURE, '');
    }
    static isEqual(comparator1, comparator2) {
        return PubKey.cast(comparator1).isEqual(comparator2);
    }
    isEqual(comparator) {
        return comparator instanceof PubKey
            ? this.key === comparator.key
            : this.key === comparator.toLowerCase();
    }
    withoutPrefix() {
        return PubKey.remove05PrefixIfNeeded(this.key);
    }
    toArray() {
        return (0, String_1.fromHexToArray)(this.key);
    }
    withoutPrefixToArray() {
        return (0, String_1.fromHexToArray)(PubKey.remove05PrefixIfNeeded(this.key));
    }
    static isBlinded(key) {
        return key.startsWith(KeyPrefixType.blinded15) || key.startsWith(KeyPrefixType.blinded25);
    }
}
exports.PubKey = PubKey;
