"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.computePeaks = exports.redactAttachmentUrl = void 0;
const promise_queue_1 = __importDefault(require("promise-queue"));
const data_1 = require("../../data/data");
const REDACTION_PLACEHOLDER = '[REDACTED]';
const MAX_PARALLEL_COMPUTE = 8;
const MAX_AUDIO_DURATION = 15 * 60;
const redactAttachmentUrl = (urlString) => {
    try {
        const url = new URL(urlString);
        url.search = '';
        return url.toString();
    }
    catch {
        return REDACTION_PLACEHOLDER;
    }
};
exports.redactAttachmentUrl = redactAttachmentUrl;
const inProgressMap = new Map();
const computeQueue = new promise_queue_1.default(MAX_PARALLEL_COMPUTE, Infinity);
async function getAudioDuration(url, buffer) {
    const blob = new Blob([buffer]);
    const blobURL = URL.createObjectURL(blob);
    const urlForLogging = (0, exports.redactAttachmentUrl)(url);
    const audio = new Audio();
    audio.muted = true;
    audio.src = blobURL;
    await new Promise((resolve, reject) => {
        audio.addEventListener('loadedmetadata', () => {
            resolve();
        });
        audio.addEventListener('error', event => {
            const error = new Error(`Failed to load audio from: ${urlForLogging} due to error: ${event.type}`);
            reject(error);
        });
    });
    if (Number.isNaN(audio.duration)) {
        throw new Error(`Invalid audio duration for: ${urlForLogging}`);
    }
    return audio.duration;
}
async function doComputePeaks(url, barCount, slicedSrc) {
    const cacheKey = `cacheKey${slicedSrc}:${barCount}`;
    const existing = await (0, data_1.getLRUCache)(cacheKey);
    const urlForLogging = (0, exports.redactAttachmentUrl)(url);
    const logId = `AudioContext(${urlForLogging})`;
    if (existing) {
        console.info(`${logId}: waveform cache hit`);
        return Promise.resolve(existing);
    }
    console.info(`${logId}: waveform cache miss`);
    const response = await fetch(url);
    const raw = await response.arrayBuffer();
    const duration = await getAudioDuration(url, raw);
    const peaks = new Array(barCount).fill(0);
    if (duration > MAX_AUDIO_DURATION) {
        console.info(`${logId}: duration ${duration}s is too long`);
        const emptyResult = { peaks, duration };
        const data = { key: cacheKey, value: emptyResult };
        await (0, data_1.setLRUCache)(data);
        return emptyResult;
    }
    const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
    const data = await audioCtx.decodeAudioData(raw);
    const norms = new Array(barCount).fill(0);
    const samplesPerPeak = data.length / peaks.length;
    for (let channelNum = 0; channelNum < data.numberOfChannels; channelNum += 1) {
        const channel = data.getChannelData(channelNum);
        for (let sample = 0; sample < channel.length; sample += 1) {
            const i = Math.floor(sample / samplesPerPeak);
            peaks[i] += channel[sample] ** 2;
            norms[i] += 1;
        }
    }
    let max = 1e-23;
    for (let i = 0; i < peaks.length; i += 1) {
        peaks[i] = Math.sqrt(peaks[i] / Math.max(1, norms[i]));
        max = Math.max(max, peaks[i]);
    }
    for (let i = 0; i < peaks.length; i += 1) {
        peaks[i] /= max;
    }
    const result = { peaks, duration };
    const cachedata = { key: cacheKey, value: result };
    await (0, data_1.setLRUCache)(cachedata);
    return result;
}
async function computePeaks(url, barCount, slicedSrc) {
    const computeKey = `${url}:${barCount}`;
    const logId = `AudioContext(${(0, exports.redactAttachmentUrl)(url)})`;
    const pending = inProgressMap.get(computeKey);
    if (pending) {
        console.info(`${logId}: already computing peaks`);
        return pending;
    }
    console.info(`${logId}: queueing computing peaks`);
    const promise = computeQueue.add(() => doComputePeaks(url, barCount, slicedSrc));
    inProgressMap.set(computeKey, promise);
    try {
        return await promise;
    }
    finally {
        inProgressMap.delete(computeKey);
    }
}
exports.computePeaks = computePeaks;
