"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.groupMediaItemsByDate = void 0;
const moment_1 = __importDefault(require("moment"));
const lodash_1 = require("lodash");
const groupMediaItemsByDate = (timestamp, mediaItems) => {
    const referenceDateTime = moment_1.default.utc(timestamp);
    const sortedMediaItem = (0, lodash_1.sortBy)(mediaItems, mediaItem => {
        const { messageTimestamp } = mediaItem;
        return -messageTimestamp;
    });
    const messagesWithSection = sortedMediaItem.map(withSection(referenceDateTime));
    const groupedMediaItem = (0, lodash_1.groupBy)(messagesWithSection, 'type');
    const yearMonthMediaItem = Object.values((0, lodash_1.groupBy)(groupedMediaItem.yearMonth, 'order')).reverse();
    return (0, lodash_1.compact)([
        toSection(groupedMediaItem.today),
        toSection(groupedMediaItem.yesterday),
        toSection(groupedMediaItem.thisWeek),
        toSection(groupedMediaItem.thisMonth),
        ...yearMonthMediaItem.map(toSection),
    ]);
};
exports.groupMediaItemsByDate = groupMediaItemsByDate;
const toSection = (messagesWithSection) => {
    if (!messagesWithSection || messagesWithSection.length === 0) {
        return;
    }
    const firstMediaItemWithSection = messagesWithSection[0];
    if (!firstMediaItemWithSection) {
        return;
    }
    const mediaItems = messagesWithSection.map(messageWithSection => messageWithSection.mediaItem);
    switch (firstMediaItemWithSection.type) {
        case 'today':
        case 'yesterday':
        case 'thisWeek':
        case 'thisMonth':
            return {
                type: firstMediaItemWithSection.type,
                mediaItems,
            };
        case 'yearMonth':
            return {
                type: firstMediaItemWithSection.type,
                year: firstMediaItemWithSection.year,
                month: firstMediaItemWithSection.month,
                mediaItems,
            };
        default:
            return;
    }
};
const withSection = (referenceDateTime) => (mediaItem) => {
    const today = (0, moment_1.default)(referenceDateTime).startOf('day');
    const yesterday = (0, moment_1.default)(referenceDateTime)
        .subtract(1, 'day')
        .startOf('day');
    const thisWeek = (0, moment_1.default)(referenceDateTime).startOf('isoWeek');
    const thisMonth = (0, moment_1.default)(referenceDateTime).startOf('month');
    const { messageTimestamp } = mediaItem;
    const mediaItemReceivedDate = moment_1.default.utc(messageTimestamp);
    if (mediaItemReceivedDate.isAfter(today)) {
        return {
            order: 0,
            type: 'today',
            mediaItem,
        };
    }
    if (mediaItemReceivedDate.isAfter(yesterday)) {
        return {
            order: 1,
            type: 'yesterday',
            mediaItem,
        };
    }
    if (mediaItemReceivedDate.isAfter(thisWeek)) {
        return {
            order: 2,
            type: 'thisWeek',
            mediaItem,
        };
    }
    if (mediaItemReceivedDate.isAfter(thisMonth)) {
        return {
            order: 3,
            type: 'thisMonth',
            mediaItem,
        };
    }
    const month = mediaItemReceivedDate.month();
    const year = mediaItemReceivedDate.year();
    return {
        order: year * 100 + month,
        type: 'yearMonth',
        month,
        year,
        mediaItem,
    };
};
