"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.initData = exports.callChannel = exports._shutdown = exports._jobs = void 0;
const electron_1 = require("electron");
const channels_1 = require("./channels");
const channelsToMakeForOpengroupV2 = [
    'getAllV2OpenGroupRooms',
    'getV2OpenGroupRoom',
    'getV2OpenGroupRoomByRoomId',
    'saveV2OpenGroupRoom',
    'removeV2OpenGroupRoom',
    'getAllOpenGroupV2Conversations',
];
const channelsToMake = new Set([
    'shutdown',
    'close',
    'removeDB',
    'getPasswordHash',
    'getGuardNodes',
    'updateGuardNodes',
    'createOrUpdateItem',
    'getItemById',
    'getAllItems',
    'removeItemById',
    'getSwarmNodesForPubkey',
    'updateSwarmNodesForPubkey',
    'saveConversation',
    'getConversationById',
    'updateConversation',
    'updateConversationAddress',
    'updateWalletAddressInConversation',
    'removeConversation',
    'getAllConversations',
    'getAllOpenGroupV1Conversations',
    'getPubkeysInPublicConversation',
    'getAllGroupsInvolvingId',
    'searchConversations',
    'searchMessages',
    'searchMessagesInConversation',
    'saveMessage',
    'cleanSeenMessages',
    'cleanLastHashes',
    'updateLastHash',
    'saveSeenMessageHashes',
    'saveMessages',
    'removeMessage',
    '_removeMessages',
    'getUnreadByConversation',
    'getUnreadCountByConversation',
    'getMessageCountByType',
    'removeAllMessagesInConversation',
    'getMessageCount',
    'getMessageBySenderAndSentAt',
    'filterAlreadyFetchedOpengroupMessage',
    'getMessageBySenderAndTimestamp',
    'getMessageIdsFromServerIds',
    'getMessageById',
    'getMessageByServerId',
    'getMessagesBySentAt',
    'getExpiredMessages',
    'getOutgoingWithoutExpiresAt',
    'getNextExpiringMessage',
    'getMessagesByConversation',
    'getLastMessagesByConversation',
    'getOldestMessageInConversation',
    'getFirstUnreadMessageIdInConversation',
    'getFirstUnreadMessageWithMention',
    'hasConversationOutgoingMessage',
    'getSeenMessagesByHashList',
    'getLastHashBySnode',
    'getUnprocessedCount',
    'getAllUnprocessed',
    'getUnprocessedById',
    'saveUnprocessed',
    'updateUnprocessedAttempts',
    'updateUnprocessedWithData',
    'removeUnprocessed',
    'removeAllUnprocessed',
    'getNextAttachmentDownloadJobs',
    'saveAttachmentDownloadJob',
    'resetAttachmentDownloadPending',
    'setAttachmentDownloadJobPending',
    'removeAttachmentDownloadJob',
    'removeAllAttachmentDownloadJobs',
    'removeAll',
    'removeAllWithOutRecipient',
    'removeAllConversations',
    'removeOtherData',
    'cleanupOrphanedAttachments',
    'getMessagesWithVisualMediaAttachments',
    'getMessagesWithFileAttachments',
    'getAllEncryptionKeyPairsForGroup',
    'getLatestClosedGroupEncryptionKeyPair',
    'addClosedGroupEncryptionKeyPair',
    'removeAllClosedGroupEncryptionKeyPairs',
    'fillWithTestData',
    ...channelsToMakeForOpengroupV2,
    'getRecipientAddress',
    'saveRecipientAddress',
    'updateLRUCache',
    'getLRUCache'
]);
const SQL_CHANNEL_KEY = 'sql-channel';
let _shutdownPromise = null;
const DATABASE_UPDATE_TIMEOUT = 2 * 60 * 1000;
exports._jobs = Object.create(null);
const _DEBUG = false;
let _jobCounter = 0;
let _shuttingDown = false;
let _shutdownCallback = null;
async function _shutdown() {
    if (_shutdownPromise) {
        return _shutdownPromise;
    }
    _shuttingDown = true;
    const jobKeys = Object.keys(exports._jobs);
    window?.log?.info(`data.shutdown: starting process. ${jobKeys.length} jobs outstanding`);
    if (jobKeys.length === 0) {
        return null;
    }
    _shutdownPromise = new Promise((resolve, reject) => {
        _shutdownCallback = (error) => {
            window?.log?.info('data.shutdown: process complete');
            if (error) {
                return reject(error);
            }
            return resolve(undefined);
        };
    });
    return _shutdownPromise;
}
exports._shutdown = _shutdown;
function _getJob(id) {
    return exports._jobs[id];
}
function makeChannel(fnName) {
    channels_1.channels[fnName] = async (...args) => {
        const jobId = _makeJob(fnName);
        return new Promise((resolve, reject) => {
            electron_1.ipcRenderer.send(SQL_CHANNEL_KEY, jobId, fnName, ...args);
            _updateJob(jobId, {
                resolve,
                reject,
                args: _DEBUG ? args : null,
            });
            exports._jobs[jobId].timer = setTimeout(() => reject(new Error(`SQL channel job ${jobId} (${fnName}) timed out`)), DATABASE_UPDATE_TIMEOUT);
        });
    };
}
async function callChannel(name) {
    return new Promise((resolve, reject) => {
        electron_1.ipcRenderer.send(name);
        electron_1.ipcRenderer.once(`${name}-done`, (_event, error) => {
            if (error) {
                return reject(error);
            }
            return resolve(undefined);
        });
        setTimeout(() => reject(new Error(`callChannel call to ${name} timed out`)), DATABASE_UPDATE_TIMEOUT);
    });
}
exports.callChannel = callChannel;
function initData() {
    electron_1.ipcRenderer.setMaxListeners(0);
    channelsToMake.forEach(makeChannel);
    electron_1.ipcRenderer.on(`${SQL_CHANNEL_KEY}-done`, (_event, jobId, errorForDisplay, result) => {
        const job = _getJob(jobId);
        if (!job) {
            throw new Error(`Received SQL channel reply to job ${jobId}, but did not have it in our registry!`);
        }
        const { resolve, reject, fnName } = job;
        if (errorForDisplay) {
            return reject(new Error(`Error received from SQL channel job ${jobId} (${fnName}): ${errorForDisplay}`));
        }
        return resolve(result);
    });
}
exports.initData = initData;
function _updateJob(id, data) {
    const { resolve, reject } = data;
    const { fnName, start } = exports._jobs[id];
    exports._jobs[id] = {
        ...exports._jobs[id],
        ...data,
        resolve: (value) => {
            _removeJob(id);
            if (_DEBUG) {
                const end = Date.now();
                const delta = end - start;
                if (delta > 10) {
                    window?.log?.debug(`SQL channel job ${id} (${fnName}) succeeded in ${end - start}ms`);
                }
            }
            return resolve(value);
        },
        reject: (error) => {
            _removeJob(id);
            const end = Date.now();
            window?.log?.warn(`SQL channel job ${id} (${fnName}) failed in ${end - start}ms`);
            return reject(error);
        },
    };
}
function _removeJob(id) {
    if (_DEBUG) {
        exports._jobs[id].complete = true;
        return;
    }
    if (exports._jobs[id].timer) {
        global.clearTimeout(exports._jobs[id].timer);
        exports._jobs[id].timer = null;
    }
    delete exports._jobs[id];
    if (_shutdownCallback) {
        const keys = Object.keys(exports._jobs);
        if (keys.length === 0) {
            _shutdownCallback();
        }
    }
}
function _makeJob(fnName) {
    if (_shuttingDown && fnName !== 'close') {
        throw new Error(`Rejecting SQL channel job (${fnName}); application is shutting down`);
    }
    _jobCounter += 1;
    const id = _jobCounter;
    if (_DEBUG) {
        window?.log?.debug(`SQL channel job ${id} (${fnName}) started`);
    }
    exports._jobs[id] = {
        fnName,
        start: Date.now(),
    };
    return id;
}
