"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteMessagesById = exports.deleteMessagesByIdForEveryone = exports.deleteMessageLocallyOnly = exports.deleteMessagesFromSwarmAndMarkAsDeletedLocally = exports.deleteMessagesFromSwarmAndCompletelyLocally = exports.deleteMessagesFromSwarmOnly = void 0;
const lodash_1 = require("lodash");
const data_1 = require("../../data/data");
const opengroupV2_1 = require("../../bchat/apis/open_group_api/opengroupV2");
const bchat_1 = require("../../bchat");
const conversations_1 = require("../../bchat/conversations");
const UnsendMessage_1 = require("../../bchat/messages/outgoing/controlMessage/UnsendMessage");
const onionPath_1 = require("../../bchat/onions/onionPath");
const SNodeAPI_1 = require("../../bchat/apis/snode_api/SNodeAPI");
const types_1 = require("../../bchat/types");
const utils_1 = require("../../bchat/utils");
const conversations_2 = require("../../state/ducks/conversations");
const modalDialog_1 = require("../../state/ducks/modalDialog");
const BchatButton_1 = require("../../components/basic/BchatButton");
async function unsendMessagesForEveryone(conversation, msgsToDelete) {
    window?.log?.info('Deleting messages for all users in this conversation');
    const destinationId = conversation.id;
    if (!destinationId) {
        return;
    }
    if (conversation.isOpenGroupV2()) {
        throw new Error('Cannot unsend a message for an opengroup v2. This has to be a deleteMessage api call');
    }
    const unsendMsgObjects = getUnsendMessagesObjects(msgsToDelete);
    if (conversation.isPrivate()) {
        await Promise.all(unsendMsgObjects.map(unsendObject => (0, bchat_1.getMessageQueue)()
            .sendToPubKey(new types_1.PubKey(destinationId), unsendObject)
            .catch(window?.log?.error)));
        await Promise.all(unsendMsgObjects.map(unsendObject => (0, bchat_1.getMessageQueue)()
            .sendSyncMessage(unsendObject)
            .catch(window?.log?.error)));
    }
    else if (conversation.isClosedGroup()) {
        await Promise.all(unsendMsgObjects.map(unsendObject => {
            (0, bchat_1.getMessageQueue)()
                .sendToGroup(unsendObject, undefined, new types_1.PubKey(destinationId))
                .catch(window?.log?.error);
        }));
    }
    await deleteMessagesFromSwarmAndCompletelyLocally(conversation, msgsToDelete);
    window.inboxStore?.dispatch((0, conversations_2.resetSelectedMessageIds)());
    utils_1.ToastUtils.pushDeleted(msgsToDelete.length);
}
function getUnsendMessagesObjects(messages) {
    return (0, lodash_1.compact)(messages.map(message => {
        const author = message.get('source');
        const timestamp = message.getPropsForMessage().timestamp;
        if (!timestamp) {
            window?.log?.error('cannot find timestamp - aborting unsend request');
            return undefined;
        }
        const unsendParams = {
            timestamp,
            author,
        };
        return new UnsendMessage_1.UnsendMessage(unsendParams);
    }));
}
async function deleteMessagesFromSwarmOnly(messages) {
    try {
        const deletionMessageHashes = (0, lodash_1.compact)(messages.map(m => m.get('messageHash')));
        if (deletionMessageHashes.length > 0) {
            const errorOnSnode = await (0, SNodeAPI_1.networkDeleteMessages)(deletionMessageHashes);
            return errorOnSnode === null || errorOnSnode.length === 0;
        }
        window.log?.warn('deleteMessagesFromSwarmOnly: We do not have hashes for some of those messages');
        return false;
    }
    catch (e) {
        window.log?.error('deleteMessagesFromSwarmOnly: Error deleting message from swarm', e);
        return false;
    }
}
exports.deleteMessagesFromSwarmOnly = deleteMessagesFromSwarmOnly;
async function deleteMessagesFromSwarmAndCompletelyLocally(conversation, messages) {
    if (conversation.isMediumGroup()) {
        window.log.info('Cannot delete message from a closed group swarm, so we just complete delete.');
        await Promise.all(messages.map(async (message) => {
            return deleteMessageLocallyOnly({ conversation, message, deletionType: 'complete' });
        }));
        return;
    }
    window.log.warn('Deleting from swarm of ', (0, onionPath_1.ed25519Str)(conversation.id), ' hashes: ', messages.map(m => m.get('messageHash')));
    const deletedFromSwarm = await deleteMessagesFromSwarmOnly(messages);
    if (!deletedFromSwarm) {
        window.log.warn('deleteMessagesFromSwarmAndCompletelyLocally: some messages failed to be deleted. Maybe they were already deleted?');
    }
    await Promise.all(messages.map(async (message) => {
        return deleteMessageLocallyOnly({ conversation, message, deletionType: 'complete' });
    }));
}
exports.deleteMessagesFromSwarmAndCompletelyLocally = deleteMessagesFromSwarmAndCompletelyLocally;
async function deleteMessagesFromSwarmAndMarkAsDeletedLocally(conversation, messages) {
    if (conversation.isMediumGroup()) {
        window.log.info('Cannot delete messages from a closed group swarm, so we just markDeleted.');
        await Promise.all(messages.map(async (message) => {
            return deleteMessageLocallyOnly({ conversation, message, deletionType: 'markDeleted' });
        }));
        return;
    }
    const deletedFromSwarm = await deleteMessagesFromSwarmOnly(messages);
    if (!deletedFromSwarm) {
        window.log.warn('deleteMessagesFromSwarmAndMarkAsDeletedLocally: some messages failed to be deleted but still removing the messages content... ');
    }
    await Promise.all(messages.map(async (message) => {
        return deleteMessageLocallyOnly({ conversation, message, deletionType: 'markDeleted' });
    }));
}
exports.deleteMessagesFromSwarmAndMarkAsDeletedLocally = deleteMessagesFromSwarmAndMarkAsDeletedLocally;
async function deleteMessageLocallyOnly({ conversation, message, deletionType, }) {
    if (deletionType === 'complete') {
        await conversation.removeMessage(message.get('id'));
    }
    else {
        await message.markAsDeleted();
        await message.markRead(Date.now());
    }
    conversation.updateLastMessage();
}
exports.deleteMessageLocallyOnly = deleteMessageLocallyOnly;
async function unsendMessageJustForThisUser(conversation, msgsToDelete) {
    window?.log?.warn('Deleting messages just for this user');
    const unsendMsgObjects = getUnsendMessagesObjects(msgsToDelete);
    await Promise.all(unsendMsgObjects.map(unsendObject => (0, bchat_1.getMessageQueue)()
        .sendSyncMessage(unsendObject)
        .catch(window?.log?.error)));
    await deleteMessagesFromSwarmAndCompletelyLocally(conversation, msgsToDelete);
    window.inboxStore?.dispatch((0, conversations_2.resetSelectedMessageIds)());
    utils_1.ToastUtils.pushDeleted(msgsToDelete.length);
}
const doDeleteSelectedMessagesInSOGS = async (selectedMessages, conversation, isAllOurs) => {
    const ourDevicePubkey = utils_1.UserUtils.getOurPubKeyStrFromCache();
    if (!ourDevicePubkey) {
        return;
    }
    const isAdmin = conversation.isAdmin(ourDevicePubkey);
    if (!isAllOurs && !isAdmin) {
        utils_1.ToastUtils.pushMessageDeleteForbidden();
        window.inboxStore?.dispatch((0, conversations_2.resetSelectedMessageIds)());
        return;
    }
    const toDeleteLocallyIds = await deleteOpenGroupMessages(selectedMessages, conversation);
    if (toDeleteLocallyIds.length === 0) {
        return;
    }
    await Promise.all(toDeleteLocallyIds.map(async (id) => {
        const msgToDeleteLocally = await (0, data_1.getMessageById)(id);
        if (msgToDeleteLocally) {
            return deleteMessageLocallyOnly({
                conversation,
                message: msgToDeleteLocally,
                deletionType: 'complete',
            });
        }
    }));
    utils_1.ToastUtils.pushDeleted(toDeleteLocallyIds.length);
    window.inboxStore?.dispatch((0, conversations_2.resetSelectedMessageIds)());
};
const doDeleteSelectedMessages = async ({ conversation, selectedMessages, deleteForEveryone, }) => {
    const ourDevicePubkey = utils_1.UserUtils.getOurPubKeyStrFromCache();
    if (!ourDevicePubkey) {
        return;
    }
    const isAllOurs = selectedMessages.every(message => ourDevicePubkey === message.getSource());
    if (conversation.isPublic()) {
        return doDeleteSelectedMessagesInSOGS(selectedMessages, conversation, isAllOurs);
    }
    if (deleteForEveryone) {
        if (!isAllOurs) {
            utils_1.ToastUtils.pushMessageDeleteForbidden();
            window.inboxStore?.dispatch((0, conversations_2.resetSelectedMessageIds)());
            return;
        }
        return unsendMessagesForEveryone(conversation, selectedMessages);
    }
    if (conversation.isClosedGroup()) {
        await deleteMessagesFromSwarmAndCompletelyLocally(conversation, selectedMessages);
        window.inboxStore?.dispatch((0, conversations_2.resetSelectedMessageIds)());
        utils_1.ToastUtils.pushDeleted(selectedMessages.length);
        return;
    }
    return unsendMessageJustForThisUser(conversation, selectedMessages);
};
async function deleteMessagesByIdForEveryone(messageIds, conversationId) {
    const conversation = (0, conversations_1.getConversationController)().getOrThrow(conversationId);
    const selectedMessages = (0, lodash_1.compact)(await Promise.all(messageIds.map(m => (0, data_1.getMessageById)(m, false))));
    const messageCount = selectedMessages.length;
    const moreThanOne = messageCount > 1;
    window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)({
        title: window.i18n('deleteForEveryone'),
        message: moreThanOne
            ? window.i18n('deleteMessagesQuestion', [messageCount.toString()])
            : window.i18n('deleteMessageQuestion'),
        okText: window.i18n('deleteForEveryone'),
        okTheme: BchatButton_1.BchatButtonColor.Danger,
        onClickOk: async () => {
            await doDeleteSelectedMessages({ selectedMessages, conversation, deleteForEveryone: true });
            window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)(null));
            return;
        },
        closeAfterInput: false,
    }));
}
exports.deleteMessagesByIdForEveryone = deleteMessagesByIdForEveryone;
async function deleteMessagesById(messageIds, conversationId) {
    const conversation = (0, conversations_1.getConversationController)().getOrThrow(conversationId);
    const selectedMessages = (0, lodash_1.compact)(await Promise.all(messageIds.map(m => (0, data_1.getMessageById)(m, false))));
    const messageCount = selectedMessages.length;
    const moreThanOne = selectedMessages.length > 1;
    window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)({
        title: 'Delete selected messages?',
        message: moreThanOne
            ? window.i18n('deleteMessagesQuestion', [messageCount.toString()])
            : window.i18n('deleteMessageQuestion'),
        okText: window.i18n('delete'),
        okTheme: BchatButton_1.BchatButtonColor.Danger,
        onClickOk: async () => {
            await doDeleteSelectedMessages({
                selectedMessages,
                conversation,
                deleteForEveryone: false,
            });
            window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)(null));
        },
        closeAfterInput: false,
    }));
}
exports.deleteMessagesById = deleteMessagesById;
async function deleteOpenGroupMessages(messages, convo) {
    if (!convo.isPublic()) {
        throw new Error('cannot delete public message on a non public groups');
    }
    if (!convo.isOpenGroupV2()) {
        throw new Error('Opengroupv1 are not supported anymore');
    }
    const roomInfos = convo.toOpenGroupV2();
    const validServerIdsToRemove = (0, lodash_1.compact)(messages.map(msg => {
        return msg.get('serverId');
    }));
    const validMessageModelsToRemove = (0, lodash_1.compact)(messages.map(msg => {
        const serverId = msg.get('serverId');
        if (serverId) {
            return msg;
        }
        return undefined;
    }));
    let allMessagesAreDeleted = false;
    if (validServerIdsToRemove.length) {
        allMessagesAreDeleted = await opengroupV2_1.ApiV2.deleteMessageByServerIds(validServerIdsToRemove, roomInfos);
    }
    if (allMessagesAreDeleted) {
        window?.log?.info('Removed all those serverIds messages successfully');
        return validMessageModelsToRemove.map(m => m.id);
    }
    else {
        window?.log?.info('failed to remove all those serverIds message. not removing them locally neither');
        return [];
    }
}
