"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConversationCollection = exports.ConversationModel = exports.fillConvoAttributesWithDefaults = exports.ConversationNotificationSetting = exports.ConversationTypeEnum = void 0;
const backbone_1 = __importDefault(require("backbone"));
const lodash_1 = __importDefault(require("lodash"));
const bchat_1 = require("../bchat");
const conversations_1 = require("../bchat/conversations");
const ClosedGroupVisibleMessage_1 = require("../bchat/messages/outgoing/visibleMessage/ClosedGroupVisibleMessage");
const types_1 = require("../bchat/types");
const utils_1 = require("../bchat/utils");
const util_1 = require("../util");
const closed_group_1 = require("../bchat/group/closed-group");
const protobuf_1 = require("../protobuf");
const message_1 = require("./message");
const messageType_1 = require("./messageType");
const auto_bind_1 = __importDefault(require("auto-bind"));
const data_1 = require("../../ts/data/data");
const String_1 = require("../bchat/utils/String");
const conversations_2 = require("../state/ducks/conversations");
const ExpirationTimerUpdateMessage_1 = require("../bchat/messages/outgoing/controlMessage/ExpirationTimerUpdateMessage");
const TypingMessage_1 = require("../bchat/messages/outgoing/controlMessage/TypingMessage");
const VisibleMessage_1 = require("../bchat/messages/outgoing/visibleMessage/VisibleMessage");
const ReadReceiptMessage_1 = require("../bchat/messages/outgoing/controlMessage/receipt/ReadReceiptMessage");
const utils_2 = require("../bchat/apis/open_group_api/utils");
const OpenGroupVisibleMessage_1 = require("../bchat/messages/outgoing/visibleMessage/OpenGroupVisibleMessage");
const OpenGroupUtils_1 = require("../bchat/apis/open_group_api/utils/OpenGroupUtils");
const TaskWithTimeout_1 = require("../bchat/utils/TaskWithTimeout");
const Performance_1 = require("../bchat/utils/Performance");
const onionPath_1 = require("../bchat/onions/onionPath");
const DecryptedAttachmentsManager_1 = require("../bchat/crypto/DecryptedAttachmentsManager");
const MIME_1 = require("../types/MIME");
const syncUtils_1 = require("../bchat/utils/syncUtils");
const SNodeAPI_1 = require("../bchat/apis/snode_api/SNodeAPI");
const Conversation_1 = require("../types/Conversation");
const settings_key_1 = require("../data/settings-key");
const MessageAttachment_1 = require("../types/MessageAttachment");
const User_1 = require("../bchat/utils/User");
const MessageRequestResponse_1 = require("../bchat/messages/outgoing/controlMessage/MessageRequestResponse");
const notifications_1 = require("../util/notifications");
const storage_1 = require("../util/storage");
const reactions_1 = require("../util/reactions");
var ConversationTypeEnum;
(function (ConversationTypeEnum) {
    ConversationTypeEnum["GROUP"] = "group";
    ConversationTypeEnum["PRIVATE"] = "private";
})(ConversationTypeEnum = exports.ConversationTypeEnum || (exports.ConversationTypeEnum = {}));
exports.ConversationNotificationSetting = ['all', 'disabled', 'mentions_only'];
const fillConvoAttributesWithDefaults = (optAttributes) => {
    return lodash_1.default.defaults(optAttributes, {
        members: [],
        zombies: [],
        left: false,
        unreadCount: 0,
        lastMessageStatus: null,
        lastJoinedTimestamp: new Date('1970-01-01Z00:00:00:000').getTime(),
        groupAdmins: [],
        isKickedFromGroup: false,
        isMe: false,
        subscriberCount: 0,
        is_medium_group: false,
        lastMessage: null,
        expireTimer: 0,
        mentionedUs: false,
        active_at: 0,
        triggerNotificationsFor: 'all',
        isTrustedForAttachmentDownload: false,
        isPinned: false,
        isApproved: false,
        didApproveMe: false,
        walletAddress: null,
        walletUserName: null,
        walletCreatedDaemonHeight: null,
        isBnsHolder: false,
    });
};
exports.fillConvoAttributesWithDefaults = fillConvoAttributesWithDefaults;
class ConversationModel extends backbone_1.default.Model {
    updateLastMessage;
    throttledBumpTyping;
    throttledNotify;
    markRead;
    initialPromise;
    typingRefreshTimer;
    typingPauseTimer;
    typingTimer;
    lastReadTimestamp;
    pending;
    constructor(attributes) {
        super((0, exports.fillConvoAttributesWithDefaults)(attributes));
        this.initialPromise = Promise.resolve();
        (0, auto_bind_1.default)(this);
        this.throttledBumpTyping = lodash_1.default.throttle(this.bumpTyping, 300);
        this.updateLastMessage = lodash_1.default.throttle(this.bouncyUpdateLastMessage.bind(this), 1000, {
            trailing: true,
            leading: true,
        });
        this.throttledNotify = lodash_1.default.debounce(this.notify, 500, { maxWait: 5000, trailing: true });
        const markReadDebounced = lodash_1.default.debounce(this.markReadBouncy, 1000, {
            leading: true,
            trailing: true,
        });
        this.markRead = (newestUnreadDate) => {
            const lastReadTimestamp = this.lastReadTimestamp;
            if (newestUnreadDate > lastReadTimestamp) {
                this.lastReadTimestamp = newestUnreadDate;
            }
            if (newestUnreadDate !== lastReadTimestamp) {
                void markReadDebounced(newestUnreadDate);
            }
        };
        this.typingRefreshTimer = null;
        this.typingPauseTimer = null;
        this.lastReadTimestamp = 0;
        window.inboxStore?.dispatch((0, conversations_2.conversationChanged)({ id: this.id, data: this.getConversationModelProps() }));
    }
    static hasValidIncomingRequestValues({ isMe, isApproved, isBlocked, isPrivate, }) {
        return Boolean(!isMe && !isApproved && isPrivate && !isBlocked);
    }
    static hasValidOutgoingRequestValues({ isMe, didApproveMe, isApproved, isBlocked, isPrivate, }) {
        return Boolean(!isMe && isApproved && isPrivate && !isBlocked && !didApproveMe);
    }
    idForLogging() {
        if (this.isPrivate()) {
            return this.id;
        }
        if (this.isPublic()) {
            return `opengroup(${this.id})`;
        }
        return `group(${(0, onionPath_1.ed25519Str)(this.id)})`;
    }
    isMe() {
        return utils_1.UserUtils.isUsFromCache(this.id);
    }
    isPublic() {
        return Boolean(this.id && this.id.match(utils_2.OpenGroupUtils.openGroupPrefixRegex));
    }
    isOpenGroupV2() {
        return utils_2.OpenGroupUtils.isOpenGroupV2(this.id);
    }
    isClosedGroup() {
        return this.get('type') === ConversationTypeEnum.GROUP && !this.isPublic();
    }
    isBlocked() {
        if (!this.id || this.isMe()) {
            return false;
        }
        if (this.isClosedGroup()) {
            return util_1.BlockedNumberController.isGroupBlocked(this.id);
        }
        if (this.isPrivate()) {
            return util_1.BlockedNumberController.isBlocked(this.id);
        }
        return false;
    }
    isMediumGroup() {
        return this.get('is_medium_group');
    }
    isActive() {
        return Boolean(this.get('active_at'));
    }
    async cleanup() {
        await (0, MessageAttachment_1.deleteExternalFilesOfConversation)(this.attributes);
    }
    async onExpired(_message) {
        await this.updateLastMessage();
    }
    getGroupAdmins() {
        const groupAdmins = this.get('groupAdmins');
        return groupAdmins && groupAdmins?.length > 0 ? groupAdmins : [];
    }
    getConversationModelProps() {
        const groupAdmins = this.getGroupAdmins();
        const isPublic = this.isPublic();
        const walletAddress = this.isWalletAddress();
        const walletUserName = this.getProfileName();
        const walletCreatedDaemonHeight = this.getwalletCreatedDaemonHeight();
        const isBnsHolder = this.bnsHolder();
        const members = this.isGroup() && !isPublic ? this.get('members') : [];
        const zombies = this.isGroup() && !isPublic ? this.get('zombies') : [];
        const ourNumber = utils_1.UserUtils.getOurPubKeyStrFromCache();
        const avatarPath = this.getAvatarPath();
        const isPrivate = this.isPrivate();
        const isGroup = !isPrivate;
        const weAreAdmin = this.isAdmin(ourNumber);
        const isMe = this.isMe();
        const isTyping = !!this.typingTimer;
        const name = this.getName();
        const profileName = this.getProfileName();
        const unreadCount = this.get('unreadCount') || undefined;
        const mentionedUs = this.get('mentionedUs') || undefined;
        const isBlocked = this.isBlocked();
        const subscriberCount = this.get('subscriberCount');
        const isPinned = this.isPinned();
        const isApproved = this.isApproved();
        const didApproveMe = this.didApproveMe();
        const hasNickname = !!this.getNickname();
        const isKickedFromGroup = !!this.get('isKickedFromGroup');
        const left = !!this.get('left');
        const expireTimer = this.get('expireTimer');
        const currentNotificationSetting = this.get('triggerNotificationsFor');
        const toRet = {
            id: this.id,
            activeAt: this.get('active_at'),
            type: isPrivate ? ConversationTypeEnum.PRIVATE : ConversationTypeEnum.GROUP,
        };
        toRet.isBnsHolder = isBnsHolder;
        if (isPrivate) {
            toRet.isPrivate = true;
        }
        if (isGroup) {
            toRet.isGroup = true;
        }
        if (weAreAdmin) {
            toRet.weAreAdmin = true;
        }
        if (isMe) {
            toRet.isMe = true;
        }
        if (isPublic) {
            toRet.isPublic = true;
        }
        if (isTyping) {
            toRet.isTyping = true;
        }
        if (isTyping) {
            toRet.isTyping = true;
        }
        if (avatarPath) {
            toRet.avatarPath = avatarPath;
        }
        if (name) {
            toRet.name = name;
        }
        if (profileName) {
            toRet.profileName = profileName;
        }
        if (unreadCount) {
            toRet.unreadCount = unreadCount;
        }
        if (mentionedUs) {
            toRet.mentionedUs = mentionedUs;
        }
        if (isBlocked) {
            toRet.isBlocked = isBlocked;
        }
        if (hasNickname) {
            toRet.hasNickname = hasNickname;
        }
        if (isKickedFromGroup) {
            toRet.isKickedFromGroup = isKickedFromGroup;
        }
        if (left) {
            toRet.left = left;
        }
        if (isPinned) {
            toRet.isPinned = isPinned;
        }
        if (didApproveMe) {
            toRet.didApproveMe = didApproveMe;
        }
        if (isApproved) {
            toRet.isApproved = isApproved;
        }
        if (subscriberCount) {
            toRet.subscriberCount = subscriberCount;
        }
        if (groupAdmins && groupAdmins.length) {
            toRet.groupAdmins = lodash_1.default.uniq(groupAdmins);
        }
        if (members && members.length) {
            toRet.members = lodash_1.default.uniq(members);
        }
        if (zombies && zombies.length) {
            toRet.zombies = lodash_1.default.uniq(zombies);
        }
        if (expireTimer) {
            toRet.expireTimer = expireTimer;
        }
        if (currentNotificationSetting &&
            currentNotificationSetting !== exports.ConversationNotificationSetting[0]) {
            toRet.currentNotificationSetting = currentNotificationSetting;
        }
        const lastMessageText = this.get('lastMessage');
        if (lastMessageText && lastMessageText.length) {
            const lastMessageStatus = this.get('lastMessageStatus');
            toRet.lastMessage = {
                status: lastMessageStatus,
                text: lastMessageText,
            };
        }
        if (walletAddress) {
            toRet.walletAddress = walletAddress;
        }
        if (walletUserName) {
            toRet.walletUserName = walletUserName;
        }
        if (walletCreatedDaemonHeight) {
            toRet.walletCreatedDaemonHeight = walletCreatedDaemonHeight;
        }
        return toRet;
    }
    async updateGroupAdmins(groupAdmins) {
        const existingAdmins = lodash_1.default.uniq(lodash_1.default.sortBy(this.getGroupAdmins()));
        const newAdmins = lodash_1.default.uniq(lodash_1.default.sortBy(groupAdmins));
        if (lodash_1.default.isEqual(existingAdmins, newAdmins)) {
            return;
        }
        this.set({ groupAdmins });
        await this.commit();
    }
    async onReadMessage(message, readAt) {
        return this.queueJob(() => this.markReadBouncy(message.get('received_at'), {
            sendReadReceipts: false,
            readAt,
        }));
    }
    async getUnread() {
        return (0, data_1.getUnreadByConversation)(this.id);
    }
    async getUnreadCount() {
        const unreadCount = await (0, data_1.getUnreadCountByConversation)(this.id);
        return unreadCount;
    }
    async queueJob(callback) {
        const previous = this.pending || Promise.resolve();
        const taskWithTimeout = (0, TaskWithTimeout_1.createTaskWithTimeout)(callback, `conversation ${this.idForLogging()}`);
        this.pending = previous.then(taskWithTimeout, taskWithTimeout);
        const current = this.pending;
        void current.then(() => {
            if (this.pending === current) {
                delete this.pending;
            }
        });
        return current;
    }
    async getQuoteAttachment(attachments, preview) {
        if (attachments && attachments.length) {
            return Promise.all(attachments
                .filter((attachment) => attachment && attachment.contentType && !attachment.pending && !attachment.error)
                .slice(0, 1)
                .map(async (attachment) => {
                const { fileName, thumbnail, contentType } = attachment;
                return {
                    contentType,
                    fileName: fileName || null,
                    thumbnail: thumbnail
                        ? {
                            ...(await (0, MessageAttachment_1.loadAttachmentData)(thumbnail)),
                            objectUrl: (0, MessageAttachment_1.getAbsoluteAttachmentPath)(thumbnail.path),
                        }
                        : null,
                };
            }));
        }
        if (preview && preview.length) {
            return Promise.all(preview
                .filter((item) => item && item.image)
                .slice(0, 1)
                .map(async (attachment) => {
                const { image } = attachment;
                const { contentType } = image;
                return {
                    contentType,
                    fileName: null,
                    thumbnail: image
                        ? {
                            ...(await (0, MessageAttachment_1.loadAttachmentData)(image)),
                            objectUrl: (0, MessageAttachment_1.getAbsoluteAttachmentPath)(image.path),
                        }
                        : null,
                };
            }));
        }
        return [];
    }
    async makeQuote(quotedMessage) {
        const attachments = quotedMessage.get('attachments');
        const preview = quotedMessage.get('preview');
        const direction = quotedMessage.get('direction');
        const body = quotedMessage.get('body');
        const groupInvitation = quotedMessage.get('groupInvitation');
        const paymentDetails = quotedMessage.get('payment');
        const sharedContactList = quotedMessage.get('sharedContact');
        const quotedAttachments = await this.getQuoteAttachment(attachments, preview);
        if (!quotedMessage.get('sent_at')) {
            window.log.warn('tried to make a quote without a sent_at timestamp');
            return null;
        }
        const quoteObj = {
            author: quotedMessage.getSource(),
            id: `${quotedMessage.get('sent_at')}` || '',
            text: body?.slice(0, 100),
            attachments: quotedAttachments,
            timestamp: quotedMessage.get('sent_at') || 0,
            convoId: this.id,
            direction: direction,
        };
        if (groupInvitation) {
            const { name, url } = groupInvitation;
            const groupInviteTypedData = {
                kind: {
                    '@type': 'OpenGroupInvitation',
                    groupName: name,
                    groupUrl: `${url}`,
                },
            };
            quoteObj.text = JSON.stringify(groupInviteTypedData);
        }
        if (paymentDetails) {
            const { amount, txnId } = paymentDetails;
            const payment = {
                kind: {
                    '@type': 'Payment',
                    amount: amount,
                    txnId: `${txnId}`
                },
            };
            quoteObj.text = JSON.stringify(payment);
        }
        if (sharedContactList) {
            const { address, name } = sharedContactList;
            const sharedContact = {
                kind: {
                    '@type': 'SharedContact',
                    address: address,
                    name: name,
                },
            };
            quoteObj.text = JSON.stringify(sharedContact);
        }
        return quoteObj;
    }
    toOpenGroupV2() {
        if (!this.isOpenGroupV2()) {
            throw new Error('tried to run toOpenGroup for not public group v2');
        }
        return (0, OpenGroupUtils_1.getOpenGroupV2FromConversationId)(this.id);
    }
    async sendReactionJob(sourceMessage, reaction) {
        try {
            const destination = this.id;
            const sentAt = sourceMessage.get('sent_at');
            if (!sentAt) {
                throw new Error('sendReactMessageJob() sent_at must be set.');
            }
            if (this.isPublic() && !this.isOpenGroupV2()) {
                throw new Error('Only opengroupv2 are supported now');
            }
            let sender = utils_1.UserUtils.getOurPubKeyStrFromCache();
            const chatMessageParams = {
                body: '',
                timestamp: sentAt,
                reaction,
                lokiProfile: utils_1.UserUtils.getOurProfile(),
            };
            await this.handleMessageApproval();
            if (this.isOpenGroupV2()) {
                await this.handleOpenGroupV2Message(chatMessageParams);
                await (0, reactions_1.handleMessageReaction)(reaction, sender, true);
                return;
            }
            else {
                await (0, reactions_1.handleMessageReaction)(reaction, sender, false);
            }
            const destinationPubkey = new types_1.PubKey(destination);
            if (this.isPrivate()) {
                const chatMessageMe = new VisibleMessage_1.VisibleMessage({ ...chatMessageParams, syncTarget: this.id });
                await (0, bchat_1.getMessageQueue)().sendSyncMessage(chatMessageMe);
                const chatMessagePrivate = new VisibleMessage_1.VisibleMessage(chatMessageParams);
                await (0, bchat_1.getMessageQueue)().sendToPubKey(destinationPubkey, chatMessagePrivate);
                return;
            }
            if (this.isMediumGroup()) {
                await this.handleMediumGroup(chatMessageParams, destination);
                return;
            }
            if (this.isClosedGroup()) {
                this.handleLegacyClosedGroup();
            }
            throw new TypeError(`Invalid conversation type: '${this.get('type')}'`);
        }
        catch (e) {
            window.log.error(`Reaction job failed id:${reaction.id} error:`, e);
            return null;
        }
    }
    async sendMessageJob(message, expireTimer) {
        try {
            const uploads = await message.uploadData();
            const { id } = message;
            const destination = this.id;
            const sentAt = message.get('sent_at');
            if (!sentAt) {
                throw new Error('sendMessageJob() sent_at must be set.');
            }
            if (this.isPublic() && !this.isOpenGroupV2()) {
                throw new Error('Only opengroupv2 are supported now');
            }
            const chatMessageParams = {
                body: uploads.body,
                identifier: id,
                timestamp: sentAt,
                attachments: uploads.attachments,
                expireTimer,
                preview: uploads.preview,
                quote: uploads.quote,
                lokiProfile: utils_1.UserUtils.getOurProfile(),
            };
            await this.handleMessageApproval();
            if (this.isOpenGroupV2()) {
                await this.handleOpenGroupV2Message(chatMessageParams);
                return;
            }
            if (message.get('sharedContact')) {
                chatMessageParams.sharedContact = message.get('sharedContact');
            }
            const destinationPubkey = new types_1.PubKey(destination);
            if (this.isPrivate()) {
                if (this.isMe()) {
                    chatMessageParams.syncTarget = this.id;
                    const chatMessageMe = new VisibleMessage_1.VisibleMessage(chatMessageParams);
                    await (0, bchat_1.getMessageQueue)().sendSyncMessage(chatMessageMe);
                    return;
                }
                if (message.get('payment')) {
                    chatMessageParams.payment = message.get('payment');
                }
                if (message.get('groupInvitation')) {
                    chatMessageParams.openGroupInvitation = message.get('groupInvitation');
                }
                const chatMessagePrivate = new VisibleMessage_1.VisibleMessage(chatMessageParams);
                await (0, bchat_1.getMessageQueue)().sendToPubKey(destinationPubkey, chatMessagePrivate);
                return;
            }
            if (this.isMediumGroup()) {
                await this.handleMediumGroup(chatMessageParams, destination);
                return;
            }
            if (this.isClosedGroup()) {
                this.handleLegacyClosedGroup();
            }
            throw new TypeError(`Invalid conversation type: '${this.get('type')}'`);
        }
        catch (e) {
            await message.saveErrors(e);
            return null;
        }
    }
    isIncomingRequest() {
        return ConversationModel.hasValidIncomingRequestValues({
            isMe: this.isMe(),
            isApproved: this.isApproved(),
            isBlocked: this.isBlocked(),
            isPrivate: this.isPrivate(),
        });
    }
    isOutgoingRequest() {
        return ConversationModel.hasValidOutgoingRequestValues({
            isMe: this.isMe(),
            isApproved: this.isApproved(),
            didApproveMe: this.didApproveMe(),
            isBlocked: this.isBlocked(),
            isPrivate: this.isPrivate(),
        });
    }
    async addOutgoingApprovalMessage(timestamp) {
        await this.addSingleOutgoingMessage({
            sent_at: timestamp,
            messageRequestResponse: {
                isApproved: 1,
            },
            expireTimer: 0,
        });
        this.updateLastMessage();
    }
    async addIncomingApprovalMessage(timestamp, source) {
        await this.addSingleIncomingMessage({
            sent_at: timestamp,
            source,
            messageRequestResponse: {
                isApproved: 1,
            },
            unread: 1,
            expireTimer: 0,
        });
        this.updateLastMessage();
    }
    async sendMessageRequestResponse(isApproved) {
        if (!this.isPrivate()) {
            return;
        }
        const publicKey = (0, User_1.getOurPubKeyStrFromCache)();
        const timestamp = Date.now();
        const messageRequestResponseParams = {
            timestamp,
            publicKey,
            isApproved,
        };
        const messageRequestResponse = new MessageRequestResponse_1.MessageRequestResponse(messageRequestResponseParams);
        const pubkeyForSending = new types_1.PubKey(this.id);
        await (0, bchat_1.getMessageQueue)()
            .sendToPubKey(pubkeyForSending, messageRequestResponse)
            .catch(window?.log?.error);
    }
    async sendMessage(msg) {
        const { attachments, body, groupInvitation, preview, quote, payment, sharedContact } = msg;
        this.clearTypingTimers();
        const expireTimer = this.get('expireTimer');
        const networkTimestamp = (0, SNodeAPI_1.getNowWithNetworkOffset)();
        window?.log?.info('Sending message to conversation', this.idForLogging(), 'with networkTimestamp: ', networkTimestamp);
        const messageModel = await this.addSingleOutgoingMessage({
            body,
            quote: lodash_1.default.isEmpty(quote) ? undefined : quote,
            preview,
            attachments,
            sent_at: networkTimestamp,
            expireTimer,
            serverTimestamp: this.isPublic() ? Date.now() : undefined,
            groupInvitation,
            payment,
            sharedContact,
        });
        if (!window.isOnline) {
            const error = new Error('Network is not available');
            error.name = 'SendMessageNetworkError';
            error.number = this.id;
            await messageModel.saveErrors([error]);
            await this.commit();
            return;
        }
        this.set({
            lastMessage: messageModel.getNotificationText(),
            lastMessageStatus: 'sending',
            active_at: networkTimestamp,
        });
        await this.commit();
        await this.queueJob(async () => {
            await this.sendMessageJob(messageModel, expireTimer);
        });
    }
    async sendReaction(sourceId, reaction) {
        const sourceMessage = await (0, data_1.getMessageById)(sourceId);
        if (!sourceMessage) {
            return;
        }
        await this.queueJob(async () => {
            await this.sendReactionJob(sourceMessage, reaction);
        });
    }
    async bouncyUpdateLastMessage() {
        if (!this.id) {
            return;
        }
        if (!this.get('active_at')) {
            return;
        }
        const messages = await (0, data_1.getLastMessagesByConversation)(this.id, 1, true);
        const lastMessageModel = messages.at(0);
        const lastMessageJSON = lastMessageModel ? lastMessageModel.toJSON() : null;
        const lastMessageStatusModel = lastMessageModel
            ? lastMessageModel.getMessagePropStatus()
            : undefined;
        const lastMessageUpdate = (0, Conversation_1.createLastMessageUpdate)({
            currentTimestamp: this.get('active_at'),
            lastMessage: lastMessageJSON,
            lastMessageStatus: lastMessageStatusModel,
            lastMessageNotificationText: lastMessageModel
                ? lastMessageModel.getNotificationText()
                : undefined,
        });
        this.set(lastMessageUpdate);
        await this.commit();
    }
    async updateExpireTimer(providedExpireTimer, providedSource, receivedAt, options = {}, shouldCommit = true) {
        let expireTimer = providedExpireTimer;
        let source = providedSource;
        lodash_1.default.defaults(options, { fromSync: false });
        if (!expireTimer) {
            expireTimer = 0;
        }
        if (this.get('expireTimer') === expireTimer || (!expireTimer && !this.get('expireTimer'))) {
            return;
        }
        window?.log?.info("Update conversation 'expireTimer'", {
            id: this.idForLogging(),
            expireTimer,
            source,
        });
        const isOutgoing = Boolean(!receivedAt);
        source = source || utils_1.UserUtils.getOurPubKeyStrFromCache();
        const timestamp = (receivedAt || Date.now()) - 1;
        this.set({ expireTimer });
        const commonAttributes = {
            flags: protobuf_1.SignalService.DataMessage.Flags.EXPIRATION_TIMER_UPDATE,
            expirationTimerUpdate: {
                expireTimer,
                source,
                fromSync: options.fromSync,
            },
            expireTimer: 0,
        };
        let message;
        if (isOutgoing) {
            message = await this.addSingleOutgoingMessage({
                ...commonAttributes,
                sent_at: timestamp,
            });
        }
        else {
            message = await this.addSingleIncomingMessage({
                ...commonAttributes,
                unread: 1,
                source,
                sent_at: timestamp,
                received_at: timestamp,
            });
        }
        if (this.isActive()) {
            this.set('active_at', timestamp);
        }
        if (shouldCommit) {
            await this.commit();
        }
        if (receivedAt) {
            return;
        }
        const expireUpdate = {
            identifier: message.id,
            timestamp,
            expireTimer: expireTimer ? expireTimer : null,
        };
        if (this.isMe()) {
            const expirationTimerMessage = new ExpirationTimerUpdateMessage_1.ExpirationTimerUpdateMessage(expireUpdate);
            return message.sendSyncMessageOnly(expirationTimerMessage);
        }
        if (this.isPrivate()) {
            const expirationTimerMessage = new ExpirationTimerUpdateMessage_1.ExpirationTimerUpdateMessage(expireUpdate);
            const pubkey = new types_1.PubKey(this.get('id'));
            await (0, bchat_1.getMessageQueue)().sendToPubKey(pubkey, expirationTimerMessage);
        }
        else {
            window?.log?.warn('TODO: Expiration update for closed groups are to be updated');
            const expireUpdateForGroup = {
                ...expireUpdate,
                groupId: this.get('id'),
            };
            const expirationTimerMessage = new ExpirationTimerUpdateMessage_1.ExpirationTimerUpdateMessage(expireUpdateForGroup);
            await (0, bchat_1.getMessageQueue)().sendToGroup(expirationTimerMessage);
        }
        return;
    }
    triggerUIRefresh() {
        updatesToDispatch.set(this.id, this.getConversationModelProps());
        throttledAllConversationsDispatch();
    }
    async commit() {
        (0, Performance_1.perfStart)(`conversationCommit-${this.attributes.id}`);
        await (0, data_1.updateConversation)(this.attributes);
        this.triggerUIRefresh();
        (0, Performance_1.perfEnd)(`conversationCommit-${this.attributes.id}`, 'conversationCommit');
    }
    async addSingleOutgoingMessage(messageAttributes) {
        return this.addSingleMessage({
            ...messageAttributes,
            conversationId: this.id,
            source: utils_1.UserUtils.getOurPubKeyStrFromCache(),
            type: 'outgoing',
            direction: 'outgoing',
            unread: 0,
            received_at: messageAttributes.sent_at,
        });
    }
    async addSingleIncomingMessage(messageAttributes) {
        if (!this.didApproveMe() && this.isPrivate()) {
            await this.setDidApproveMe(true);
        }
        return this.addSingleMessage({
            ...messageAttributes,
            conversationId: this.id,
            type: 'incoming',
            direction: 'outgoing',
        });
    }
    async leaveClosedGroup() {
        if (this.isMediumGroup()) {
            await (0, closed_group_1.leaveClosedGroup)(this.id);
        }
        else {
            window?.log?.error('Cannot leave a non-medium group conversation');
            throw new Error('Legacy group are not supported anymore. You need to create this group again.');
        }
    }
    async markReadBouncy(newestUnreadDate, providedOptions = {}) {
        const lastReadTimestamp = this.lastReadTimestamp;
        if (newestUnreadDate < lastReadTimestamp) {
            return;
        }
        const options = providedOptions || {};
        lodash_1.default.defaults(options, { sendReadReceipts: true });
        const conversationId = this.id;
        notifications_1.Notifications.clearByConversationID(conversationId);
        let allUnreadMessagesInConvo = (await this.getUnread()).models;
        const oldUnreadNowRead = allUnreadMessagesInConvo.filter(message => message.get('received_at') <= newestUnreadDate);
        let read = [];
        for (const nowRead of oldUnreadNowRead) {
            nowRead.markReadNoCommit(options.readAt);
            const errors = nowRead.get('errors');
            read.push({
                sender: nowRead.get('source'),
                timestamp: nowRead.get('sent_at'),
                hasErrors: Boolean(errors && errors.length),
            });
        }
        const oldUnreadNowReadAttrs = oldUnreadNowRead.map(m => m.attributes);
        if (oldUnreadNowReadAttrs?.length) {
            await (0, data_1.saveMessages)(oldUnreadNowReadAttrs);
        }
        const allProps = [];
        for (const nowRead of oldUnreadNowRead) {
            allProps.push(nowRead.getMessageModelProps());
        }
        if (allProps.length) {
            window.inboxStore?.dispatch(conversations_2.actions.messagesChanged(allProps));
        }
        read = lodash_1.default.filter(read, m => Boolean(m.sender));
        const realUnreadCount = await this.getUnreadCount();
        if (read.length === 0) {
            const cachedUnreadCountOnConvo = this.get('unreadCount');
            if (cachedUnreadCountOnConvo !== realUnreadCount) {
                this.set({ unreadCount: realUnreadCount });
                await this.commit();
            }
            else {
            }
            return;
        }
        allUnreadMessagesInConvo = allUnreadMessagesInConvo.filter((m) => Boolean(m.isIncoming()));
        this.set({ unreadCount: realUnreadCount });
        const mentionRead = (() => {
            const stillUnread = allUnreadMessagesInConvo.filter((m) => m.get('received_at') > newestUnreadDate);
            const ourNumber = utils_1.UserUtils.getOurPubKeyStrFromCache();
            return !stillUnread.some(m => m.get('body')?.indexOf(`@${ourNumber}`) !== -1);
        })();
        if (mentionRead) {
            this.set({ mentionedUs: false });
        }
        await this.commit();
        read = read.filter(item => !item.hasErrors);
        if (read.length && options.sendReadReceipts) {
            const timestamps = lodash_1.default.map(read, 'timestamp').filter(t => !!t);
            await this.sendReadReceiptsIfNeeded(timestamps);
        }
    }
    async sendReadReceiptsIfNeeded(timestamps) {
        if (!this.isPrivate() || !timestamps.length) {
            return;
        }
        const settingsReadReceiptEnabled = storage_1.Storage.get(settings_key_1.SettingsKey.settingsReadReceipt) || false;
        const sendReceipt = settingsReadReceiptEnabled && !this.isBlocked() && !this.isIncomingRequest();
        if (sendReceipt) {
            window?.log?.info(`Sending ${timestamps.length} read receipts.`);
            const receiptMessage = new ReadReceiptMessage_1.ReadReceiptMessage({
                timestamp: Date.now(),
                timestamps,
            });
            const device = new types_1.PubKey(this.id);
            await (0, bchat_1.getMessageQueue)().sendToPubKey(device, receiptMessage);
        }
    }
    async setNickname(nickname) {
        if (!this.isPrivate()) {
            window.log.info('cannot setNickname to a non private conversation.');
            return;
        }
        const trimmed = nickname && nickname.trim();
        if (this.get('nickname') === trimmed) {
            return;
        }
        const realUserName = this.getBchatProfile()?.displayName;
        if (!trimmed || !trimmed.length) {
            this.set({ nickname: undefined, name: realUserName });
        }
        else {
            this.set({ nickname: trimmed, name: realUserName });
        }
        await this.commit();
        await this.updateProfileName();
    }
    async setBchatProfile(newProfile) {
        if (!lodash_1.default.isEqual(this.get('profile'), newProfile)) {
            this.set({ profile: newProfile });
            await this.commit();
        }
        if (newProfile.avatar) {
            await this.setProfileAvatar({ path: newProfile.avatar }, newProfile.avatarHash);
        }
        await this.updateProfileName();
    }
    async updateProfileName() {
        const nickname = this.getNickname();
        const displayName = this.getBchatProfile()?.displayName;
        const profileName = nickname || displayName || null;
        await this.setProfileName(profileName);
    }
    getBchatProfile() {
        return this.get('profile');
    }
    getNickname() {
        return this.get('nickname');
    }
    isAdmin(pubKey) {
        if (!this.isPublic() && !this.isGroup()) {
            return false;
        }
        if (!pubKey) {
            throw new Error('isAdmin() pubKey is falsy');
        }
        const groupAdmins = this.getGroupAdmins();
        return Array.isArray(groupAdmins) && groupAdmins.includes(pubKey);
    }
    async setProfileName(name) {
        const profileName = this.get('profileName');
        if (profileName !== name) {
            this.set({ profileName: name });
            await this.commit();
        }
    }
    async setIsPinned(value) {
        if (value !== this.isPinned()) {
            this.set({
                isPinned: value,
            });
            await this.commit();
        }
    }
    async setIsApproved(value, shouldCommit = true) {
        if (value !== this.isApproved()) {
            window?.log?.info(`Setting ${(0, onionPath_1.ed25519Str)(this.id)} isApproved to: ${value}`);
            this.set({
                isApproved: value,
            });
            if (shouldCommit) {
                await this.commit();
            }
        }
    }
    async setDidApproveMe(value, shouldCommit = true) {
        if (value !== this.didApproveMe()) {
            window?.log?.info(`Setting ${(0, onionPath_1.ed25519Str)(this.id)} didApproveMe to: ${value}`);
            this.set({
                didApproveMe: value,
            });
            if (shouldCommit) {
                await this.commit();
            }
        }
    }
    async setwalletCreatedDaemonHeight(value, shouldCommit = true) {
        window?.log?.info(`Setting ${(0, onionPath_1.ed25519Str)(this.id)} walletCreatedDaemonHeight to: ${value}`);
        this.set({
            walletCreatedDaemonHeight: value,
        });
        if (shouldCommit) {
            await this.commit();
        }
    }
    async setwalletAddress(value, shouldCommit = true) {
        window?.log?.info(`Setting ${(0, onionPath_1.ed25519Str)(this.id)} walletCreatedDaemonHeight to: ${value}`);
        this.set({
            walletAddress: value,
        });
        if (shouldCommit) {
            await this.commit();
        }
    }
    async setIsBnsHolder(value, shouldCommit = true) {
        window?.log?.info(`Setting ${(0, onionPath_1.ed25519Str)(this.id)} setIsBnsHolder to: ${value}`);
        this.set({
            isBnsHolder: value,
        });
        if (shouldCommit) {
            await this.commit();
        }
    }
    async setSubscriberCount(count) {
        if (this.get('subscriberCount') !== count) {
            this.set({ subscriberCount: count });
            await this.commit();
        }
    }
    async setProfileAvatar(avatar, avatarHash) {
        const profileAvatar = this.get('avatar');
        const existingHash = this.get('avatarHash');
        let shouldCommit = false;
        if (!lodash_1.default.isEqual(profileAvatar, avatar)) {
            this.set({ avatar });
            shouldCommit = true;
        }
        if (existingHash !== avatarHash) {
            this.set({ avatarHash });
            shouldCommit = true;
        }
        if (shouldCommit) {
            await this.commit();
        }
    }
    async setProfileKey(profileKey, autoCommit = true) {
        if (!profileKey) {
            return;
        }
        const profileKeyHex = (0, String_1.toHex)(profileKey);
        if (this.get('profileKey') !== profileKeyHex) {
            this.set({
                profileKey: profileKeyHex,
            });
            if (autoCommit) {
                await this.commit();
            }
        }
    }
    hasMember(pubkey) {
        return lodash_1.default.includes(this.get('members'), pubkey);
    }
    hasReactions() {
        if (this.isPrivate() && !this.isApproved()) {
            return false;
        }
        if (this.isOpenGroupV2()) {
            return false;
        }
        else {
            return true;
        }
    }
    isGroup() {
        return this.get('type') === ConversationTypeEnum.GROUP;
    }
    async removeMessage(messageId) {
        await (0, data_1.removeMessage)(messageId);
        this.updateLastMessage();
        window.inboxStore?.dispatch(conversations_2.actions.messageDeleted({
            conversationKey: this.id,
            messageId,
        }));
    }
    getName() {
        if (this.isPrivate()) {
            return this.get('name');
        }
        return this.get('name') || window.i18n('unknown');
    }
    isPinned() {
        return Boolean(this.get('isPinned'));
    }
    didApproveMe() {
        return Boolean(this.get('didApproveMe'));
    }
    walletCreatedDaemonHeight() {
        return Number(this.get('walletCreatedDaemonHeight'));
    }
    isApproved() {
        return Boolean(this.get('isApproved'));
    }
    getTitle() {
        if (this.isPrivate()) {
            const profileName = this.getProfileName();
            const number = this.getNumber();
            const name = profileName ? `${profileName} (${types_1.PubKey.shorten(number)})` : number;
            return this.get('name') || name;
        }
        return this.get('name') || 'Unknown group';
    }
    getContactProfileNameOrShortenedPubKey() {
        if (!this.isPrivate()) {
            throw new Error('getContactProfileNameOrShortenedPubKey() cannot be called with a non private convo.');
        }
        const profileName = this.get('profileName');
        const pubkey = this.id;
        if (utils_1.UserUtils.isUsFromCache(pubkey)) {
            return window.i18n('you');
        }
        return profileName || types_1.PubKey.shorten(pubkey);
    }
    getContactProfileNameOrFullPubKey() {
        if (!this.isPrivate()) {
            throw new Error('getContactProfileNameOrFullPubKey() cannot be called with a non private convo.');
        }
        const profileName = this.get('profileName');
        const pubkey = this.id;
        if (utils_1.UserUtils.isUsFromCache(pubkey)) {
            return window.i18n('you');
        }
        return profileName || pubkey;
    }
    getProfileName() {
        if (this.isPrivate()) {
            return this.get('profileName');
        }
        return undefined;
    }
    getNumber() {
        if (!this.isPrivate()) {
            return '';
        }
        return this.id;
    }
    getwalletCreatedDaemonHeight() {
        return this.get('walletCreatedDaemonHeight');
    }
    isWalletAddress() {
        return this.attributes.walletAddress;
    }
    bnsHolder() {
        if (!this.isPrivate()) {
            return false;
        }
        return Boolean(this.get('isBnsHolder'));
    }
    isPrivate() {
        return this.get('type') === ConversationTypeEnum.PRIVATE;
    }
    getAvatarPath() {
        const avatar = this.get('avatar') || this.get('profileAvatar');
        if (typeof avatar === 'string') {
            return avatar;
        }
        if (typeof avatar?.path === 'string') {
            return (0, MessageAttachment_1.getAbsoluteAttachmentPath)(avatar.path);
        }
        return null;
    }
    async getNotificationIcon() {
        const avatarUrl = this.getAvatarPath();
        const noIconUrl = 'images/bchat/bchat_logo.png';
        if (avatarUrl) {
            const decryptedAvatarUrl = await (0, DecryptedAttachmentsManager_1.getDecryptedMediaUrl)(avatarUrl, MIME_1.IMAGE_JPEG, true);
            if (!decryptedAvatarUrl) {
                window.log.warn('Could not decrypt avatar stored locally for getNotificationIcon..');
                return noIconUrl;
            }
            return decryptedAvatarUrl;
        }
        else {
            return noIconUrl;
        }
    }
    async notify(message) {
        if (!message.isIncoming()) {
            return;
        }
        const conversationId = this.id;
        let friendRequestText;
        if (!this.isApproved()) {
            window?.log?.info('notification cancelled for unapproved convo', this.idForLogging());
            const hadNoRequestsPrior = (0, conversations_1.getConversationController)()
                .getConversations()
                .filter(conversation => {
                return (!conversation.isApproved() &&
                    !conversation.isBlocked() &&
                    conversation.isPrivate() &&
                    !conversation.isMe());
            }).length === 1;
            const isFirstMessageOfConvo = (await (0, data_1.getMessagesByConversation)(this.id, { messageId: null })).length === 1;
            if (hadNoRequestsPrior && isFirstMessageOfConvo) {
                friendRequestText = window.i18n('youHaveANewFriendRequest');
            }
            else {
                window?.log?.info('notification cancelled for as pending requests already exist', this.idForLogging());
                return;
            }
        }
        const convNotif = this.get('triggerNotificationsFor');
        if (convNotif === 'disabled') {
            window?.log?.info('notifications disabled for convo', this.idForLogging());
            return;
        }
        if (convNotif === 'mentions_only') {
            const regex = new RegExp(`@${types_1.PubKey.regexForPubkeys}`, 'g');
            const text = message.get('body');
            const mentions = text?.match(regex) || [];
            const mentionMe = mentions && mentions.some(m => utils_1.UserUtils.isUsFromCache(m.slice(1)));
            const quotedMessageAuthor = message.get('quote')?.author;
            const isReplyToOurMessage = quotedMessageAuthor && utils_1.UserUtils.isUsFromCache(quotedMessageAuthor);
            if (!mentionMe && !isReplyToOurMessage) {
                window?.log?.info('notifications disabled for non mentions or reply for convo', conversationId);
                return;
            }
        }
        const convo = await (0, conversations_1.getConversationController)().getOrCreateAndWait(message.get('source'), ConversationTypeEnum.PRIVATE);
        const iconUrl = await this.getNotificationIcon();
        const messageJSON = message.toJSON();
        const messageSentAt = messageJSON.sent_at;
        const messageId = message.id;
        const isExpiringMessage = this.isExpiringMessage(messageJSON);
        notifications_1.Notifications.addNotification({
            conversationId,
            iconUrl,
            isExpiringMessage,
            message: friendRequestText ? friendRequestText : message.getNotificationText(),
            messageId,
            messageSentAt,
            title: friendRequestText ? '' : convo.getTitle(),
        });
    }
    async notifyIncomingCall() {
        if (!this.isPrivate()) {
            window?.log?.info('notifyIncomingCall: not a private convo', this.idForLogging());
            return;
        }
        const conversationId = this.id;
        const convNotif = this.get('triggerNotificationsFor');
        if (convNotif === 'disabled') {
            window?.log?.info('notifyIncomingCall: notifications disabled for convo', this.idForLogging());
            return;
        }
        const now = Date.now();
        const iconUrl = await this.getNotificationIcon();
        notifications_1.Notifications.addNotification({
            conversationId,
            iconUrl,
            isExpiringMessage: false,
            message: window.i18n('incomingCallFrom', this.getTitle()),
            messageSentAt: now,
            title: this.getTitle(),
        });
    }
    async notifyTypingNoCommit({ isTyping, sender }) {
        if (utils_1.UserUtils.isUsFromCache(sender)) {
            return;
        }
        if (!this.isPrivate()) {
            return;
        }
        if (this.typingTimer) {
            global.clearTimeout(this.typingTimer);
            this.typingTimer = null;
        }
        this.typingTimer = isTyping
            ? global.setTimeout(this.clearContactTypingTimer.bind(this, sender), 15 * 1000)
            : null;
    }
    async addSingleMessage(messageAttributes) {
        const model = new message_1.MessageModel(messageAttributes);
        const messageId = await model.commit(false);
        model.set({ id: messageId });
        await model.setToExpire();
        const messageModelProps = model.getMessageModelProps();
        window.inboxStore?.dispatch(conversations_2.actions.messagesChanged([messageModelProps]));
        const unreadCount = await this.getUnreadCount();
        this.set({ unreadCount });
        this.updateLastMessage();
        await this.commit();
        return model;
    }
    async clearContactTypingTimer(_sender) {
        if (!!this.typingTimer) {
            global.clearTimeout(this.typingTimer);
            this.typingTimer = null;
            await this.commit();
        }
    }
    isExpiringMessage(json) {
        if (json.type === 'incoming') {
            return false;
        }
        const { expireTimer } = json;
        return isFinite(expireTimer) && expireTimer > 0;
    }
    shouldDoTyping() {
        if (!this.isActive() ||
            !storage_1.Storage.get(settings_key_1.SettingsKey.settingsTypingIndicator) ||
            this.isBlocked() ||
            !this.isPrivate()) {
            return false;
        }
        return Boolean(this.get('isApproved'));
    }
    async bumpTyping() {
        if (!this.shouldDoTyping()) {
            return;
        }
        if (!this.typingRefreshTimer) {
            const isTyping = true;
            this.setTypingRefreshTimer();
            this.sendTypingMessage(isTyping);
        }
        this.setTypingPauseTimer();
    }
    setTypingRefreshTimer() {
        if (this.typingRefreshTimer) {
            global.clearTimeout(this.typingRefreshTimer);
        }
        this.typingRefreshTimer = global.setTimeout(this.onTypingRefreshTimeout.bind(this), 10 * 1000);
    }
    onTypingRefreshTimeout() {
        const isTyping = true;
        this.sendTypingMessage(isTyping);
        this.setTypingRefreshTimer();
    }
    setTypingPauseTimer() {
        if (this.typingPauseTimer) {
            global.clearTimeout(this.typingPauseTimer);
        }
        this.typingPauseTimer = global.setTimeout(this.onTypingPauseTimeout.bind(this), 10 * 1000);
    }
    onTypingPauseTimeout() {
        const isTyping = false;
        this.sendTypingMessage(isTyping);
        this.clearTypingTimers();
    }
    clearTypingTimers() {
        if (this.typingPauseTimer) {
            global.clearTimeout(this.typingPauseTimer);
            this.typingPauseTimer = null;
        }
        if (this.typingRefreshTimer) {
            global.clearTimeout(this.typingRefreshTimer);
            this.typingRefreshTimer = null;
        }
    }
    sendTypingMessage(isTyping) {
        if (!this.isPrivate()) {
            return;
        }
        const recipientId = this.id;
        if (!recipientId) {
            throw new Error('Need to provide either recipientId');
        }
        if (!this.isApproved()) {
            return;
        }
        if (this.isMe()) {
            return;
        }
        const typingParams = {
            timestamp: Date.now(),
            isTyping,
            typingTimestamp: Date.now(),
        };
        const typingMessage = new TypingMessage_1.TypingMessage(typingParams);
        const device = new types_1.PubKey(recipientId);
        (0, bchat_1.getMessageQueue)()
            .sendToPubKey(device, typingMessage)
            .catch(window?.log?.error);
    }
    async handleMessageApproval() {
        const shouldApprove = !this.isApproved() && this.isPrivate();
        const incomingMessageCount = await (0, data_1.getMessageCountByType)(this.id, messageType_1.MessageDirection.incoming);
        const hasIncomingMessages = incomingMessageCount > 0;
        if (shouldApprove) {
            await this.setIsApproved(true);
            if (hasIncomingMessages) {
                await this.addOutgoingApprovalMessage(Date.now());
                if (!this.didApproveMe()) {
                    await this.setDidApproveMe(true);
                }
                await this.sendMessageRequestResponse(true);
                void (0, syncUtils_1.forceSyncConfigurationNowIfNeeded)();
            }
        }
    }
    async handleOpenGroupV2Message(chatMessageParams) {
        const chatMessageOpenGroupV2 = new OpenGroupVisibleMessage_1.OpenGroupVisibleMessage(chatMessageParams);
        const roomInfos = this.toOpenGroupV2();
        if (!roomInfos) {
            throw new Error('Could not find this room in db');
        }
        await (0, bchat_1.getMessageQueue)().sendToOpenGroupV2(chatMessageOpenGroupV2, roomInfos);
    }
    async handleMediumGroup(chatMessageParams, destination) {
        const chatMessageMediumGroup = new VisibleMessage_1.VisibleMessage(chatMessageParams);
        const closedGroupVisibleMessage = new ClosedGroupVisibleMessage_1.ClosedGroupVisibleMessage({
            chatMessage: chatMessageMediumGroup,
            groupId: destination,
        });
        await (0, bchat_1.getMessageQueue)().sendToGroup(closedGroupVisibleMessage);
    }
    handleLegacyClosedGroup() {
        throw new Error('Legacy group are not supported anymore. You need to recreate this group.');
    }
}
exports.ConversationModel = ConversationModel;
const throttledAllConversationsDispatch = lodash_1.default.debounce(() => {
    if (updatesToDispatch.size === 0) {
        return;
    }
    window.inboxStore?.dispatch((0, conversations_2.conversationsChanged)([...updatesToDispatch.values()]));
    updatesToDispatch.clear();
}, 500, { trailing: true, leading: true, maxWait: 1000 });
const updatesToDispatch = new Map();
class ConversationCollection extends backbone_1.default.Collection {
    constructor(models) {
        super(models);
        this.comparator = (m) => {
            return -m.get('active_at');
        };
    }
}
exports.ConversationCollection = ConversationCollection;
ConversationCollection.prototype.model = ConversationModel;
