"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sqlNode = void 0;
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
const rimraf_1 = __importDefault(require("rimraf"));
const BetterSqlite3 = __importStar(require("better-sqlite3"));
const electron_1 = require("electron");
const lodash_1 = require("lodash");
const privacy_1 = require("../util/privacy");
const getRootPath_1 = require("./getRootPath");
const openDbOptions = {
    verbose: false ? console.log : undefined,
    nativeBinding: path_1.default.join((0, getRootPath_1.getAppRootPath)(), 'node_modules', 'better-sqlite3', 'build', 'Release', 'better_sqlite3.node'),
};
const CONVERSATIONS_TABLE = 'conversations';
const MESSAGES_TABLE = 'messages';
const MESSAGES_FTS_TABLE = 'messages_fts';
const NODES_FOR_PUBKEY_TABLE = 'nodesForPubkey';
const OPEN_GROUP_ROOMS_V2_TABLE = 'openGroupRoomsV2';
const IDENTITY_KEYS_TABLE = 'identityKeys';
const GUARD_NODE_TABLE = 'guardNodes';
const ITEMS_TABLE = 'items';
const ATTACHMENT_DOWNLOADS_TABLE = 'attachment_downloads';
const CLOSED_GROUP_V2_KEY_PAIRS_TABLE = 'encryptionKeyPairsForClosedGroupV2';
const LAST_HASHES_TABLE = 'lastHashes';
const RECIPIENT_ADDRESS = 'recipient_address';
const LRU_CACHE_TABLE = 'lru_cache';
const MAX_PUBKEYS_MEMBERS = 300;
const MAX_ENTRIES = 1000;
function objectToJSON(data) {
    return JSON.stringify(data);
}
function jsonToObject(json) {
    return JSON.parse(json);
}
function getSQLiteVersion(db) {
    const { sqlite_version } = db.prepare('select sqlite_version() as sqlite_version').get();
    return sqlite_version;
}
function getSchemaVersion(db) {
    return db.pragma('schema_version', { simple: true });
}
function getSQLCipherVersion(db) {
    return db.pragma('cipher_version', { simple: true });
}
function getSQLCipherIntegrityCheck(db) {
    const rows = db.pragma('cipher_integrity_check');
    if (rows.length === 0) {
        return undefined;
    }
    return rows.map((row) => row.cipher_integrity_check);
}
function keyDatabase(db, key) {
    const deriveKey = HEX_KEY.test(key);
    const value = deriveKey ? `'${key}'` : `"x'${key}'"`;
    const pragramToRun = `key = ${value}`;
    db.pragma(pragramToRun);
}
function switchToWAL(db) {
    db.pragma('journal_mode = WAL');
    db.pragma('synchronous = FULL');
}
function getSQLIntegrityCheck(db) {
    const checkResult = db.pragma('quick_check', { simple: true });
    if (checkResult !== 'ok') {
        return checkResult;
    }
    return undefined;
}
const HEX_KEY = /[^0-9A-Fa-f]/;
function migrateSchemaVersion(db) {
    const userVersion = getUserVersion(db);
    if (userVersion > 0) {
        return;
    }
    const schemaVersion = getSchemaVersion(db);
    const newUserVersion = schemaVersion > 18 ? 16 : schemaVersion;
    console.log('migrateSchemaVersion: Migrating from schema_version ' +
        `${schemaVersion} to user_version ${newUserVersion}`);
    setUserVersion(db, newUserVersion);
}
function getUserVersion(db) {
    try {
        return db.pragma('user_version', { simple: true });
    }
    catch (e) {
        console.error('getUserVersion error', e);
        return 0;
    }
}
function setUserVersion(db, version) {
    if (!(0, lodash_1.isNumber)(version)) {
        throw new Error(`setUserVersion: version ${version} is not a number`);
    }
    db.pragma(`user_version = ${version}`);
}
function openAndMigrateDatabase(filePath, key) {
    let db;
    try {
        db = new BetterSqlite3.default(filePath, openDbOptions);
        keyDatabase(db, key);
        switchToWAL(db);
        migrateSchemaVersion(db);
        db.pragma('secure_delete = ON');
        return db;
    }
    catch (error) {
        if (db) {
            db.close();
        }
        console.log('migrateDatabase: Migration without cipher change failed', error);
    }
    let db1;
    try {
        db1 = new BetterSqlite3.default(filePath, openDbOptions);
        keyDatabase(db1, key);
        db1.pragma('cipher_compatibility = 3');
        migrateSchemaVersion(db1);
        db1.close();
    }
    catch (error) {
        if (db1) {
            db1.close();
        }
        console.log('migrateDatabase: migrateSchemaVersion failed', error);
        return null;
    }
    let db2;
    try {
        db2 = new BetterSqlite3.default(filePath, openDbOptions);
        keyDatabase(db2, key);
        db2.pragma('cipher_migrate');
        switchToWAL(db2);
        db2.pragma('foreign_keys = OFF');
        return db2;
    }
    catch (error) {
        if (db2) {
            db2.close();
        }
        console.log('migrateDatabase: switchToWAL failed');
        return null;
    }
}
function openAndSetUpSQLCipher(filePath, { key }) {
    return openAndMigrateDatabase(filePath, key);
}
function setSQLPassword(password) {
    if (!globalInstance) {
        throw new Error('setSQLPassword: db is not initialized');
    }
    const deriveKey = HEX_KEY.test(password);
    const value = deriveKey ? `'${password}'` : `"x'${password}'"`;
    globalInstance.pragma(`rekey = ${value}`);
}
function vacuumDatabase(db) {
    if (!db) {
        throw new Error('vacuum: db is not initialized');
    }
    const start = Date.now();
    console.info('Vacuuming DB. This might take a while.');
    db.exec('VACUUM;');
    console.info(`Vacuuming DB Finished in ${Date.now() - start}ms.`);
}
function updateToSchemaVersion1(currentVersion, db) {
    if (currentVersion >= 1) {
        return;
    }
    console.log('updateToSchemaVersion1: starting...');
    db.transaction(() => {
        db.exec(`CREATE TABLE ${MESSAGES_TABLE}(
      id STRING PRIMARY KEY ASC,
      json TEXT,

      unread INTEGER,
      expires_at INTEGER,
      sent BOOLEAN,
      sent_at INTEGER,
      schemaVersion INTEGER,
      conversationId STRING,
      received_at INTEGER,
      source STRING,
      sourceDevice STRING,
      hasAttachments INTEGER,
      hasFileAttachments INTEGER,
      hasVisualMediaAttachments INTEGER,
      walletUserName STRING,
      walletAddress STRING
    );

    CREATE INDEX messages_unread ON ${MESSAGES_TABLE} (
      unread
    );

    CREATE INDEX messages_expires_at ON ${MESSAGES_TABLE} (
      expires_at
    );

    CREATE INDEX messages_receipt ON ${MESSAGES_TABLE} (
      sent_at
    );

    CREATE INDEX messages_schemaVersion ON ${MESSAGES_TABLE} (
      schemaVersion
    );

    CREATE INDEX messages_conversation ON ${MESSAGES_TABLE} (
      conversationId,
      received_at
    );

    CREATE INDEX messages_duplicate_check ON ${MESSAGES_TABLE} (
      source,
      sourceDevice,
      sent_at
    );

    CREATE INDEX messages_hasAttachments ON ${MESSAGES_TABLE} (
      conversationId,
      hasAttachments,
      received_at
    );

    CREATE INDEX messages_hasFileAttachments ON ${MESSAGES_TABLE} (
      conversationId,
      hasFileAttachments,
      received_at
    );

    CREATE INDEX messages_hasVisualMediaAttachments ON ${MESSAGES_TABLE} (
      conversationId,
      hasVisualMediaAttachments,
      received_at
    );

    CREATE TABLE unprocessed(
      id STRING,
      timestamp INTEGER,
      json TEXT
    );

    CREATE INDEX unprocessed_id ON unprocessed (
      id
    );

    CREATE INDEX unprocessed_timestamp ON unprocessed (
      timestamp
    );


    `);
        db.pragma('user_version = 1');
    })();
    console.log('updateToSchemaVersion1: success!');
}
function updateToSchemaVersion2(currentVersion, db) {
    if (currentVersion >= 2) {
        return;
    }
    console.log('updateToSchemaVersion2: starting...');
    db.transaction(() => {
        db.exec(`ALTER TABLE ${MESSAGES_TABLE}
     ADD COLUMN expireTimer INTEGER;

     ALTER TABLE ${MESSAGES_TABLE}
     ADD COLUMN expirationStartTimestamp INTEGER;

     ALTER TABLE ${MESSAGES_TABLE}
     ADD COLUMN type STRING;

     CREATE INDEX messages_expiring ON ${MESSAGES_TABLE} (
      expireTimer,
      expirationStartTimestamp,
      expires_at
    );

    UPDATE ${MESSAGES_TABLE} SET
      expirationStartTimestamp = json_extract(json, '$.expirationStartTimestamp'),
      expireTimer = json_extract(json, '$.expireTimer'),
      type = json_extract(json, '$.type');


     `);
        db.pragma('user_version = 2');
    })();
    console.log('updateToSchemaVersion2: success!');
}
function updateToSchemaVersion3(currentVersion, db) {
    if (currentVersion >= 3) {
        return;
    }
    console.log('updateToSchemaVersion3: starting...');
    db.transaction(() => {
        db.exec(`
    DROP INDEX messages_expiring;
    DROP INDEX messages_unread;

    CREATE INDEX messages_without_timer ON ${MESSAGES_TABLE} (
      expireTimer,
      expires_at,
      type
    ) WHERE expires_at IS NULL AND expireTimer IS NOT NULL;

    CREATE INDEX messages_unread ON ${MESSAGES_TABLE} (
      conversationId,
      unread
    ) WHERE unread IS NOT NULL;

    ANALYZE;

    `);
        db.pragma('user_version = 3');
    })();
    console.log('updateToSchemaVersion3: success!');
}
function updateToSchemaVersion4(currentVersion, db) {
    if (currentVersion >= 4) {
        return;
    }
    console.log('updateToSchemaVersion4: starting...');
    db.transaction(() => {
        db.exec(`

    CREATE TABLE ${CONVERSATIONS_TABLE}(
      id STRING PRIMARY KEY ASC,
      json TEXT,

      active_at INTEGER,
      type STRING,
      members TEXT,
      name TEXT,
      profileName TEXT,
      walletUserName STRING,
      walletAddress STRING,
      walletCreatedDaemonHeight INTEGER
    );

    CREATE INDEX conversations_active ON ${CONVERSATIONS_TABLE} (
      active_at
    ) WHERE active_at IS NOT NULL;
    CREATE INDEX conversations_type ON ${CONVERSATIONS_TABLE} (
      type
    ) WHERE type IS NOT NULL;

    `);
        db.pragma('user_version = 4');
    })();
    console.log('updateToSchemaVersion4: success!');
}
function updateToSchemaVersion6(currentVersion, db) {
    if (currentVersion >= 6) {
        return;
    }
    console.log('updateToSchemaVersion6: starting...');
    db.transaction(() => {
        db.exec(`
    CREATE TABLE ${LAST_HASHES_TABLE}(
      snode TEXT PRIMARY KEY,
      hash TEXT,
      expiresAt INTEGER
    );

    CREATE TABLE seenMessages(
      hash TEXT PRIMARY KEY,
      expiresAt INTEGER
    );


    CREATE TABLE bchats(
      id STRING PRIMARY KEY ASC,
      number STRING,
      json TEXT
    );

    CREATE INDEX bchats_number ON bchats (
      number
    ) WHERE number IS NOT NULL;

    CREATE TABLE groups(
      id STRING PRIMARY KEY ASC,
      json TEXT
    );


    CREATE TABLE ${IDENTITY_KEYS_TABLE}(
      id STRING PRIMARY KEY ASC,
      json TEXT
    );

    CREATE TABLE ${ITEMS_TABLE}(
      id STRING PRIMARY KEY ASC,
      json TEXT
    );


    CREATE TABLE preKeys(
      id INTEGER PRIMARY KEY ASC,
      recipient STRING,
      json TEXT
    );


    CREATE TABLE signedPreKeys(
      id INTEGER PRIMARY KEY ASC,
      json TEXT
    );

    CREATE TABLE contactPreKeys(
      id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,
      identityKeyString VARCHAR(255),
      keyId INTEGER,
      json TEXT
    );

    CREATE UNIQUE INDEX contact_prekey_identity_key_string_keyid ON contactPreKeys (
      identityKeyString,
      keyId
    );

    CREATE TABLE contactSignedPreKeys(
      id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,
      identityKeyString VARCHAR(255),
      keyId INTEGER,
      json TEXT
    );

    CREATE UNIQUE INDEX contact_signed_prekey_identity_key_string_keyid ON contactSignedPreKeys (
      identityKeyString,
      keyId
    );

    CREATE TABLE ${RECIPIENT_ADDRESS}(
      address STRING,
      tx_hash STRING
    );



    `);
        db.pragma('user_version = 6');
    })();
    console.log('updateToSchemaVersion6: success!');
}
function updateToSchemaVersion7(currentVersion, db) {
    if (currentVersion >= 7) {
        return;
    }
    console.log('updateToSchemaVersion7: starting...');
    db.transaction(() => {
        db.exec(`
      -- SQLite has been coercing our STRINGs into numbers, so we force it with TEXT
      -- We create a new table then copy the data into it, since we can't modify columns
      DROP INDEX bchats_number;
      ALTER TABLE bchats RENAME TO bchats_old;

      CREATE TABLE bchats(
        id TEXT PRIMARY KEY,
        number TEXT,
        json TEXT
      );
      CREATE INDEX bchats_number ON bchats (
        number
      ) WHERE number IS NOT NULL;
      INSERT INTO bchats(id, number, json)
    SELECT "+" || id, number, json FROM bchats_old;
      DROP TABLE bchats_old;
    `);
        db.pragma('user_version = 7');
    })();
    console.log('updateToSchemaVersion7: success!');
}
function updateToSchemaVersion8(currentVersion, db) {
    if (currentVersion >= 8) {
        return;
    }
    console.log('updateToSchemaVersion8: starting...');
    db.transaction(() => {
        db.exec(`
    -- First, we pull a new body field out of the message table's json blob
    ALTER TABLE ${MESSAGES_TABLE}
      ADD COLUMN body TEXT;
    UPDATE ${MESSAGES_TABLE} SET body = json_extract(json, '$.body');

    -- Then we create our full-text search table and populate it
    CREATE VIRTUAL TABLE ${MESSAGES_FTS_TABLE}
      USING fts5(id UNINDEXED, body);

    INSERT INTO ${MESSAGES_FTS_TABLE}(id, body)
      SELECT id, body FROM ${MESSAGES_TABLE};

    -- Then we set up triggers to keep the full-text search table up to date
    CREATE TRIGGER messages_on_insert AFTER INSERT ON ${MESSAGES_TABLE} BEGIN
      INSERT INTO ${MESSAGES_FTS_TABLE} (
        id,
        body
      ) VALUES (
        new.id,
        new.body
      );
    END;
    CREATE TRIGGER messages_on_delete AFTER DELETE ON ${MESSAGES_TABLE} BEGIN
      DELETE FROM ${MESSAGES_FTS_TABLE} WHERE id = old.id;
    END;
    CREATE TRIGGER messages_on_update AFTER UPDATE ON ${MESSAGES_TABLE} BEGIN
      DELETE FROM ${MESSAGES_FTS_TABLE} WHERE id = old.id;
      INSERT INTO ${MESSAGES_FTS_TABLE}(
        id,
        body
      ) VALUES (
        new.id,
        new.body
      );
    END;

    `);
        db.pragma('user_version = 8');
    })();
    console.log('updateToSchemaVersion8: success!');
}
function updateToSchemaVersion9(currentVersion, db) {
    if (currentVersion >= 9) {
        return;
    }
    console.log('updateToSchemaVersion9: starting...');
    db.transaction(() => {
        db.exec(`
      CREATE TABLE ${ATTACHMENT_DOWNLOADS_TABLE}(
        id STRING primary key,
        timestamp INTEGER,
        pending INTEGER,
        json TEXT
      );

      CREATE INDEX attachment_downloads_timestamp
        ON ${ATTACHMENT_DOWNLOADS_TABLE} (
          timestamp
      ) WHERE pending = 0;
      CREATE INDEX attachment_downloads_pending
        ON ${ATTACHMENT_DOWNLOADS_TABLE} (
          pending
      ) WHERE pending != 0;
    `);
        db.pragma('user_version = 9');
    })();
    console.log('updateToSchemaVersion9: success!');
}
function updateToSchemaVersion10(currentVersion, db) {
    if (currentVersion >= 10) {
        return;
    }
    console.log('updateToSchemaVersion10: starting...');
    db.transaction(() => {
        db.exec(`
      DROP INDEX unprocessed_id;
      DROP INDEX unprocessed_timestamp;
      ALTER TABLE unprocessed RENAME TO unprocessed_old;

      CREATE TABLE unprocessed(
        id STRING,
        timestamp INTEGER,
        version INTEGER,
        attempts INTEGER,
        envelope TEXT,
        decrypted TEXT,
        source TEXT,
        sourceDevice TEXT,
        serverTimestamp INTEGER
      );

      CREATE INDEX unprocessed_id ON unprocessed (
        id
      );
      CREATE INDEX unprocessed_timestamp ON unprocessed (
        timestamp
      );

      INSERT INTO unprocessed (
        id,
        timestamp,
        version,
        attempts,
        envelope,
        decrypted,
        source,
        sourceDevice,
        serverTimestamp
      ) SELECT
        id,
        timestamp,
        json_extract(json, '$.version'),
        json_extract(json, '$.attempts'),
        json_extract(json, '$.envelope'),
        json_extract(json, '$.decrypted'),
        json_extract(json, '$.source'),
        json_extract(json, '$.sourceDevice'),
        json_extract(json, '$.serverTimestamp')
      FROM unprocessed_old;

      DROP TABLE unprocessed_old;
    `);
        db.pragma('user_version = 10');
    })();
    console.log('updateToSchemaVersion10: success!');
}
function updateToSchemaVersion11(currentVersion, db) {
    if (currentVersion >= 11) {
        return;
    }
    console.log('updateToSchemaVersion11: starting...');
    db.transaction(() => {
        db.exec(`
      DROP TABLE groups;
    `);
        db.pragma('user_version = 11');
    })();
    console.log('updateToSchemaVersion11: success!');
}
const SCHEMA_VERSIONS = [
    updateToSchemaVersion1,
    updateToSchemaVersion2,
    updateToSchemaVersion3,
    updateToSchemaVersion4,
    () => null,
    updateToSchemaVersion6,
    updateToSchemaVersion7,
    updateToSchemaVersion8,
    updateToSchemaVersion9,
    updateToSchemaVersion10,
    updateToSchemaVersion11,
];
function updateSchema(db) {
    const sqliteVersion = getSQLiteVersion(db);
    const sqlcipherVersion = getSQLCipherVersion(db);
    const userVersion = getUserVersion(db);
    const maxUserVersion = SCHEMA_VERSIONS.length;
    const schemaVersion = getSchemaVersion(db);
    console.log('updateSchema:');
    console.log(` Current user_version: ${userVersion}`);
    console.log(` Most recent db schema: ${maxUserVersion}`);
    console.log(` SQLite version: ${sqliteVersion}`);
    console.log(` SQLCipher version: ${sqlcipherVersion}`);
    console.log(` (deprecated) schema_version: ${schemaVersion}`);
    for (let index = 0, max = SCHEMA_VERSIONS.length; index < max; index += 1) {
        const runSchemaUpdate = SCHEMA_VERSIONS[index];
        runSchemaUpdate(schemaVersion, db);
    }
    updateBchatSchema(db);
}
const BCHAT_SCHEMA_VERSIONS = [
    updateToBchatSchemaVersion1,
    updateToBchatSchemaVersion2,
    updateToBchatSchemaVersion3,
    updateToBchatSchemaVersion4,
    updateToBchatSchemaVersion5,
    updateToBchatSchemaVersion6,
    updateToBchatSchemaVersion7,
    updateToBchatSchemaVersion8,
    updateToBchatSchemaVersion9,
    updateToBchatSchemaVersion10,
    updateToBchatSchemaVersion11,
    updateToBchatSchemaVersion12,
    updateToBchatSchemaVersion13,
    updateToBchatSchemaVersion14,
    updateToBchatSchemaVersion15,
    updateToBchatSchemaVersion16,
    updateToBchatSchemaVersion17,
    updateToBchatSchemaVersion18,
    updateToBchatSchemaVersion19,
    updateToBchatSchemaVersion20,
    updateToBchatSchemaVersion21,
    updateToBchatSchemaVersion22,
    updateToBchatSchemaVersion23,
    updateToBchatSchemaVersion24
];
function updateToBchatSchemaVersion1(currentVersion, db) {
    const targetVersion = 1;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    ALTER TABLE ${MESSAGES_TABLE}
    ADD COLUMN serverId INTEGER;

    CREATE TABLE servers(
      serverUrl STRING PRIMARY KEY ASC,
      token TEXT
    );
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion2(currentVersion, db) {
    const targetVersion = 2;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    CREATE TABLE pairingAuthorisations(
      id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,
      primaryDevicePubKey VARCHAR(255),
      secondaryDevicePubKey VARCHAR(255),
      isGranted BOOLEAN,
      json TEXT,
      UNIQUE(primaryDevicePubKey, secondaryDevicePubKey)
    );
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion3(currentVersion, db) {
    const targetVersion = 3;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    CREATE TABLE ${GUARD_NODE_TABLE}(
      id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,
      ed25519PubKey VARCHAR(64)
    );
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion4(currentVersion, db) {
    const targetVersion = 4;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    DROP TABLE ${LAST_HASHES_TABLE};
    CREATE TABLE ${LAST_HASHES_TABLE}(
      id TEXT,
      snode TEXT,
      hash TEXT,
      expiresAt INTEGER,
      PRIMARY KEY (id, snode)
    );
    -- Add senderIdentity field to unprocessed needed for medium size groups
    ALTER TABLE unprocessed ADD senderIdentity TEXT;
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion5(currentVersion, db) {
    const targetVersion = 5;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    CREATE TABLE ${NODES_FOR_PUBKEY_TABLE} (
      pubkey TEXT PRIMARY KEY,
      json TEXT
    );

    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion6(currentVersion, db) {
    const targetVersion = 6;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    -- Remove RSS Feed conversations
    DELETE FROM ${CONVERSATIONS_TABLE} WHERE
    type = 'group' AND
    id LIKE 'rss://%';

    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion7(currentVersion, db) {
    const targetVersion = 7;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    -- Remove multi device data

    DELETE FROM pairingAuthorisations;
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion8(currentVersion, db) {
    const targetVersion = 8;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`

    ALTER TABLE ${MESSAGES_TABLE}
    ADD COLUMN serverTimestamp INTEGER;
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion9(currentVersion, db) {
    const targetVersion = 9;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        const rows = db
            .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE} WHERE
      type = 'group' AND
      id LIKE '__textsecure_group__!%';
    `)
            .all();
        const objs = (0, lodash_1.map)(rows, row => jsonToObject(row.json));
        const conversationIdRows = db
            .prepare(`SELECT id FROM ${CONVERSATIONS_TABLE} ORDER BY id ASC;`)
            .all();
        const allOldConversationIds = (0, lodash_1.map)(conversationIdRows, row => row.id);
        objs.forEach(o => {
            const oldId = o.id;
            const newId = oldId.replace('__textsecure_group__!', '');
            console.log(`migrating conversation, ${oldId} to ${newId}`);
            if (allOldConversationIds.includes(newId)) {
                console.log('Found a duplicate conversation after prefix removing. We need to take care of it');
                const countMessagesOld = getMessagesCountByConversation(oldId, db);
                const countMessagesNew = getMessagesCountByConversation(newId, db);
                console.log(`countMessagesOld: ${countMessagesOld}, countMessagesNew: ${countMessagesNew}`);
                const deleteId = countMessagesOld > countMessagesNew ? newId : oldId;
                db.prepare(`DELETE FROM ${CONVERSATIONS_TABLE} WHERE id = $deleteId;`).run({ deleteId });
            }
            const morphedObject = {
                ...o,
                id: newId,
            };
            db.prepare(`UPDATE ${CONVERSATIONS_TABLE} SET
        id = $newId,
        json = $json
        WHERE id = $oldId;`).run({
                newId,
                json: objectToJSON(morphedObject),
                oldId,
            });
        });
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion10(currentVersion, db) {
    const targetVersion = 10;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    CREATE TABLE ${CLOSED_GROUP_V2_KEY_PAIRS_TABLE} (
      id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,
      groupPublicKey TEXT,
      timestamp NUMBER,
      json TEXT
    );

    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion11(currentVersion, db) {
    const targetVersion = 11;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        updateExistingClosedGroupV1ToClosedGroupV2(db);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion12(currentVersion, db) {
    const targetVersion = 12;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    CREATE TABLE ${OPEN_GROUP_ROOMS_V2_TABLE} (
      serverUrl TEXT NOT NULL,
      roomId TEXT NOT NULL,
      conversationId TEXT,
      json TEXT,
      PRIMARY KEY (serverUrl, roomId)
    );

    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion13(currentVersion, db) {
    const targetVersion = 13;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.pragma('secure_delete = ON');
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion14(currentVersion, db) {
    const targetVersion = 14;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
    DROP TABLE IF EXISTS servers;
    DROP TABLE IF EXISTS bchats;
    DROP TABLE IF EXISTS preKeys;
    DROP TABLE IF EXISTS contactPreKeys;
    DROP TABLE IF EXISTS contactSignedPreKeys;
    DROP TABLE IF EXISTS signedPreKeys;
    DROP TABLE IF EXISTS senderKeys;
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion15(currentVersion, db) {
    const targetVersion = 15;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
      DROP TABLE pairingAuthorisations;
      DROP TRIGGER messages_on_delete;
      DROP TRIGGER messages_on_update;
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion16(currentVersion, db) {
    const targetVersion = 16;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
      ALTER TABLE ${MESSAGES_TABLE} ADD COLUMN serverHash TEXT;
      ALTER TABLE ${MESSAGES_TABLE} ADD COLUMN isDeleted BOOLEAN;

      CREATE INDEX messages_serverHash ON ${MESSAGES_TABLE} (
        serverHash
      ) WHERE serverHash IS NOT NULL;

      CREATE INDEX messages_isDeleted ON ${MESSAGES_TABLE} (
        isDeleted
      ) WHERE isDeleted IS NOT NULL;

      ALTER TABLE unprocessed ADD serverHash TEXT;
      CREATE INDEX messages_messageHash ON unprocessed (
        serverHash
      ) WHERE serverHash IS NOT NULL;
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion17(currentVersion, db) {
    const targetVersion = 17;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
      UPDATE ${CONVERSATIONS_TABLE} SET
      json = json_set(json, '$.isApproved', 1)
    `);
        db.exec(`
      UPDATE ${CONVERSATIONS_TABLE} SET
      json = json_remove(json, '$.moderators', '$.dataMessage', '$.accessKey', '$.profileSharing', '$.bchatRestoreSeen')
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function dropFtsAndTriggers(db) {
    console.info('dropping fts5 table');
    db.exec(`
  DROP TRIGGER IF EXISTS messages_on_insert;
  DROP TRIGGER IF EXISTS messages_on_delete;
  DROP TRIGGER IF EXISTS messages_on_update;
  DROP TABLE IF EXISTS ${MESSAGES_FTS_TABLE};
`);
}
function rebuildFtsTable(db) {
    console.info('rebuildFtsTable');
    db.exec(`
    -- Then we create our full-text search table and populate it
    CREATE VIRTUAL TABLE ${MESSAGES_FTS_TABLE}
      USING fts5(id UNINDEXED, body);
    INSERT INTO ${MESSAGES_FTS_TABLE}(id, body)
      SELECT id, body FROM ${MESSAGES_TABLE};
    -- Then we set up triggers to keep the full-text search table up to date
    CREATE TRIGGER messages_on_insert AFTER INSERT ON ${MESSAGES_TABLE} BEGIN
      INSERT INTO ${MESSAGES_FTS_TABLE} (
        id,
        body
      ) VALUES (
        new.id,
        new.body
      );
    END;
    CREATE TRIGGER messages_on_delete AFTER DELETE ON ${MESSAGES_TABLE} BEGIN
      DELETE FROM ${MESSAGES_FTS_TABLE} WHERE id = old.id;
    END;
    CREATE TRIGGER messages_on_update AFTER UPDATE ON ${MESSAGES_TABLE} BEGIN
      DELETE FROM ${MESSAGES_FTS_TABLE} WHERE id = old.id;
      INSERT INTO ${MESSAGES_FTS_TABLE}(
        id,
        body
      ) VALUES (
        new.id,
        new.body
      );
    END;
    `);
    console.info('rebuildFtsTable built');
}
function updateToBchatSchemaVersion18(currentVersion, db) {
    const targetVersion = 18;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        dropFtsAndTriggers(db);
        rebuildFtsTable(db);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion19(currentVersion, db) {
    const targetVersion = 19;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
      DROP INDEX messages_schemaVersion;
      ALTER TABLE ${MESSAGES_TABLE} DROP COLUMN schemaVersion;
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion20(currentVersion, db) {
    const targetVersion = 20;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        const rowsToUpdate = db
            .prepare(`SELECT * FROM ${CONVERSATIONS_TABLE} WHERE type = 'private' AND (name IS NULL or name = '') AND json_extract(json, '$.nickname') <> '';`)
            .all();
        (rowsToUpdate || []).forEach(r => {
            const obj = jsonToObject(r.json);
            if (obj?.nickname?.length && obj?.profile?.displayName?.length) {
                obj.name = obj.profile.displayName;
                updateConversation(obj, db);
            }
        });
        writeBchatSchemaVersion(targetVersion, db);
    });
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion21(currentVersion, db) {
    const targetVersion = 21;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
        UPDATE ${CONVERSATIONS_TABLE} SET
        json = json_set(json, '$.didApproveMe', 1, '$.isApproved', 1)
        WHERE type = 'private';
      `);
        const closedGroups = getAllClosedGroupConversations(db) || [];
        const adminIds = closedGroups.map(g => g.groupAdmins);
        const flattenedAdmins = (0, lodash_1.uniq)((0, lodash_1.flattenDeep)(adminIds)) || [];
        (0, lodash_1.forEach)(flattenedAdmins, id => {
            db.prepare(`
        UPDATE ${CONVERSATIONS_TABLE} SET
        json = json_set(json, '$.didApproveMe', 1, '$.isApproved', 1)
        WHERE id = $id;
      `).run({
                id,
            });
        });
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion22(currentVersion, db) {
    const targetVersion = 22;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`DROP INDEX messages_duplicate_check;`);
        db.exec(`
    ALTER TABLE ${MESSAGES_TABLE} DROP sourceDevice;
    `);
        db.exec(`
    ALTER TABLE unprocessed DROP sourceDevice;
    `);
        db.exec(`
    CREATE INDEX messages_duplicate_check ON ${MESSAGES_TABLE} (
      source,
      sent_at
    );
    `);
        dropFtsAndTriggers(db);
        db.exec(`
        UPDATE ${MESSAGES_TABLE} SET
        json = json_remove(json, '$.schemaVersion', '$.recipients', '$.decrypted_at', '$.sourceDevice', '$.read_by')
      `);
        rebuildFtsTable(db);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion23(currentVersion, db) {
    const targetVersion = 23;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
      ALTER TABLE ${LAST_HASHES_TABLE} RENAME TO ${LAST_HASHES_TABLE}_old;
      CREATE TABLE ${LAST_HASHES_TABLE}(
        id TEXT,
        snode TEXT,
        hash TEXT,
        expiresAt INTEGER,
        namespace INTEGER NOT NULL DEFAULT 0,
        PRIMARY KEY (id, snode, namespace)
      );`);
        db.exec(`INSERT INTO ${LAST_HASHES_TABLE}(id, snode, hash, expiresAt) SELECT id, snode, hash, expiresAt FROM ${LAST_HASHES_TABLE}_old;`);
        db.exec(`DROP TABLE ${LAST_HASHES_TABLE}_old;`);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function updateToBchatSchemaVersion24(currentVersion, db) {
    const targetVersion = 24;
    if (currentVersion >= targetVersion) {
        return;
    }
    console.log(`updateToBchatSchemaVersion${targetVersion}: starting...`);
    db.transaction(() => {
        db.exec(`
     CREATE TABLE ${LRU_CACHE_TABLE} (
    key STRING PRIMARY KEY,
    value STRING,
    accessed_at INTEGER
  );
    `);
        writeBchatSchemaVersion(targetVersion, db);
    })();
    console.log(`updateToBchatSchemaVersion${targetVersion}: success!`);
}
function writeBchatSchemaVersion(newVersion, db) {
    db.prepare(`INSERT INTO bchat_schema(
      version
    ) values (
      $newVersion
    )`).run({ newVersion });
}
function updateBchatSchema(db) {
    const result = db
        .prepare(`SELECT name FROM sqlite_master WHERE type = 'table' AND name='bchat_schema';`)
        .get();
    if (!result) {
        createBchatSchemaTable(db);
    }
    const bchatSchemaVersion = getBchatSchemaVersion(db);
    console.log('updateBchatiSchema:', `Current bchat schema version: ${bchatSchemaVersion};`, `Most recent schema version: ${BCHAT_SCHEMA_VERSIONS.length};`);
    for (let index = 0, max = BCHAT_SCHEMA_VERSIONS.length; index < max; index += 1) {
        const runSchemaUpdate = BCHAT_SCHEMA_VERSIONS[index];
        runSchemaUpdate(bchatSchemaVersion, db);
    }
}
function getBchatSchemaVersion(db) {
    const result = db
        .prepare(`
    SELECT MAX(version) as version FROM bchat_schema;
    `)
        .get();
    if (!result || !result.version) {
        return 0;
    }
    return result.version;
}
function createBchatSchemaTable(db) {
    db.transaction(() => {
        db.exec(`
    CREATE TABLE bchat_schema(
      id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT,
      version INTEGER
    );
    INSERT INTO bchat_schema (
      version
    ) values (
      0
    );
    `);
    })();
}
let globalInstance = null;
function assertGlobalInstance() {
    if (!globalInstance) {
        throw new Error('globalInstance is not initialized.');
    }
    return globalInstance;
}
function assertGlobalInstanceOrInstance(instance) {
    if (!globalInstance && !instance) {
        throw new Error('neither globalInstance nor initialized is initialized.');
    }
    return globalInstance || instance;
}
let databaseFilePath;
function _initializePaths(configDir) {
    const dbDir = path_1.default.join(configDir, 'sql');
    fs_1.default.mkdirSync(dbDir, { recursive: true });
    databaseFilePath = path_1.default.join(dbDir, 'db.sqlite');
}
function showFailedToStart() {
    const notification = new electron_1.Notification({
        title: 'bchat failed to start',
        body: 'Please start from terminal and open a github issue',
    });
    notification.show();
}
async function initializeSql({ configDir, key, messages, passwordAttempt, }) {
    console.info('initializeSql sqlnode');
    if (globalInstance) {
        throw new Error('Cannot initialize more than once!');
    }
    if (!(0, lodash_1.isString)(configDir)) {
        throw new Error('initialize: configDir is required!');
    }
    if (!(0, lodash_1.isString)(key)) {
        throw new Error('initialize: key is required!');
    }
    if (!(0, lodash_1.isObject)(messages)) {
        throw new Error('initialize: message is required!');
    }
    _initializePaths(configDir);
    let db;
    try {
        if (!databaseFilePath) {
            throw new Error('databaseFilePath is not set');
        }
        db = openAndSetUpSQLCipher(databaseFilePath, { key });
        if (!db) {
            throw new Error('db is not set');
        }
        updateSchema(db);
        const cipherIntegrityResult = getSQLCipherIntegrityCheck(db);
        if (cipherIntegrityResult) {
            console.log('Database cipher integrity check failed:', cipherIntegrityResult);
            throw new Error(`Cipher integrity check failed: ${cipherIntegrityResult}`);
        }
        const integrityResult = getSQLIntegrityCheck(db);
        if (integrityResult) {
            console.log('Database integrity check failed:', integrityResult);
            throw new Error(`Integrity check failed: ${integrityResult}`);
        }
        globalInstance = db;
        console.info('total message count before cleaning: ', getMessageCount());
        console.info('total conversation count before cleaning: ', getConversationCount());
        cleanUpOldOpengroups();
        cleanUpUnusedNodeForKeyEntries();
        printDbStats();
        console.info('total message count after cleaning: ', getMessageCount());
        console.info('total conversation count after cleaning: ', getConversationCount());
        vacuumDatabase(db);
    }
    catch (error) {
        console.error('error', error);
        if (passwordAttempt) {
            throw error;
        }
        console.log('Database startup error:', error.stack);
        const button = await electron_1.dialog.showMessageBox({
            buttons: [messages.copyErrorAndQuit, messages.clearAllData],
            defaultId: 0,
            detail: (0, privacy_1.redactAll)(error.stack),
            message: messages.databaseError,
            noLink: true,
            type: 'error',
        });
        if (button.response === 0) {
            electron_1.clipboard.writeText(`Database startup error:\n\n${(0, privacy_1.redactAll)(error.stack)}`);
        }
        else {
            close();
            showFailedToStart();
        }
        electron_1.app.exit(1);
        return false;
    }
    return true;
}
function close() {
    if (!globalInstance) {
        return;
    }
    const dbRef = globalInstance;
    globalInstance = null;
    dbRef.pragma('optimize');
    dbRef.close();
}
function removeDB(configDir = null) {
    if (globalInstance) {
        throw new Error('removeDB: Cannot erase database when it is open!');
    }
    if (!databaseFilePath && configDir) {
        _initializePaths(configDir);
    }
    if (databaseFilePath) {
        rimraf_1.default.sync(databaseFilePath);
        rimraf_1.default.sync(`${databaseFilePath}-shm`);
        rimraf_1.default.sync(`${databaseFilePath}-wal`);
    }
}
const PASS_HASH_ID = 'passHash';
function getPasswordHash() {
    const item = getItemById(PASS_HASH_ID);
    return item && item.value;
}
function savePasswordHash(hash) {
    if ((0, lodash_1.isEmpty)(hash)) {
        removePasswordHash();
        return;
    }
    const data = { id: PASS_HASH_ID, value: hash };
    createOrUpdateItem(data);
}
function removePasswordHash() {
    removeItemById(PASS_HASH_ID);
}
function getIdentityKeyById(id, instance) {
    return getById(IDENTITY_KEYS_TABLE, id, instance);
}
function getGuardNodes() {
    const nodes = assertGlobalInstance()
        .prepare(`SELECT ed25519PubKey FROM ${GUARD_NODE_TABLE};`)
        .all();
    if (!nodes) {
        return null;
    }
    return nodes;
}
function updateGuardNodes(nodes) {
    assertGlobalInstance().transaction(() => {
        assertGlobalInstance().exec(`DELETE FROM ${GUARD_NODE_TABLE}`);
        nodes.map(edkey => assertGlobalInstance()
            .prepare(`INSERT INTO ${GUARD_NODE_TABLE} (
        ed25519PubKey
      ) values ($ed25519PubKey)`)
            .run({
            ed25519PubKey: edkey,
        }));
    })();
}
function createOrUpdateItem(data, instance) {
    createOrUpdate(ITEMS_TABLE, data, instance);
}
function getItemById(id) {
    return getById(ITEMS_TABLE, id);
}
function getAllItems() {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${ITEMS_TABLE} ORDER BY id ASC;`)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function removeItemById(id) {
    removeById(ITEMS_TABLE, id);
    return;
}
function createOrUpdate(table, data, instance) {
    const { id } = data;
    if (!id) {
        throw new Error('createOrUpdate: Provided data did not have a truthy id');
    }
    assertGlobalInstanceOrInstance(instance)
        .prepare(`INSERT OR REPLACE INTO ${table} (
      id,
      json
    ) values (
      $id,
      $json
    )`)
        .run({
        id,
        json: objectToJSON(data),
    });
}
function getById(table, id, instance) {
    const row = assertGlobalInstanceOrInstance(instance)
        .prepare(`SELECT * FROM ${table} WHERE id = $id;`)
        .get({
        id,
    });
    if (!row) {
        return null;
    }
    return jsonToObject(row.json);
}
function removeById(table, id) {
    if (!Array.isArray(id)) {
        assertGlobalInstance()
            .prepare(`DELETE FROM ${table} WHERE id = $id;`)
            .run({ id });
        return;
    }
    if (!id.length) {
        throw new Error('removeById: No ids to delete!');
    }
    assertGlobalInstance()
        .prepare(`DELETE FROM ${table} WHERE id IN ( ${id.map(() => '?').join(', ')} );`)
        .run({ id });
}
function getSwarmNodesForPubkey(pubkey) {
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${NODES_FOR_PUBKEY_TABLE} WHERE pubkey = $pubkey;`)
        .get({
        pubkey,
    });
    if (!row) {
        return [];
    }
    return jsonToObject(row.json);
}
function updateSwarmNodesForPubkey(pubkey, snodeEdKeys) {
    assertGlobalInstance()
        .prepare(`INSERT OR REPLACE INTO ${NODES_FOR_PUBKEY_TABLE} (
        pubkey,
        json
        ) values (
          $pubkey,
          $json
          );`)
        .run({
        pubkey,
        json: objectToJSON(snodeEdKeys),
    });
}
function getConversationCount() {
    const row = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${CONVERSATIONS_TABLE};`)
        .get();
    if (!row) {
        throw new Error(`getConversationCount: Unable to get count of ${CONVERSATIONS_TABLE}`);
    }
    return row['count(*)'];
}
function saveConversation(data, instance) {
    const { id, active_at, type, members, name, profileName, walletUserName, walletCreatedDaemonHeight } = data;
    assertGlobalInstanceOrInstance(instance)
        .prepare(`INSERT INTO ${CONVERSATIONS_TABLE} (
    id,
    json,
    active_at,
    type,
    members,
    name,
    profileName,
    walletUserName,
    walletCreatedDaemonHeight
   
  ) values (
    $id,
    $json,
    $active_at,
    $type,
    $members,
    $name,
    $profileName,
    $walletUserName,
    $walletCreatedDaemonHeight
    
  );`)
        .run({
        id,
        json: objectToJSON(data),
        active_at,
        type,
        members: members ? members.join(' ') : null,
        name,
        profileName,
        walletUserName,
        walletCreatedDaemonHeight
    });
}
function updateConversation(data, instance) {
    const { id, active_at, type, members, name, profileName, } = data;
    assertGlobalInstanceOrInstance(instance)
        .prepare(`UPDATE ${CONVERSATIONS_TABLE} SET
    json = $json,

    active_at = $active_at,
    type = $type,
    members = $members,
    name = $name,
    profileName = $profileName
    WHERE id = $id;`)
        .run({
        id,
        json: objectToJSON(data),
        active_at,
        type,
        members: members ? members.join(' ') : null,
        name,
        profileName,
    });
}
function updateConversationAddress(data, instance) {
    const { id, walletAddress, } = data;
    assertGlobalInstanceOrInstance(instance)
        .prepare(`UPDATE ${CONVERSATIONS_TABLE} SET
    walletAddress = $walletAddress
    WHERE id = $id;`)
        .run({
        id,
        walletAddress
    });
}
function updateWalletAddressInConversation(data, instance) {
    const { id, walletAddress } = data;
    assertGlobalInstanceOrInstance(instance)
        .prepare(`UPDATE ${CONVERSATIONS_TABLE} SET
      walletAddress=$walletAddress
  
    WHERE id = $id;`)
        .run({
        id,
        walletAddress
    });
}
function updateLRUCache(data, instance) {
    const { key } = data;
    const value = objectToJSON(data.value);
    const now = Date.now();
    assertGlobalInstanceOrInstance(instance)
        .prepare(`
      INSERT INTO ${LRU_CACHE_TABLE} (key, value, accessed_at)
      VALUES ($key, $value, $now)
    `).run({ key, value, now });
    const totalRows = assertGlobalInstance().prepare(`
      SELECT COUNT(*) as count FROM ${LRU_CACHE_TABLE}
    `).get().count;
    const overLimit = totalRows - MAX_ENTRIES;
    if (overLimit > 0) {
        assertGlobalInstance().prepare(`
        DELETE FROM ${LRU_CACHE_TABLE}
        WHERE key IN (
          SELECT key FROM ${LRU_CACHE_TABLE}
          ORDER BY accessed_at ASC
          LIMIT ?
        )
      `).run(overLimit);
    }
}
function removeConversation(id) {
    if (!Array.isArray(id)) {
        assertGlobalInstance()
            .prepare(`DELETE FROM ${CONVERSATIONS_TABLE} WHERE id = $id;`)
            .run({
            id,
        });
        return;
    }
    if (!id.length) {
        throw new Error('removeConversation: No ids to delete!');
    }
    assertGlobalInstance()
        .prepare(`DELETE FROM ${CONVERSATIONS_TABLE} WHERE id IN ( ${id.map(() => '?').join(', ')} );`)
        .run(id);
}
function getConversationById(id) {
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${CONVERSATIONS_TABLE} WHERE id = $id;`)
        .get({
        id,
    });
    if (!row) {
        return null;
    }
    return jsonToObject(row.json);
}
function getAllConversations() {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE} ORDER BY id ASC;`)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getAllOpenGroupV1Conversations() {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE} WHERE
      type = 'group' AND
      id LIKE 'publicChat:1@%'
     ORDER BY id ASC;`)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getAllOpenGroupV2Conversations() {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE} WHERE
      type = 'group' AND
      id LIKE 'publicChat:__%@%'
     ORDER BY id ASC;`)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getPubkeysInPublicConversation(conversationId) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT DISTINCT source FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId
     ORDER BY received_at DESC LIMIT ${MAX_PUBKEYS_MEMBERS};`)
        .all({
        conversationId,
    });
    return (0, lodash_1.map)(rows, row => row.source);
}
function getAllGroupsInvolvingId(id) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE} WHERE
      type = 'group' AND
      members LIKE $id
     ORDER BY id ASC;`)
        .all({
        id: `%${id}%`,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function searchConversations(query) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE} WHERE
      (
        name LIKE $name OR
        profileName LIKE $profileName
      ) AND active_at IS NOT NULL AND active_at > 0
     ORDER BY active_at DESC
     LIMIT $limit`)
        .all({
        name: `%${query}%`,
        profileName: `%${query}%`,
        limit: 50,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
const orderByMessageCoalesceClause = `ORDER BY COALESCE(${MESSAGES_TABLE}.serverTimestamp, ${MESSAGES_TABLE}.sent_at, ${MESSAGES_TABLE}.received_at) DESC`;
function searchMessages(query, limit) {
    if (!limit) {
        throw new Error('searchMessages limit must be set');
    }
    const rows = assertGlobalInstance()
        .prepare(`SELECT
      ${MESSAGES_TABLE}.json,
      snippet(${MESSAGES_FTS_TABLE}, -1, '<<left>>', '<<right>>', '...', 5) as snippet
    FROM ${MESSAGES_FTS_TABLE}
    INNER JOIN ${MESSAGES_TABLE} on ${MESSAGES_FTS_TABLE}.id = ${MESSAGES_TABLE}.id
    WHERE
     ${MESSAGES_FTS_TABLE} match $query
    ${orderByMessageCoalesceClause}
    LIMIT $limit;`)
        .all({
        query,
        limit,
    });
    return (0, lodash_1.map)(rows, row => ({
        ...jsonToObject(row.json),
        snippet: row.snippet,
    }));
}
function searchMessagesInConversation(query, conversationId, limit) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT
      ${MESSAGES_TABLE}.json,
      snippet(${MESSAGES_FTS_TABLE}, -1, '<<left>>', '<<right>>', '...', 15) as snippet
    FROM ${MESSAGES_FTS_TABLE}
    INNER JOIN ${MESSAGES_TABLE} on ${MESSAGES_FTS_TABLE}.id = ${MESSAGES_TABLE}.id
    WHERE
    ${MESSAGES_FTS_TABLE} match $query AND
      ${MESSAGES_TABLE}.conversationId = $conversationId
    ${orderByMessageCoalesceClause}
      LIMIT $limit;`)
        .all({
        query,
        conversationId,
        limit: limit || 100,
    });
    return (0, lodash_1.map)(rows, row => ({
        ...jsonToObject(row.json),
        snippet: row.snippet,
    }));
}
function getMessageCount() {
    const row = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${MESSAGES_TABLE};`)
        .get();
    if (!row) {
        throw new Error(`getMessageCount: Unable to get count of ${MESSAGES_TABLE}`);
    }
    return row['count(*)'];
}
function saveMessage(data) {
    const { body, conversationId, expires_at, hasAttachments, hasFileAttachments, hasVisualMediaAttachments, id, serverId, serverTimestamp, received_at, sent, sent_at, source, type, unread, expireTimer, expirationStartTimestamp, walletAddress } = data;
    if (!id) {
        throw new Error('id is required');
    }
    if (!conversationId) {
        throw new Error('conversationId is required');
    }
    const payload = {
        id,
        json: objectToJSON(data),
        serverId,
        serverTimestamp,
        body,
        conversationId,
        expirationStartTimestamp,
        expires_at,
        expireTimer,
        hasAttachments,
        hasFileAttachments,
        hasVisualMediaAttachments,
        received_at,
        sent,
        sent_at,
        source,
        type: type || '',
        unread,
        walletAddress
    };
    assertGlobalInstance()
        .prepare(`INSERT OR REPLACE INTO ${MESSAGES_TABLE} (
    id,
    json,
    serverId,
    serverTimestamp,
    body,
    conversationId,
    expirationStartTimestamp,
    expires_at,
    expireTimer,
    hasAttachments,
    hasFileAttachments,
    hasVisualMediaAttachments,
    received_at,
    sent,
    sent_at,
    source,
    type,
    unread,
    walletAddress
  ) values (
    $id,
    $json,
    $serverId,
    $serverTimestamp,
    $body,
    $conversationId,
    $expirationStartTimestamp,
    $expires_at,
    $expireTimer,
    $hasAttachments,
    $hasFileAttachments,
    $hasVisualMediaAttachments,
    $received_at,
    $sent,
    $sent_at,
    $source,
    $type,
    $unread,
    $walletAddress
  );`)
        .run(payload);
    return id;
}
function saveSeenMessageHashes(arrayOfHashes) {
    assertGlobalInstance().transaction(() => {
        (0, lodash_1.map)(arrayOfHashes, saveSeenMessageHash);
    })();
}
function updateLastHash(data) {
    const { convoId, snode, hash, expiresAt, namespace } = data;
    if (!(0, lodash_1.isNumber)(namespace)) {
        throw new Error('updateLastHash: namespace must be set to a number');
    }
    assertGlobalInstance()
        .prepare(`INSERT OR REPLACE INTO ${LAST_HASHES_TABLE} (
      id,
      snode,
      hash,
      expiresAt,
      namespace
    ) values (
      $id,
      $snode,
      $hash,
      $expiresAt,
      $namespace
    )`)
        .run({
        id: convoId,
        snode,
        hash,
        expiresAt,
        namespace,
    });
}
function saveSeenMessageHash(data) {
    const { expiresAt, hash } = data;
    try {
        assertGlobalInstance()
            .prepare(`INSERT INTO seenMessages (
      expiresAt,
      hash
      ) values (
        $expiresAt,
        $hash
        );`)
            .run({
            expiresAt,
            hash,
        });
    }
    catch (e) {
        console.error('saveSeenMessageHash failed:', e.message);
    }
}
function cleanLastHashes() {
    assertGlobalInstance()
        .prepare(`DELETE FROM ${LAST_HASHES_TABLE} WHERE expiresAt <= $now;`)
        .run({
        now: Date.now(),
    });
}
function cleanSeenMessages() {
    assertGlobalInstance()
        .prepare('DELETE FROM seenMessages WHERE expiresAt <= $now;')
        .run({
        now: Date.now(),
    });
}
function saveMessages(arrayOfMessages) {
    console.info('saveMessages of length: ', arrayOfMessages.length);
    assertGlobalInstance().transaction(() => {
        (0, lodash_1.map)(arrayOfMessages, saveMessage);
    })();
}
function removeMessage(id, instance) {
    if (!Array.isArray(id)) {
        assertGlobalInstanceOrInstance(instance)
            .prepare(`DELETE FROM ${MESSAGES_TABLE} WHERE id = $id;`)
            .run({ id });
        return;
    }
    if (!id.length) {
        throw new Error('removeMessages: No ids to delete!');
    }
    assertGlobalInstanceOrInstance(instance)
        .prepare(`DELETE FROM ${MESSAGES_TABLE} WHERE id IN ( ${id.map(() => '?').join(', ')} );`)
        .run(id);
}
function getMessageIdsFromServerIds(serverIds, conversationId) {
    if (!Array.isArray(serverIds)) {
        return [];
    }
    const validIds = serverIds.map(Number).filter(n => !Number.isNaN(n));
    const rows = assertGlobalInstance()
        .prepare(`SELECT id FROM ${MESSAGES_TABLE} WHERE
    serverId IN (${validIds.join(',')}) AND
    conversationId = $conversationId;`)
        .all({
        conversationId,
    });
    return rows.map(row => row.id);
}
function getMessageById(id) {
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${MESSAGES_TABLE} WHERE id = $id;`)
        .get({
        id,
    });
    if (!row) {
        return null;
    }
    return jsonToObject(row.json);
}
function getMessageBySenderAndSentAt({ source, sentAt }) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${MESSAGES_TABLE} WHERE
      source = $source AND
      sent_at = $sent_at;`)
        .all({
        source,
        sent_at: sentAt,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getMessageByServerId(serverId) {
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${MESSAGES_TABLE} WHERE serverId = $serverId;`)
        .get({
        serverId,
    });
    if (!row) {
        return null;
    }
    return jsonToObject(row.json);
}
function getMessagesCountBySender({ source }) {
    if (!source) {
        throw new Error('source must be set');
    }
    const count = assertGlobalInstance()
        .prepare(`SELECT count(*) FROM ${MESSAGES_TABLE} WHERE
      source = $source;`)
        .get({
        source,
    });
    if (!count) {
        return 0;
    }
    return count['count(*)'] || 0;
}
function getMessageBySenderAndTimestamp({ source, timestamp, }) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${MESSAGES_TABLE} WHERE
      source = $source AND
      sent_at = $timestamp;`)
        .all({
        source,
        timestamp,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function filterAlreadyFetchedOpengroupMessage(msgDetails) {
    return msgDetails.filter(msg => {
        const rows = assertGlobalInstance()
            .prepare(`SELECT source, serverTimestamp  FROM ${MESSAGES_TABLE} WHERE
      source = $sender AND
      serverTimestamp = $serverTimestamp;`)
            .all({
            sender: msg.sender,
            serverTimestamp: msg.serverTimestamp,
        });
        if (rows.length) {
            console.info(`filtering out already received sogs message from ${msg.sender} at ${msg.serverTimestamp} `);
            return false;
        }
        return true;
    });
}
function getUnreadByConversation(conversationId) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${MESSAGES_TABLE} WHERE
      unread = $unread AND
      conversationId = $conversationId
     ORDER BY received_at DESC;`)
        .all({
        unread: 1,
        conversationId,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getUnreadCountByConversation(conversationId) {
    const row = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${MESSAGES_TABLE} WHERE
    unread = $unread AND
    conversationId = $conversationId
    ORDER BY received_at DESC;`)
        .get({
        unread: 1,
        conversationId,
    });
    if (!row) {
        throw new Error(`getUnreadCountByConversation: Unable to get unread count of ${conversationId}`);
    }
    return row['count(*)'];
}
function getMessageCountByType(conversationId, type = '%') {
    const row = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${MESSAGES_TABLE}
      WHERE conversationId = $conversationId
      AND type = $type;`)
        .get({
        conversationId,
        type,
    });
    if (!row) {
        throw new Error(`getIncomingMessagesCountByConversation: Unable to get incoming messages count of ${conversationId}`);
    }
    return row['count(*)'];
}
const orderByClause = 'ORDER BY COALESCE(serverTimestamp, sent_at, received_at) DESC';
const orderByClauseASC = 'ORDER BY COALESCE(serverTimestamp, sent_at, received_at) ASC';
function getMessagesByConversation(conversationId, { messageId = null } = {}) {
    const absLimit = 30;
    const firstUnread = getFirstUnreadMessageIdInConversation(conversationId);
    const numberOfMessagesInConvo = getMessagesCountByConversation(conversationId, globalInstance);
    const floorLoadAllMessagesInConvo = 70;
    if (messageId || firstUnread) {
        const messageFound = getMessageById(messageId || firstUnread);
        if (messageFound && messageFound.conversationId === conversationId) {
            const rows = assertGlobalInstance()
                .prepare(`WITH cte AS (
            SELECT id, conversationId, json, row_number() OVER (${orderByClause}) as row_number
              FROM ${MESSAGES_TABLE} WHERE conversationId = $conversationId
          ), current AS (
          SELECT row_number
            FROM cte
          WHERE id = $messageId

        )
        SELECT cte.*
          FROM cte, current
            WHERE ABS(cte.row_number - current.row_number) <= $limit
          ORDER BY cte.row_number;
          `)
                .all({
                conversationId,
                messageId: messageId || firstUnread,
                limit: numberOfMessagesInConvo < floorLoadAllMessagesInConvo
                    ? floorLoadAllMessagesInConvo
                    : absLimit,
            });
            return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
        }
        console.info(`getMessagesByConversation: Could not find messageId ${messageId} in db with conversationId: ${conversationId}. Just fetching the convo as usual.`);
    }
    const limit = numberOfMessagesInConvo < floorLoadAllMessagesInConvo
        ? floorLoadAllMessagesInConvo
        : absLimit * 2;
    const rows = assertGlobalInstance()
        .prepare(`
    SELECT json FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId
      ${orderByClause}
    LIMIT $limit;
    `)
        .all({
        conversationId,
        limit,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getLastMessagesByConversation(conversationId, limit) {
    if (!(0, lodash_1.isNumber)(limit)) {
        throw new Error('limit must be a number');
    }
    const rows = assertGlobalInstance()
        .prepare(`
    SELECT json FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId
      ${orderByClause}
    LIMIT $limit;
    `)
        .all({
        conversationId,
        limit,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getOldestMessageInConversation(conversationId) {
    const rows = assertGlobalInstance()
        .prepare(`
    SELECT json FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId
      ${orderByClauseASC}
    LIMIT $limit;
    `)
        .all({
        conversationId,
        limit: 1,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function hasConversationOutgoingMessage(conversationId) {
    const row = assertGlobalInstance()
        .prepare(`
    SELECT count(*)  FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId AND
      type IS 'outgoing'
    `)
        .get({
        conversationId,
    });
    if (!row) {
        throw new Error('hasConversationOutgoingMessage: Unable to get coun');
    }
    return Boolean(row['count(*)']);
}
function getFirstUnreadMessageIdInConversation(conversationId) {
    const rows = assertGlobalInstance()
        .prepare(`
    SELECT id FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId AND
      unread = $unread
      ORDER BY serverTimestamp ASC, serverId ASC, sent_at ASC, received_at ASC
    LIMIT 1;
    `)
        .all({
        conversationId,
        unread: 1,
    });
    if (rows.length === 0) {
        return undefined;
    }
    return rows[0].id;
}
function getFirstUnreadMessageWithMention(conversationId, ourpubkey) {
    if (!ourpubkey || !ourpubkey.length) {
        throw new Error('getFirstUnreadMessageWithMention needs our pubkey but nothing was given');
    }
    const likeMatch = `%@${ourpubkey}%`;
    const rows = assertGlobalInstance()
        .prepare(`
    SELECT id, json FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId AND
      unread = $unread AND
      body LIKE $likeMatch
      ORDER BY serverTimestamp ASC, serverId ASC, sent_at ASC, received_at ASC
    LIMIT 1;
    `)
        .all({
        conversationId,
        unread: 1,
        likeMatch,
    });
    if (rows.length === 0) {
        return undefined;
    }
    return rows[0].id;
}
function getMessagesBySentAt(sentAt) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT * FROM ${MESSAGES_TABLE}
     WHERE sent_at = $sent_at
     ORDER BY received_at DESC;`)
        .all({
        sent_at: sentAt,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getLastHashBySnode(convoId, snode, namespace) {
    if (!(0, lodash_1.isNumber)(namespace)) {
        throw new Error('getLastHashBySnode: namespace must be set to a number');
    }
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${LAST_HASHES_TABLE} WHERE snode = $snode AND id = $id AND namespace = $namespace;`)
        .get({
        snode,
        id: convoId,
        namespace,
    });
    if (!row) {
        return null;
    }
    return row.hash;
}
function getSeenMessagesByHashList(hashes) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT * FROM seenMessages WHERE hash IN ( ${hashes.map(() => '?').join(', ')} );`)
        .all(hashes);
    return (0, lodash_1.map)(rows, row => row.hash);
}
function getExpiredMessages() {
    const now = Date.now();
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${MESSAGES_TABLE} WHERE
      expires_at IS NOT NULL AND
      expires_at <= $expires_at
     ORDER BY expires_at ASC;`)
        .all({
        expires_at: now,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getOutgoingWithoutExpiresAt() {
    const rows = assertGlobalInstance()
        .prepare(`
    SELECT json FROM ${MESSAGES_TABLE}
    WHERE
      expireTimer > 0 AND
      expires_at IS NULL AND
      type IS 'outgoing'
    ORDER BY expires_at ASC;
  `)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getNextExpiringMessage() {
    const rows = assertGlobalInstance()
        .prepare(`
    SELECT json FROM ${MESSAGES_TABLE}
    WHERE expires_at > 0
    ORDER BY expires_at ASC
    LIMIT 1;
  `)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function saveUnprocessed(data) {
    const { id, timestamp, version, attempts, envelope, senderIdentity, messageHash } = data;
    if (!id) {
        throw new Error(`saveUnprocessed: id was falsey: ${id}`);
    }
    assertGlobalInstance()
        .prepare(`INSERT OR REPLACE INTO unprocessed (
      id,
      timestamp,
      version,
      attempts,
      envelope,
      senderIdentity,
      serverHash
    ) values (
      $id,
      $timestamp,
      $version,
      $attempts,
      $envelope,
      $senderIdentity,
      $messageHash
    );`)
        .run({
        id,
        timestamp,
        version,
        attempts,
        envelope,
        senderIdentity,
        messageHash,
    });
    return id;
}
function updateUnprocessedAttempts(id, attempts) {
    assertGlobalInstance()
        .prepare('UPDATE unprocessed SET attempts = $attempts WHERE id = $id;')
        .run({
        id,
        attempts,
    });
}
function updateUnprocessedWithData(id, data = {}) {
    const { source, serverTimestamp, decrypted, senderIdentity } = data;
    assertGlobalInstance()
        .prepare(`UPDATE unprocessed SET
      source = $source,
      serverTimestamp = $serverTimestamp,
      decrypted = $decrypted,
      senderIdentity = $senderIdentity
    WHERE id = $id;`)
        .run({
        id,
        source,
        serverTimestamp,
        decrypted,
        senderIdentity,
    });
}
function getUnprocessedById(id) {
    const row = assertGlobalInstance()
        .prepare('SELECT * FROM unprocessed WHERE id = $id;')
        .get({
        id,
    });
    return row;
}
function getUnprocessedCount() {
    const row = assertGlobalInstance()
        .prepare('SELECT count(*) from unprocessed;')
        .get();
    if (!row) {
        throw new Error('getMessageCount: Unable to get count of unprocessed');
    }
    return row['count(*)'];
}
function getAllUnprocessed() {
    const rows = assertGlobalInstance()
        .prepare('SELECT * FROM unprocessed ORDER BY timestamp ASC;')
        .all();
    return rows;
}
function removeUnprocessed(id) {
    if (!Array.isArray(id)) {
        assertGlobalInstance()
            .prepare('DELETE FROM unprocessed WHERE id = $id;')
            .run({ id });
        return;
    }
    if (!id.length) {
        throw new Error('removeUnprocessed: No ids to delete!');
    }
    assertGlobalInstance()
        .prepare(`DELETE FROM unprocessed WHERE id IN ( ${id.map(() => '?').join(', ')} );`)
        .run(id);
}
function removeAllUnprocessed() {
    assertGlobalInstance()
        .prepare('DELETE FROM unprocessed;')
        .run();
}
function getNextAttachmentDownloadJobs(limit) {
    const timestamp = Date.now();
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${ATTACHMENT_DOWNLOADS_TABLE}
    WHERE pending = 0 AND timestamp < $timestamp
    ORDER BY timestamp DESC
    LIMIT $limit;`)
        .all({
        limit,
        timestamp,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function saveAttachmentDownloadJob(job) {
    const { id, pending, timestamp } = job;
    if (!id) {
        throw new Error('saveAttachmentDownloadJob: Provided job did not have a truthy id');
    }
    assertGlobalInstance()
        .prepare(`INSERT OR REPLACE INTO ${ATTACHMENT_DOWNLOADS_TABLE} (
      id,
      pending,
      timestamp,
      json
    ) values (
      $id,
      $pending,
      $timestamp,
      $json
    )`)
        .run({
        id,
        pending,
        timestamp,
        json: objectToJSON(job),
    });
}
function setAttachmentDownloadJobPending(id, pending) {
    assertGlobalInstance()
        .prepare(`UPDATE ${ATTACHMENT_DOWNLOADS_TABLE} SET pending = $pending WHERE id = $id;`)
        .run({
        id,
        pending,
    });
}
function resetAttachmentDownloadPending() {
    assertGlobalInstance()
        .prepare(`UPDATE ${ATTACHMENT_DOWNLOADS_TABLE} SET pending = 0 WHERE pending != 0;`)
        .run();
}
function removeAttachmentDownloadJob(id) {
    removeById(ATTACHMENT_DOWNLOADS_TABLE, id);
}
function removeAllAttachmentDownloadJobs() {
    assertGlobalInstance().exec(`DELETE FROM ${ATTACHMENT_DOWNLOADS_TABLE};`);
}
function removeAll() {
    assertGlobalInstance().exec(`
    DELETE FROM ${IDENTITY_KEYS_TABLE};
    DELETE FROM ${ITEMS_TABLE};
    DELETE FROM unprocessed;
    DELETE FROM ${LAST_HASHES_TABLE};
    DELETE FROM ${NODES_FOR_PUBKEY_TABLE};
    DELETE FROM ${CLOSED_GROUP_V2_KEY_PAIRS_TABLE};
    DELETE FROM seenMessages;
    DELETE FROM ${CONVERSATIONS_TABLE};
    DELETE FROM ${MESSAGES_TABLE};
    DELETE FROM ${ATTACHMENT_DOWNLOADS_TABLE};
    DELETE FROM ${MESSAGES_FTS_TABLE}; 
    DELETE FROM ${RECIPIENT_ADDRESS};  
`);
}
function removeAllWithOutRecipient() {
    assertGlobalInstance().exec(`
    DELETE FROM ${IDENTITY_KEYS_TABLE};
    DELETE FROM ${ITEMS_TABLE};
    DELETE FROM unprocessed;
    DELETE FROM ${LAST_HASHES_TABLE};
    DELETE FROM ${NODES_FOR_PUBKEY_TABLE};
    DELETE FROM ${CLOSED_GROUP_V2_KEY_PAIRS_TABLE};
    DELETE FROM seenMessages;
    DELETE FROM ${CONVERSATIONS_TABLE};
    DELETE FROM ${MESSAGES_TABLE};
    DELETE FROM ${ATTACHMENT_DOWNLOADS_TABLE};
    DELETE FROM ${MESSAGES_FTS_TABLE};  
`);
}
function removeAllConversations() {
    assertGlobalInstance()
        .prepare(`DELETE FROM ${CONVERSATIONS_TABLE};`)
        .run();
}
function getMessagesWithVisualMediaAttachments(conversationId, limit) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId AND
      hasVisualMediaAttachments = 1
     ORDER BY received_at DESC
     LIMIT $limit;`)
        .all({
        conversationId,
        limit,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getMessagesWithFileAttachments(conversationId, limit) {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${MESSAGES_TABLE} WHERE
      conversationId = $conversationId AND
      hasFileAttachments = 1
     ORDER BY received_at DESC
     LIMIT $limit;`)
        .all({
        conversationId,
        limit,
    });
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getExternalFilesForMessage(message) {
    const { attachments, contact, quote, preview } = message;
    const files = [];
    (0, lodash_1.forEach)(attachments, attachment => {
        const { path: file, thumbnail, screenshot } = attachment;
        if (file) {
            files.push(file);
        }
        if (thumbnail && thumbnail.path) {
            files.push(thumbnail.path);
        }
        if (screenshot && screenshot.path) {
            files.push(screenshot.path);
        }
    });
    if (quote && quote.attachments && quote.attachments.length) {
        (0, lodash_1.forEach)(quote.attachments, attachment => {
            const { thumbnail } = attachment;
            if (thumbnail && thumbnail.path) {
                files.push(thumbnail.path);
            }
        });
    }
    if (contact && contact.length) {
        (0, lodash_1.forEach)(contact, item => {
            const { avatar } = item;
            if (avatar && avatar.avatar && avatar.avatar.path) {
                files.push(avatar.avatar.path);
            }
        });
    }
    if (preview && preview.length) {
        (0, lodash_1.forEach)(preview, item => {
            const { image } = item;
            if (image && image.path) {
                files.push(image.path);
            }
        });
    }
    return files;
}
function getExternalFilesForConversation(conversation) {
    const { avatar, profileAvatar } = conversation;
    const files = [];
    if (avatar && avatar.path) {
        files.push(avatar.path);
    }
    if (profileAvatar && profileAvatar.path) {
        files.push(profileAvatar.path);
    }
    return files;
}
function removeKnownAttachments(allAttachments) {
    const lookup = (0, lodash_1.fromPairs)((0, lodash_1.map)(allAttachments, file => [file, true]));
    const chunkSize = 50;
    const total = getMessageCount();
    console.log(`removeKnownAttachments: About to iterate through ${total} messages`);
    let count = 0;
    let complete = false;
    let id = '';
    while (!complete) {
        const rows = assertGlobalInstance()
            .prepare(`SELECT json FROM ${MESSAGES_TABLE}
       WHERE id > $id
       ORDER BY id ASC
       LIMIT $chunkSize;`)
            .all({
            id,
            chunkSize,
        });
        const messages = (0, lodash_1.map)(rows, row => jsonToObject(row.json));
        (0, lodash_1.forEach)(messages, message => {
            const externalFiles = getExternalFilesForMessage(message);
            (0, lodash_1.forEach)(externalFiles, file => {
                delete lookup[file];
            });
        });
        const lastMessage = (0, lodash_1.last)(messages);
        if (lastMessage) {
            ({ id } = lastMessage);
        }
        complete = messages.length < chunkSize;
        count += messages.length;
    }
    console.log(`removeKnownAttachments: Done processing ${count} ${MESSAGES_TABLE}`);
    complete = false;
    count = 0;
    id = 0;
    const conversationTotal = getConversationCount();
    console.log(`removeKnownAttachments: About to iterate through ${conversationTotal} ${CONVERSATIONS_TABLE}`);
    while (!complete) {
        const rows = assertGlobalInstance()
            .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE}
       WHERE id > $id
       ORDER BY id ASC
       LIMIT $chunkSize;`)
            .all({
            id,
            chunkSize,
        });
        const conversations = (0, lodash_1.map)(rows, row => jsonToObject(row.json));
        (0, lodash_1.forEach)(conversations, conversation => {
            const externalFiles = getExternalFilesForConversation(conversation);
            (0, lodash_1.forEach)(externalFiles, file => {
                delete lookup[file];
            });
        });
        const lastMessage = (0, lodash_1.last)(conversations);
        if (lastMessage) {
            ({ id } = lastMessage);
        }
        complete = conversations.length < chunkSize;
        count += conversations.length;
    }
    console.log(`removeKnownAttachments: Done processing ${count} ${CONVERSATIONS_TABLE}`);
    return Object.keys(lookup);
}
function getMessagesCountByConversation(conversationId, instance) {
    const row = assertGlobalInstanceOrInstance(instance)
        .prepare(`SELECT count(*) from ${MESSAGES_TABLE} WHERE conversationId = $conversationId;`)
        .get({ conversationId });
    return row ? row['count(*)'] : 0;
}
function getAllClosedGroupConversations(instance) {
    const rows = assertGlobalInstanceOrInstance(instance)
        .prepare(`SELECT json FROM ${CONVERSATIONS_TABLE} WHERE
      type = 'group' AND
      id NOT LIKE 'publicChat:%'
      ORDER BY id ASC;`)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function remove05PrefixFromStringIfNeeded(str) {
    if (str.length === 66 && str.startsWith('bd')) {
        return str.substr(2);
    }
    return str;
}
function updateExistingClosedGroupV1ToClosedGroupV2(db) {
    const allClosedGroupV1 = getAllClosedGroupConversations(db) || [];
    allClosedGroupV1.forEach(groupV1 => {
        const groupId = groupV1.id;
        try {
            console.log('Migrating closed group v1 to v2: pubkey', groupId);
            const groupV1IdentityKey = getIdentityKeyById(groupId, db);
            if (!groupV1IdentityKey) {
                return;
            }
            const encryptionPubKeyWithoutPrefix = remove05PrefixFromStringIfNeeded(groupV1IdentityKey.id);
            const keyPair = {
                publicHex: encryptionPubKeyWithoutPrefix,
                privateHex: groupV1IdentityKey.secretKey,
            };
            addClosedGroupEncryptionKeyPair(groupId, keyPair, db);
        }
        catch (e) {
            console.error(e);
        }
    });
}
function getAllEncryptionKeyPairsForGroup(groupPublicKey) {
    const rows = getAllEncryptionKeyPairsForGroupRaw(groupPublicKey);
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getAllEncryptionKeyPairsForGroupRaw(groupPublicKey) {
    const pubkeyAsString = groupPublicKey.key
        ? groupPublicKey.key
        : groupPublicKey;
    const rows = assertGlobalInstance()
        .prepare(`SELECT * FROM ${CLOSED_GROUP_V2_KEY_PAIRS_TABLE} WHERE groupPublicKey = $groupPublicKey ORDER BY timestamp ASC;`)
        .all({
        groupPublicKey: pubkeyAsString,
    });
    return rows;
}
function getLatestClosedGroupEncryptionKeyPair(groupPublicKey) {
    const rows = getAllEncryptionKeyPairsForGroup(groupPublicKey);
    if (!rows || rows.length === 0) {
        return undefined;
    }
    return rows[rows.length - 1];
}
function addClosedGroupEncryptionKeyPair(groupPublicKey, keypair, instance) {
    const timestamp = Date.now();
    assertGlobalInstanceOrInstance(instance)
        .prepare(`INSERT OR REPLACE INTO ${CLOSED_GROUP_V2_KEY_PAIRS_TABLE} (
      groupPublicKey,
      timestamp,
        json
        ) values (
          $groupPublicKey,
          $timestamp,
          $json
          );`)
        .run({
        groupPublicKey,
        timestamp,
        json: objectToJSON(keypair),
    });
}
function removeAllClosedGroupEncryptionKeyPairs(groupPublicKey) {
    assertGlobalInstance()
        .prepare(`DELETE FROM ${CLOSED_GROUP_V2_KEY_PAIRS_TABLE} WHERE groupPublicKey = $groupPublicKey`)
        .run({
        groupPublicKey,
    });
}
function getAllV2OpenGroupRooms() {
    const rows = assertGlobalInstance()
        .prepare(`SELECT json FROM ${OPEN_GROUP_ROOMS_V2_TABLE};`)
        .all();
    return (0, lodash_1.map)(rows, row => jsonToObject(row.json));
}
function getV2OpenGroupRoom(conversationId) {
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${OPEN_GROUP_ROOMS_V2_TABLE} WHERE conversationId = $conversationId;`)
        .get({
        conversationId,
    });
    if (!row) {
        return null;
    }
    return jsonToObject(row.json);
}
function getV2OpenGroupRoomByRoomId(serverUrl, roomId) {
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${OPEN_GROUP_ROOMS_V2_TABLE} WHERE serverUrl = $serverUrl AND roomId = $roomId;`)
        .get({
        serverUrl,
        roomId,
    });
    if (!row) {
        return null;
    }
    return jsonToObject(row.json);
}
function saveV2OpenGroupRoom(opengroupsv2Room) {
    const { serverUrl, roomId, conversationId } = opengroupsv2Room;
    assertGlobalInstance()
        .prepare(`INSERT OR REPLACE INTO ${OPEN_GROUP_ROOMS_V2_TABLE} (
      serverUrl,
      roomId,
      conversationId,
      json
    ) values (
      $serverUrl,
      $roomId,
      $conversationId,
      $json
    )`)
        .run({
        serverUrl,
        roomId,
        conversationId,
        json: objectToJSON(opengroupsv2Room),
    });
}
function removeV2OpenGroupRoom(conversationId) {
    assertGlobalInstance()
        .prepare(`DELETE FROM ${OPEN_GROUP_ROOMS_V2_TABLE} WHERE conversationId = $conversationId`)
        .run({
        conversationId,
    });
}
function getEntriesCountInTable(tbl) {
    try {
        const row = assertGlobalInstance()
            .prepare(`SELECT count(*) from ${tbl};`)
            .get();
        return row['count(*)'];
    }
    catch (e) {
        console.warn(e);
        return 0;
    }
}
function printDbStats() {
    [
        'attachment_downloads',
        'conversations',
        'encryptionKeyPairsForClosedGroupV2',
        'guardNodes',
        'identityKeys',
        'items',
        'lastHashes',
        'bchat_schema',
        'messages',
        'messages_fts',
        'messages_fts_config',
        'messages_fts_content',
        'messages_fts_data',
        'messages_fts_docsize',
        'messages_fts_idx',
        'nodesForPubkey',
        'openGroupRoomsV2',
        'recipient_address',
        'seenMessages',
        'sqlite_sequence',
        'sqlite_stat1',
        'sqlite_stat4',
        'unprocessed',
    ].forEach(i => {
        console.log(`${i} count`, getEntriesCountInTable(i));
    });
}
function cleanUpUnusedNodeForKeyEntries() {
    const allIdsToKeep = assertGlobalInstance()
        .prepare(`SELECT id FROM ${CONVERSATIONS_TABLE} WHERE id NOT LIKE 'publicChat:1@%'
    `)
        .all()
        .map(m => m.id) || [];
    const allEntriesInSnodeForPubkey = assertGlobalInstance()
        .prepare(`SELECT pubkey FROM ${NODES_FOR_PUBKEY_TABLE};`)
        .all()
        .map(m => m.pubkey) || [];
    const swarmUnused = (0, lodash_1.difference)(allEntriesInSnodeForPubkey, allIdsToKeep);
    if (swarmUnused.length) {
        const start = Date.now();
        const chunks = (0, lodash_1.chunk)(swarmUnused, 500);
        chunks.forEach(ch => {
            assertGlobalInstance()
                .prepare(`DELETE FROM ${NODES_FOR_PUBKEY_TABLE} WHERE pubkey IN (${ch.map(() => '?').join(',')});`)
                .run(ch);
        });
        console.log(`Removing of ${swarmUnused.length} unused swarms took ${Date.now() - start}ms`);
    }
}
function cleanUpMessagesJson() {
    console.info('cleanUpMessagesJson ');
    const start = Date.now();
    assertGlobalInstance().transaction(() => {
        assertGlobalInstance().exec(`
      UPDATE ${MESSAGES_TABLE} SET
      json = json_remove(json, '$.schemaVersion', '$.recipients', '$.decrypted_at', '$.sourceDevice')
    `);
    })();
    console.info(`cleanUpMessagesJson took ${Date.now() - start}ms`);
}
function cleanUpOldOpengroups() {
    const ourNumber = getItemById('number_id');
    if (!ourNumber || !ourNumber.value) {
        console.info('cleanUpOldOpengroups: ourNumber is not set');
        return;
    }
    const pruneSetting = getItemById('prune-setting')?.value;
    if (pruneSetting === undefined) {
        console.info('Prune settings is undefined, skipping cleanUpOldOpengroups but we will need to ask user');
        return;
    }
    if (!pruneSetting) {
        console.info('Prune setting not enabled, skipping cleanUpOldOpengroups');
        return;
    }
    const v2Convos = getAllOpenGroupV2Conversations();
    if (!v2Convos || !v2Convos.length) {
        console.info('cleanUpOldOpengroups: v2Convos is empty');
        return;
    }
    const maxMessagePerOpengroupConvo = 2000;
    const db = assertGlobalInstance();
    db.transaction(() => {
        dropFtsAndTriggers(db);
        v2Convos.forEach(convo => {
            const convoId = convo.id;
            const messagesInConvoBefore = getMessagesCountByConversation(convoId);
            if (messagesInConvoBefore >= maxMessagePerOpengroupConvo) {
                const minute = 1000 * 60;
                const sixMonths = minute * 60 * 24 * 30 * 6;
                const limitTimestamp = Date.now() - sixMonths;
                const countToRemove = assertGlobalInstance()
                    .prepare(`SELECT count(*) from ${MESSAGES_TABLE} WHERE serverTimestamp <= $serverTimestamp AND conversationId = $conversationId;`)
                    .get({ conversationId: convoId, serverTimestamp: limitTimestamp })['count(*)'];
                const start = Date.now();
                assertGlobalInstance()
                    .prepare(`
        DELETE FROM ${MESSAGES_TABLE} WHERE serverTimestamp <= $serverTimestamp AND conversationId = $conversationId`)
                    .run({ conversationId: convoId, serverTimestamp: limitTimestamp });
                const messagesInConvoAfter = getMessagesCountByConversation(convoId);
                console.info(`Cleaning ${countToRemove} messages older than 6 months in public convo: ${convoId} took ${Date.now() -
                    start}ms. Old message count: ${messagesInConvoBefore}, new message count: ${messagesInConvoAfter}`);
                const unreadCount = getUnreadCountByConversation(convoId);
                const convoProps = getConversationById(convoId);
                if (convoProps) {
                    convoProps.unreadCount = unreadCount;
                    updateConversation(convoProps);
                }
            }
        });
        const allInactiveConvos = assertGlobalInstance()
            .prepare(`
    SELECT id FROM ${CONVERSATIONS_TABLE} WHERE type = 'private' AND (active_at IS NULL OR active_at = 0)`)
            .all();
        const ourPubkey = ourNumber.value.split('.')[0];
        const allInactiveAndWithoutMessagesConvo = allInactiveConvos
            .map(c => c.id)
            .filter(convoId => {
            return convoId !== ourPubkey && getMessagesCountBySender({ source: convoId }) === 0
                ? true
                : false;
        });
        if (allInactiveAndWithoutMessagesConvo.length) {
            console.info(`Removing ${allInactiveAndWithoutMessagesConvo.length} completely inactive convos`);
            const start = Date.now();
            const chunks = (0, lodash_1.chunk)(allInactiveAndWithoutMessagesConvo, 500);
            chunks.forEach(ch => {
                db.prepare(`DELETE FROM ${CONVERSATIONS_TABLE} WHERE id IN (${ch.map(() => '?').join(',')});`).run(ch);
            });
            console.info(`Removing of ${allInactiveAndWithoutMessagesConvo.length} completely inactive convos done in ${Date.now() - start}ms`);
        }
        cleanUpMessagesJson();
        rebuildFtsTable(db);
    })();
}
function fillWithTestData(numConvosToAdd, numMsgsToAdd) {
    const convoBeforeCount = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${CONVERSATIONS_TABLE};`)
        .get()['count(*)'];
    const lipsum = `Lorem ipsum dolor sit amet, consectetur adipiscing elit. Duis ac ornare lorem,
    non suscipit      purus. Lorem ipsum dolor sit amet, consectetur adipiscing elit.
    Suspendisse cursus aliquet       velit a dignissim. Integer at nisi sed velit consequat
    dictum. Phasellus congue tellus ante.        Ut rutrum hendrerit dapibus. Fusce
    luctus, ante nec interdum molestie, purus urna volutpat         turpis, eget mattis
    lectus velit at velit. Praesent vel tellus turpis. Praesent eget purus          at
    nisl blandit pharetra.  Cras dapibus sem vitae rutrum dapibus. Vivamus vitae mi
    ante.           Donec aliquam porta nibh, vel scelerisque orci condimentum sed.
    Proin in mattis ipsum,            ac euismod sem. Donec malesuada sem nisl, at
    vehicula ante efficitur sed. Curabitur             in sapien eros. Morbi tempor ante ut
    metus scelerisque condimentum. Integer sit amet              tempus nulla. Vivamus
    imperdiet dui ac luctus vulputate.  Sed a accumsan risus. Nulla               facilisi.
    Nulla mauris dui, luctus in sagittis at, sodales id mauris. Integer efficitur
            viverra ex, ut dignissim eros tincidunt placerat. Sed facilisis gravida
    mauris in luctus                . Fusce dapibus, est vitae tincidunt eleifend, justo
    odio porta dui, sed ultrices mi arcu                 vitae ante. Mauris ut libero
    erat. Nam ut mi quis ante tincidunt facilisis sit amet id enim.
    Vestibulum in molestie mi. In ac felis est. Vestibulum vel blandit ex. Morbi vitae
    viverra augue                  . Ut turpis quam, cursus quis ex a, convallis
    ullamcorper purus.  Nam eget libero arcu. Integer fermentum enim nunc, non consequat urna
    fermentum condimentum. Nulla vitae malesuada est. Donec imperdiet tortor interdum
    malesuada feugiat. Integer pulvinar dui ex, eget tristique arcu mattis at. Nam eu neque
    eget mauris varius suscipit. Quisque ac enim vitae mauris laoreet congue nec sed
    justo. Curabitur fermentum quam eget est tincidunt, at faucibus lacus maximus.  Donec
    auctor enim dolor, faucibus egestas diam consectetur sed. Donec eget rutrum arcu, at
    tempus mi. Fusce quis volutpat sapien. In aliquet fringilla purus. Ut eu nunc non
    augue lacinia ultrices at eget tortor. Maecenas pulvinar odio sit amet purus
    elementum, a vehicula lorem maximus. Pellentesque eu lorem magna. Vestibulum ut facilisis
    lorem. Proin et enim cursus, vulputate neque sit amet, posuere enim. Praesent
    faucibus tellus vel mi tincidunt, nec malesuada nibh malesuada. In laoreet sapien vitae
    aliquet sollicitudin.
    `;
    const msgBeforeCount = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${MESSAGES_TABLE};`)
        .get()['count(*)'];
    console.info('==== fillWithTestData ====');
    console.info({
        convoBeforeCount,
        msgBeforeCount,
        convoToAdd: numConvosToAdd,
        msgToAdd: numMsgsToAdd,
    });
    const convosIdsAdded = [];
    for (let index = 0; index < numConvosToAdd; index++) {
        const activeAt = Date.now() - index;
        const id = Date.now() - 1000 * index;
        const convoObjToAdd = {
            active_at: activeAt,
            members: [],
            profileName: `${activeAt}`,
            name: `${activeAt}`,
            id: `bd${id}`,
            type: 'group',
        };
        convosIdsAdded.push(id);
        try {
            saveConversation(convoObjToAdd);
        }
        catch (e) {
            console.warn(e);
        }
    }
    for (let index = 0; index < numMsgsToAdd; index++) {
        const activeAt = Date.now() - index;
        const id = Date.now() - 1000 * index;
        const lipsumStartIdx = Math.floor(Math.random() * lipsum.length);
        const lipsumLength = Math.floor(Math.random() * lipsum.length - lipsumStartIdx);
        const fakeBodyText = lipsum.substring(lipsumStartIdx, lipsumStartIdx + lipsumLength);
        const convoId = convosIdsAdded[Math.floor(Math.random() * convosIdsAdded.length)];
        const msgObjToAdd = {
            body: `fakeMsgIdx-spongebob-${index} ${fakeBodyText} ${activeAt}`,
            conversationId: `bd${id}`,
            expires_at: 0,
            hasAttachments: 0,
            hasFileAttachments: 0,
            hasVisualMediaAttachments: 0,
            id: `${id}`,
            serverId: 0,
            serverTimestamp: 0,
            received_at: Date.now(),
            sent: 0,
            sent_at: Date.now(),
            source: `${convoId}`,
            type: 'outgoing',
            unread: 1,
            expireTimer: 0,
            expirationStartTimestamp: 0,
        };
        if (convoId % 10 === 0) {
            console.info('uyo , convoId ', { index, convoId });
        }
        try {
            saveMessage(msgObjToAdd);
        }
        catch (e) {
            console.error(e);
        }
    }
    const convoAfterCount = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${CONVERSATIONS_TABLE};`)
        .get()['count(*)'];
    const msgAfterCount = assertGlobalInstance()
        .prepare(`SELECT count(*) from ${MESSAGES_TABLE};`)
        .get()['count(*)'];
    console.info({ convoAfterCount, msgAfterCount });
    return convosIdsAdded;
}
function saveRecipientAddress(data) {
    const { tx_hash, address } = data;
    assertGlobalInstance()
        .prepare(`INSERT INTO ${RECIPIENT_ADDRESS} (
        address,
    tx_hash
    ) values (
      $address,
    $tx_hash
    );`)
        .run({
        address,
        tx_hash,
    });
    return;
}
function getRecipientAddress(tx_hash) {
    const row = assertGlobalInstance()
        .prepare(`SELECT * FROM ${RECIPIENT_ADDRESS} WHERE tx_hash = $tx_hash;`)
        .get({
        tx_hash,
    });
    if (!row) {
        return [];
    }
    return row;
}
function getLRUCache(key) {
    const row = assertGlobalInstance()
        .prepare(`
    SELECT value FROM ${LRU_CACHE_TABLE} WHERE key = $key
  `).get({ key });
    if (row) {
        assertGlobalInstance().prepare(`UPDATE ${LRU_CACHE_TABLE} SET accessed_at = ? WHERE key = ?`)
            .run(Date.now(), key);
        return jsonToObject(row.value);
    }
    return null;
}
exports.sqlNode = {
    initializeSql,
    close,
    removeDB,
    setSQLPassword,
    getPasswordHash,
    savePasswordHash,
    removePasswordHash,
    getIdentityKeyById,
    createOrUpdateItem,
    getItemById,
    getAllItems,
    removeItemById,
    getSwarmNodesForPubkey,
    updateSwarmNodesForPubkey,
    getGuardNodes,
    updateGuardNodes,
    getConversationCount,
    saveConversation,
    getConversationById,
    updateConversation,
    updateConversationAddress,
    updateWalletAddressInConversation,
    removeConversation,
    getAllConversations,
    getAllOpenGroupV1Conversations,
    getAllOpenGroupV2Conversations,
    getPubkeysInPublicConversation,
    getAllGroupsInvolvingId,
    removeAllConversations,
    cleanUpOldOpengroups,
    searchConversations,
    searchMessages,
    searchMessagesInConversation,
    getMessageCount,
    saveMessage,
    cleanSeenMessages,
    cleanLastHashes,
    saveSeenMessageHashes,
    saveSeenMessageHash,
    updateLastHash,
    saveMessages,
    removeMessage,
    getUnreadByConversation,
    getUnreadCountByConversation,
    getMessageCountByType,
    getMessageBySenderAndSentAt,
    filterAlreadyFetchedOpengroupMessage,
    getMessageBySenderAndTimestamp,
    getMessageIdsFromServerIds,
    getMessageById,
    getMessageByServerId,
    getMessagesBySentAt,
    getSeenMessagesByHashList,
    getLastHashBySnode,
    getExpiredMessages,
    getOutgoingWithoutExpiresAt,
    getNextExpiringMessage,
    getMessagesByConversation,
    getLastMessagesByConversation,
    getOldestMessageInConversation,
    getFirstUnreadMessageIdInConversation,
    getFirstUnreadMessageWithMention,
    hasConversationOutgoingMessage,
    fillWithTestData,
    getUnprocessedCount,
    getAllUnprocessed,
    saveUnprocessed,
    updateUnprocessedAttempts,
    updateUnprocessedWithData,
    getUnprocessedById,
    removeUnprocessed,
    removeAllUnprocessed,
    getNextAttachmentDownloadJobs,
    saveAttachmentDownloadJob,
    setAttachmentDownloadJobPending,
    resetAttachmentDownloadPending,
    removeAttachmentDownloadJob,
    removeAllAttachmentDownloadJobs,
    removeKnownAttachments,
    removeAll,
    removeAllWithOutRecipient,
    getMessagesWithVisualMediaAttachments,
    getMessagesWithFileAttachments,
    getAllEncryptionKeyPairsForGroup,
    getLatestClosedGroupEncryptionKeyPair,
    addClosedGroupEncryptionKeyPair,
    removeAllClosedGroupEncryptionKeyPairs,
    getV2OpenGroupRoom,
    saveV2OpenGroupRoom,
    getAllV2OpenGroupRooms,
    getV2OpenGroupRoomByRoomId,
    removeV2OpenGroupRoom,
    getRecipientAddress,
    saveRecipientAddress,
    updateLRUCache,
    getLRUCache
};
