"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.queueAttachmentDownloads = exports.downloadAttachmentOpenGroupV2 = exports.downloadDataFromOpenGroupV2 = exports.downloadAttachment = void 0;
const lodash_1 = __importDefault(require("lodash"));
const data_1 = require("../../ts/data/data");
const utils_1 = require("../bchat/utils");
const OpenGroupAPIV2_1 = require("../bchat/apis/open_group_api/opengroupV2/OpenGroupAPIV2");
const file_server_api_1 = require("../bchat/apis/file_server_api");
const BufferPadding_1 = require("../bchat/crypto/BufferPadding");
const attachmentsEncrypter_1 = require("../util/crypto/attachmentsEncrypter");
const util_worker_interface_1 = require("../webworker/workers/util_worker_interface");
async function downloadAttachment(attachment) {
    const asURL = new URL(attachment.url);
    const serverUrl = asURL.origin;
    const defaultFsV2 = lodash_1.default.startsWith(serverUrl, file_server_api_1.FSv2.fileServerV2URL);
    let res = null;
    if (defaultFsV2) {
        let attachmentId = attachment.id;
        if (!attachmentId) {
            attachmentId = attachment.url;
        }
        window?.log?.info('Download v2 file server attachment', attachmentId);
        res = await file_server_api_1.FSv2.downloadFileFromFSv2(attachmentId);
    }
    else {
        window.log.warn(`downloadAttachment attachment is neither opengroup attachment nor fsv2... Dropping it ${asURL.href}`);
        throw new Error('Attachment url is not opengroupv2 nor fileserver v2. Unsupported');
    }
    if (!res?.byteLength) {
        window?.log?.error('Failed to download attachment. Length is 0');
        throw new Error(`Failed to download attachment. Length is 0 for ${attachment.url}`);
    }
    let data = res;
    if (!attachment.isRaw) {
        const { key, digest, size } = attachment;
        if (!key || !digest) {
            throw new Error('Attachment is not raw but we do not have a key to decode it');
        }
        if (!size) {
            throw new Error('Attachment expected size is 0');
        }
        const keyBuffer = (await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', key));
        const digestBuffer = (await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', digest));
        data = await (0, attachmentsEncrypter_1.decryptAttachment)(data, keyBuffer, digestBuffer);
        if (size !== data.byteLength) {
            const unpaddedData = (0, BufferPadding_1.getUnpaddedAttachment)(data, size);
            if (!unpaddedData) {
                throw new Error(`downloadAttachment: Size ${size} did not match downloaded attachment size ${data.byteLength}`);
            }
            data = unpaddedData;
        }
    }
    return {
        ...lodash_1.default.omit(attachment, 'digest', 'key'),
        data,
    };
}
exports.downloadAttachment = downloadAttachment;
async function downloadDataFromOpenGroupV2(fileUrl, roomInfos) {
    const dataUintFromUrl = await (0, OpenGroupAPIV2_1.downloadFileOpenGroupV2ByUrl)(fileUrl, roomInfos);
    if (!dataUintFromUrl?.length) {
        window?.log?.error('Failed to download attachment. Length is 0');
        throw new Error(`Failed to download attachment. Length is 0 for ${fileUrl}`);
    }
    return dataUintFromUrl;
}
exports.downloadDataFromOpenGroupV2 = downloadDataFromOpenGroupV2;
async function downloadAttachmentOpenGroupV2(attachment, roomInfos) {
    const dataUint = await (0, OpenGroupAPIV2_1.downloadFileOpenGroupV2)(attachment.id, roomInfos);
    if (!dataUint?.length) {
        window?.log?.error('Failed to download attachment. Length is 0');
        throw new Error(`Failed to download attachment. Length is 0 for ${attachment.url}`);
    }
    let data = dataUint;
    if (attachment.size !== dataUint.length) {
        const unpaddedData = (0, BufferPadding_1.getUnpaddedAttachment)(dataUint.buffer, attachment.size);
        if (!unpaddedData) {
            throw new Error(`downloadAttachment: Size ${attachment.size} did not match downloaded attachment size ${data.byteLength}`);
        }
        data = new Uint8Array(unpaddedData);
    }
    else {
        window?.log?.info('Received opengroupv2 unpadded attachment size:', attachment.size);
    }
    return {
        ...lodash_1.default.omit(attachment, 'digest', 'key'),
        data: data.buffer,
    };
}
exports.downloadAttachmentOpenGroupV2 = downloadAttachmentOpenGroupV2;
async function processNormalAttachments(message, normalAttachments, convo) {
    const isOpenGroupV2 = convo.isOpenGroupV2();
    if (message.isTrustedForAttachmentDownload()) {
        const openGroupV2Details = (isOpenGroupV2 && convo.toOpenGroupV2()) || undefined;
        const attachments = await Promise.all(normalAttachments.map(async (attachment, index) => {
            return utils_1.AttachmentDownloads.addJob(attachment, {
                messageId: message.id,
                type: 'attachment',
                index,
                isOpenGroupV2,
                openGroupV2Details,
            });
        }));
        message.set({ attachments });
        return attachments.length;
    }
    window.log.info('No downloading attachments yet as this user is not trusted for now.');
    return 0;
}
async function processPreviews(message, convo) {
    let addedCount = 0;
    const isOpenGroupV2 = convo.isOpenGroupV2();
    const openGroupV2Details = (isOpenGroupV2 && convo.toOpenGroupV2()) || undefined;
    const preview = await Promise.all((message.get('preview') || []).map(async (item, index) => {
        if (!item.image) {
            return item;
        }
        addedCount += 1;
        const image = message.isTrustedForAttachmentDownload()
            ? await utils_1.AttachmentDownloads.addJob(item.image, {
                messageId: message.id,
                type: 'preview',
                index,
                isOpenGroupV2,
                openGroupV2Details,
            })
            : null;
        return { ...item, image };
    }));
    message.set({ preview });
    return addedCount;
}
async function processQuoteAttachments(message, convo) {
    let addedCount = 0;
    const quote = message.get('quote');
    if (!quote || !quote.attachments || !quote.attachments.length) {
        return 0;
    }
    const isOpenGroupV2 = convo.isOpenGroupV2();
    const openGroupV2Details = (isOpenGroupV2 && convo.toOpenGroupV2()) || undefined;
    quote.attachments = await Promise.all(quote.attachments.map(async (item, index) => {
        if (!item.thumbnail || item.thumbnail.path) {
            return item;
        }
        addedCount += 1;
        const thumbnail = await utils_1.AttachmentDownloads.addJob(item.thumbnail, {
            messageId: message.id,
            type: 'quote',
            index,
            isOpenGroupV2,
            openGroupV2Details,
        });
        return { ...item, thumbnail };
    }));
    message.set({ quote });
    return addedCount;
}
async function queueAttachmentDownloads(message, conversation) {
    let count = 0;
    count += await processNormalAttachments(message, message.get('attachments') || [], conversation);
    count += await processPreviews(message, conversation);
    count += await processQuoteAttachments(message, conversation);
    if (count > 0) {
        await (0, data_1.saveMessage)(message.attributes);
    }
}
exports.queueAttachmentDownloads = queueAttachmentDownloads;
