"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleDataExtractionNotification = exports.innerHandleSwarmContentMessage = exports.isBlocked = exports.decryptWithBchatProtocol = exports.handleSwarmContentMessage = void 0;
const dataMessage_1 = require("./dataMessage");
const cache_1 = require("./cache");
const protobuf_1 = require("../protobuf");
const lodash_1 = __importStar(require("lodash")), Lodash = lodash_1;
const types_1 = require("../bchat/types");
const blockedNumberController_1 = require("../util/blockedNumberController");
const utils_1 = require("../bchat/utils");
const String_1 = require("../bchat/utils/String");
const crypto_1 = require("../bchat/crypto");
const conversations_1 = require("../bchat/conversations");
const keypairs_1 = require("./keypairs");
const configMessage_1 = require("./configMessage");
const BufferPadding_1 = require("../bchat/crypto/BufferPadding");
const Performance_1 = require("../bchat/utils/Performance");
const closedGroups_1 = require("./closedGroups");
const callMessage_1 = require("./callMessage");
const settings_key_1 = require("../data/settings-key");
const conversation_1 = require("../models/conversation");
const readReceipts_1 = require("../util/readReceipts");
const storage_1 = require("../util/storage");
const data_1 = require("../data/data");
const unsendingInteractions_1 = require("../interactions/conversations/unsendingInteractions");
const BnsVerification_1 = require("../components/conversation/BnsVerification");
async function handleSwarmContentMessage(envelope, messageHash) {
    try {
        const plaintext = await decrypt(envelope, envelope.content);
        if (!plaintext) {
            return;
        }
        else if (plaintext instanceof ArrayBuffer && plaintext.byteLength === 0) {
            return;
        }
        const sentAtTimestamp = lodash_1.default.toNumber(envelope.timestamp);
        await innerHandleSwarmContentMessage(envelope, sentAtTimestamp, plaintext, messageHash);
    }
    catch (e) {
        window?.log?.warn(e);
    }
}
exports.handleSwarmContentMessage = handleSwarmContentMessage;
async function decryptForClosedGroup(envelope, ciphertext) {
    window?.log?.info('received closed group message');
    try {
        const hexEncodedGroupPublicKey = envelope.source;
        if (!utils_1.GroupUtils.isMediumGroup(types_1.PubKey.cast(hexEncodedGroupPublicKey))) {
            window?.log?.warn('received medium group message but not for an existing medium group');
            throw new Error('Invalid group public key');
        }
        const encryptionKeyPairs = await (0, closedGroups_1.getAllCachedECKeyPair)(hexEncodedGroupPublicKey);
        const encryptionKeyPairsCount = encryptionKeyPairs?.length;
        if (!encryptionKeyPairs?.length) {
            throw new Error(`No group keypairs for group ${hexEncodedGroupPublicKey}`);
        }
        let decryptedContent;
        let keyIndex = 0;
        do {
            try {
                const hexEncryptionKeyPair = encryptionKeyPairs.pop();
                if (!hexEncryptionKeyPair) {
                    throw new Error('No more encryption keypairs to try for message.');
                }
                const encryptionKeyPair = keypairs_1.ECKeyPair.fromHexKeyPair(hexEncryptionKeyPair);
                decryptedContent = await decryptWithBchatProtocol(envelope, ciphertext, encryptionKeyPair, true);
                if (decryptedContent?.byteLength) {
                    break;
                }
                keyIndex++;
            }
            catch (e) {
                window?.log?.info(`Failed to decrypt closed group with key index ${keyIndex}. We have ${encryptionKeyPairs.length} keys to try left.`);
            }
        } while (encryptionKeyPairs.length > 0);
        if (!decryptedContent?.byteLength) {
            throw new Error(`Could not decrypt message for closed group with any of the ${encryptionKeyPairsCount} keypairs.`);
        }
        if (keyIndex !== 0) {
            window?.log?.warn('Decrypted a closed group message with not the latest encryptionkeypair we have');
        }
        window?.log?.info('ClosedGroup Message decrypted successfully with keyIndex:', keyIndex);
        return (0, BufferPadding_1.removeMessagePadding)(decryptedContent);
    }
    catch (e) {
        window?.log?.warn('decryptWithBchatProtocol for medium group message throw:', e.message);
        const groupPubKey = types_1.PubKey.cast(envelope.source);
        throw new Error(`Waiting for an encryption keypair to be received for group ${groupPubKey.key}`);
    }
}
async function decryptWithBchatProtocol(envelope, ciphertextObj, x25519KeyPair, isClosedGroup) {
    (0, Performance_1.perfStart)(`decryptWithBchatProtocol-${envelope.id}`);
    const recipientX25519PrivateKey = x25519KeyPair.privateKeyData;
    const hex = (0, String_1.toHex)(new Uint8Array(x25519KeyPair.publicKeyData));
    const recipientX25519PublicKey = types_1.PubKey.remove05PrefixIfNeeded(hex);
    const sodium = await (0, crypto_1.getSodiumRenderer)();
    const signatureSize = sodium.crypto_sign_BYTES;
    const ed25519PublicKeySize = sodium.crypto_sign_PUBLICKEYBYTES;
    const plaintextWithMetadata = sodium.crypto_box_seal_open(new Uint8Array(ciphertextObj), (0, String_1.fromHexToArray)(recipientX25519PublicKey), new Uint8Array(recipientX25519PrivateKey));
    if (plaintextWithMetadata.byteLength <= signatureSize + ed25519PublicKeySize) {
        (0, Performance_1.perfEnd)(`decryptWithBchatProtocol-${envelope.id}`, 'decryptWithBchatProtocol');
        throw new Error('Decryption failed.');
    }
    const signatureStart = plaintextWithMetadata.byteLength - signatureSize;
    const signature = plaintextWithMetadata.subarray(signatureStart);
    const pubkeyStart = plaintextWithMetadata.byteLength - (signatureSize + ed25519PublicKeySize);
    const pubkeyEnd = plaintextWithMetadata.byteLength - signatureSize;
    const senderED25519PublicKey = plaintextWithMetadata.subarray(pubkeyStart, pubkeyEnd);
    const plainTextEnd = plaintextWithMetadata.byteLength - (signatureSize + ed25519PublicKeySize);
    const plaintext = plaintextWithMetadata.subarray(0, plainTextEnd);
    const isValid = sodium.crypto_sign_verify_detached(signature, (0, crypto_1.concatUInt8Array)(plaintext, senderED25519PublicKey, (0, String_1.fromHexToArray)(recipientX25519PublicKey)), senderED25519PublicKey);
    if (!isValid) {
        (0, Performance_1.perfEnd)(`decryptWithBchatProtocol-${envelope.id}`, 'decryptWithBchatProtocol');
        throw new Error('Invalid message signature.');
    }
    const senderX25519PublicKey = sodium.crypto_sign_ed25519_pk_to_curve25519(senderED25519PublicKey);
    if (!senderX25519PublicKey) {
        (0, Performance_1.perfEnd)(`decryptWithBchatProtocol-${envelope.id}`, 'decryptWithBchatProtocol');
        throw new Error('Decryption failed.');
    }
    if (isClosedGroup) {
        envelope.senderIdentity = `bd${(0, String_1.toHex)(senderX25519PublicKey)}`;
    }
    else {
        envelope.source = `bd${(0, String_1.toHex)(senderX25519PublicKey)}`;
    }
    (0, Performance_1.perfEnd)(`decryptWithBchatProtocol-${envelope.id}`, 'decryptWithBchatProtocol');
    const addressLength = window.networkType == 'mainnet' ? 97 : 95;
    const beldexFinalAddress = new TextDecoder().decode(plaintext.subarray(0, addressLength));
    const conversation = await (0, conversations_1.getConversationController)().getOrCreateAndWait(envelope.source, conversation_1.ConversationTypeEnum.PRIVATE);
    await conversation.setwalletAddress(beldexFinalAddress);
    const message = plaintextWithMetadata.subarray(addressLength, plainTextEnd);
    return message;
}
exports.decryptWithBchatProtocol = decryptWithBchatProtocol;
async function isBlocked(number) {
    return blockedNumberController_1.BlockedNumberController.isBlockedAsync(number);
}
exports.isBlocked = isBlocked;
async function decryptUnidentifiedSender(envelope, ciphertext) {
    try {
        const userX25519KeyPair = await utils_1.UserUtils.getIdentityKeyPair();
        if (!userX25519KeyPair) {
            throw new Error('Failed to find User x25519 keypair from stage');
        }
        const ecKeyPair = keypairs_1.ECKeyPair.fromArrayBuffer(userX25519KeyPair.pubKey, userX25519KeyPair.privKey);
        (0, Performance_1.perfStart)(`decryptUnidentifiedSender-${envelope.id}`);
        const retBchatProtocol = await decryptWithBchatProtocol(envelope, ciphertext, ecKeyPair);
        const ret = (0, BufferPadding_1.removeMessagePadding)(retBchatProtocol);
        (0, Performance_1.perfEnd)(`decryptUnidentifiedSender-${envelope.id}`, 'decryptUnidentifiedSender');
        return ret;
    }
    catch (e) {
        window?.log?.warn('decryptWithBchatProtocol for unidentified message throw:', e);
        return null;
    }
}
async function doDecrypt(envelope, ciphertext) {
    if (ciphertext.byteLength === 0) {
        throw new Error('Received an empty envelope.');
    }
    switch (envelope.type) {
        case protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE:
            return decryptForClosedGroup(envelope, ciphertext);
        case protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE: {
            return decryptUnidentifiedSender(envelope, ciphertext);
        }
        default:
            throw new Error(`Unknown message type:${envelope.type}`);
    }
}
async function decrypt(envelope, ciphertext) {
    try {
        const plaintext = await doDecrypt(envelope, ciphertext);
        if (!plaintext) {
            await (0, cache_1.removeFromCache)(envelope);
            return null;
        }
        (0, Performance_1.perfStart)(`updateCache-${envelope.id}`);
        await (0, cache_1.updateCache)(envelope, plaintext).catch((error) => {
            window?.log?.error('decrypt failed to save decrypted message contents to cache:', error && error.stack ? error.stack : error);
        });
        (0, Performance_1.perfEnd)(`updateCache-${envelope.id}`, 'updateCache');
        return plaintext;
    }
    catch (error) {
        throw error;
    }
}
function shouldDropBlockedUserMessage(content) {
    if (!content?.dataMessage?.group?.id) {
        return true;
    }
    const groupId = (0, String_1.toHex)(content.dataMessage.group.id);
    const groupConvo = (0, conversations_1.getConversationController)().get(groupId);
    if (!groupConvo) {
        return true;
    }
    if (groupConvo.isBlocked()) {
        return true;
    }
    let msgWithoutDataMessage = Lodash.pickBy(content, (_value, key) => key !== 'dataMessage' && key !== 'toJSON');
    msgWithoutDataMessage = Lodash.pickBy(msgWithoutDataMessage, Lodash.identity);
    const isMessageDataMessageOnly = Lodash.isEmpty(msgWithoutDataMessage);
    if (!isMessageDataMessageOnly) {
        return true;
    }
    const data = content.dataMessage;
    const isControlDataMessageOnly = !data.body &&
        !data.preview?.length &&
        !data.attachments?.length &&
        !data.openGroupInvitation &&
        !data.quote;
    return !isControlDataMessageOnly;
}
async function innerHandleSwarmContentMessage(envelope, sentAtTimestamp, plaintext, messageHash) {
    try {
        (0, Performance_1.perfStart)(`SignalService.Content.decode-${envelope.id}`);
        window.log.info('innerHandleSwarmContentMessage');
        const content = protobuf_1.SignalService.Content.decode(new Uint8Array(plaintext));
        (0, Performance_1.perfEnd)(`SignalService.Content.decode-${envelope.id}`, 'SignalService.Content.decode');
        (0, Performance_1.perfStart)(`isBlocked-${envelope.id}`);
        const blocked = await isBlocked(envelope.source);
        (0, Performance_1.perfEnd)(`isBlocked-${envelope.id}`, 'isBlocked');
        if (blocked) {
            if (shouldDropBlockedUserMessage(content)) {
                window?.log?.info('Dropping blocked user message');
                return;
            }
            else {
                window?.log?.info('Allowing group-control message only from blocked user');
            }
        }
        const isPrivateConversationMessage = !envelope.senderIdentity;
        const senderConversationModel = await (0, conversations_1.getConversationController)().getOrCreateAndWait(isPrivateConversationMessage ? envelope.source : envelope.senderIdentity, conversation_1.ConversationTypeEnum.PRIVATE);
        (0, BnsVerification_1.bnsVerificationConvo)(senderConversationModel, isPrivateConversationMessage, envelope);
        if (!isPrivateConversationMessage) {
            await (0, conversations_1.getConversationController)().getOrCreateAndWait(envelope.source, conversation_1.ConversationTypeEnum.GROUP);
        }
        if (content.dataMessage) {
            if (content.dataMessage.profileKey && content.dataMessage.profileKey.length === 0) {
                content.dataMessage.profileKey = null;
            }
            (0, Performance_1.perfStart)(`handleSwarmDataMessage-${envelope.id}`);
            await (0, dataMessage_1.handleSwarmDataMessage)(envelope, sentAtTimestamp, content.dataMessage, messageHash, senderConversationModel);
            (0, Performance_1.perfEnd)(`handleSwarmDataMessage-${envelope.id}`, 'handleSwarmDataMessage');
            return;
        }
        if (content.receiptMessage) {
            (0, Performance_1.perfStart)(`handleReceiptMessage-${envelope.id}`);
            await handleReceiptMessage(envelope, content.receiptMessage);
            (0, Performance_1.perfEnd)(`handleReceiptMessage-${envelope.id}`, 'handleReceiptMessage');
            return;
        }
        if (content.typingMessage) {
            (0, Performance_1.perfStart)(`handleTypingMessage-${envelope.id}`);
            await handleTypingMessage(envelope, content.typingMessage);
            (0, Performance_1.perfEnd)(`handleTypingMessage-${envelope.id}`, 'handleTypingMessage');
            return;
        }
        if (content.configurationMessage) {
            void (0, configMessage_1.handleConfigurationMessage)(envelope, content.configurationMessage);
            return;
        }
        if (content.dataExtractionNotification) {
            (0, Performance_1.perfStart)(`handleDataExtractionNotification-${envelope.id}`);
            await handleDataExtractionNotification(envelope, content.dataExtractionNotification);
            (0, Performance_1.perfEnd)(`handleDataExtractionNotification-${envelope.id}`, 'handleDataExtractionNotification');
            return;
        }
        if (content.unsendMessage) {
            await handleUnsendMessage(envelope, content.unsendMessage);
        }
        if (content.callMessage) {
            await (0, callMessage_1.handleCallMessage)(envelope, content.callMessage);
        }
        if (content.messageRequestResponse) {
            await handleMessageRequestResponse(envelope, content.messageRequestResponse);
        }
    }
    catch (e) {
        window?.log?.warn(e);
    }
}
exports.innerHandleSwarmContentMessage = innerHandleSwarmContentMessage;
function onReadReceipt(readAt, timestamp, source) {
    window?.log?.info('read receipt', source, timestamp);
    if (!storage_1.Storage.get(settings_key_1.SettingsKey.settingsReadReceipt)) {
        return;
    }
    return readReceipts_1.ReadReceipts.onReadReceipt({
        source,
        timestamp,
        readAt,
    });
}
async function handleReceiptMessage(envelope, receiptMessage) {
    const receipt = receiptMessage;
    const { type, timestamp } = receipt;
    const results = [];
    if (type === protobuf_1.SignalService.ReceiptMessage.Type.READ) {
        for (const ts of timestamp) {
            const promise = onReadReceipt(Lodash.toNumber(envelope.timestamp), Lodash.toNumber(ts), envelope.source);
            results.push(promise);
        }
    }
    await Promise.all(results);
    await (0, cache_1.removeFromCache)(envelope);
}
async function handleTypingMessage(envelope, typingMessage) {
    const { timestamp, action } = typingMessage;
    const { source } = envelope;
    await (0, cache_1.removeFromCache)(envelope);
    if (!storage_1.Storage.get(settings_key_1.SettingsKey.settingsTypingIndicator)) {
        return;
    }
    if (envelope.timestamp && timestamp) {
        const envelopeTimestamp = Lodash.toNumber(envelope.timestamp);
        const typingTimestamp = Lodash.toNumber(timestamp);
        if (typingTimestamp !== envelopeTimestamp) {
            window?.log?.warn(`Typing message envelope timestamp (${envelopeTimestamp}) did not match typing timestamp (${typingTimestamp})`);
            return;
        }
    }
    const conversation = (0, conversations_1.getConversationController)().get(source);
    const started = action === protobuf_1.SignalService.TypingMessage.Action.STARTED;
    if (conversation) {
        await conversation.notifyTypingNoCommit({
            isTyping: started,
            sender: source,
        });
    }
}
async function handleUnsendMessage(envelope, unsendMessage) {
    const { author: messageAuthor, timestamp } = unsendMessage;
    window.log.info(`handleUnsendMessage from ${messageAuthor}: of timestamp: ${timestamp}`);
    if (messageAuthor !== (envelope.senderIdentity || envelope.source)) {
        window?.log?.error('handleUnsendMessage: Dropping request as the author and the sender differs.');
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    if (!unsendMessage) {
        window?.log?.error('handleUnsendMessage: Invalid parameters -- dropping message.');
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    if (!timestamp) {
        window?.log?.error('handleUnsendMessage: Invalid timestamp -- dropping message');
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    const messageToDelete = await (0, data_1.getMessageBySenderAndTimestamp)({
        source: messageAuthor,
        timestamp: Lodash.toNumber(timestamp),
    });
    const messageHash = messageToDelete?.get('messageHash');
    if (messageHash && messageToDelete) {
        window.log.info('handleUnsendMessage: got a request to delete ', messageHash);
        const conversation = (0, conversations_1.getConversationController)().get(messageToDelete.get('conversationId'));
        if (!conversation) {
            await (0, cache_1.removeFromCache)(envelope);
            return;
        }
        if (messageToDelete.getSource() === utils_1.UserUtils.getOurPubKeyStrFromCache()) {
            void (0, unsendingInteractions_1.deleteMessagesFromSwarmAndCompletelyLocally)(conversation, [messageToDelete]);
        }
        else {
            void (0, unsendingInteractions_1.deleteMessagesFromSwarmAndMarkAsDeletedLocally)(conversation, [messageToDelete]);
        }
    }
    else {
        window.log.info('handleUnsendMessage: got a request to delete an unknown messageHash:', messageHash, ' and found messageToDelete:', messageToDelete?.id);
    }
    await (0, cache_1.removeFromCache)(envelope);
}
async function handleMessageRequestResponse(envelope, messageRequestResponse) {
    const { isApproved } = messageRequestResponse;
    if (!messageRequestResponse) {
        window?.log?.error('handleMessageRequestResponse: Invalid parameters -- dropping message.');
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    const convoId = envelope.source;
    const conversationToApprove = (0, conversations_1.getConversationController)().get(convoId);
    if (!conversationToApprove || conversationToApprove.didApproveMe() === isApproved) {
        window?.log?.info('Conversation already contains the correct value for the didApproveMe field.');
        return;
    }
    await conversationToApprove.setDidApproveMe(isApproved);
    if (isApproved === true) {
        await conversationToApprove.addIncomingApprovalMessage(lodash_1.default.toNumber(envelope.timestamp), envelope.source);
    }
    await (0, cache_1.removeFromCache)(envelope);
}
async function handleDataExtractionNotification(envelope, dataNotificationMessage) {
    const { type, timestamp: referencedAttachment } = dataNotificationMessage;
    const { source, timestamp } = envelope;
    await (0, cache_1.removeFromCache)(envelope);
    const convo = (0, conversations_1.getConversationController)().get(source);
    if (!convo || !convo.isPrivate()) {
        window?.log?.info('Got DataNotification for unknown or non private convo');
        return;
    }
    if (!type || !source) {
        window?.log?.info('DataNotification pre check failed');
        return;
    }
    if (timestamp) {
        const envelopeTimestamp = Lodash.toNumber(timestamp);
        const referencedAttachmentTimestamp = Lodash.toNumber(referencedAttachment);
        await convo.addSingleIncomingMessage({
            source,
            sent_at: envelopeTimestamp,
            dataExtractionNotification: {
                type,
                referencedAttachmentTimestamp,
                source,
            },
            unread: 1,
            expireTimer: 0,
        });
        convo.updateLastMessage();
    }
}
exports.handleDataExtractionNotification = handleDataExtractionNotification;
