"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleOpenGroupV2Message = void 0;
const lodash_1 = require("lodash");
const messageFactory_1 = require("../models/messageFactory");
const protobuf_1 = require("../protobuf");
const OpenGroupUtils_1 = require("../bchat/apis/open_group_api/utils/OpenGroupUtils");
const conversations_1 = require("../bchat/conversations");
const BufferPadding_1 = require("../bchat/crypto/BufferPadding");
const utils_1 = require("../bchat/utils");
const Performance_1 = require("../bchat/utils/Performance");
const String_1 = require("../bchat/utils/String");
const queuedJob_1 = require("./queuedJob");
async function handleOpenGroupV2Message(message, roomInfos) {
    const { base64EncodedData, sentTimestamp, sender, serverId } = message;
    const { serverUrl, roomId } = roomInfos;
    if (!base64EncodedData || !sentTimestamp || !sender || !serverId) {
        window?.log?.warn('Invalid data passed to handleOpenGroupV2Message.', message);
        return;
    }
    (0, Performance_1.perfStart)(`fromBase64ToArray-${base64EncodedData.length}`);
    const arr = (0, String_1.fromBase64ToArray)(base64EncodedData);
    (0, Performance_1.perfEnd)(`fromBase64ToArray-${base64EncodedData.length}`, 'fromBase64ToArray');
    const dataUint = new Uint8Array((0, BufferPadding_1.removeMessagePadding)(arr));
    const decoded = protobuf_1.SignalService.Content.decode(dataUint);
    const conversationId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(serverUrl, roomId);
    if (!conversationId) {
        window?.log?.error('We cannot handle a message without a conversationId');
        return;
    }
    const idataMessage = decoded?.dataMessage;
    if (!idataMessage) {
        window?.log?.error('Invalid decoded opengroup message: no dataMessage');
        return;
    }
    if (!(0, conversations_1.getConversationController)()
        .get(conversationId)
        ?.isOpenGroupV2()) {
        window?.log?.error('Received a message for an unknown convo or not an v2. Skipping');
        return;
    }
    const groupConvo = (0, conversations_1.getConversationController)().get(conversationId);
    if (!groupConvo) {
        window?.log?.warn('Skipping handleJob for unknown convo: ', conversationId);
        return;
    }
    void groupConvo.queueJob(async () => {
        const isMe = utils_1.UserUtils.isUsFromCache(sender);
        const commonAttributes = { serverTimestamp: sentTimestamp, serverId, conversationId };
        const attributesForNotUs = { ...commonAttributes, sender };
        const msgModel = isMe
            ? (0, messageFactory_1.createPublicMessageSentFromUs)(commonAttributes)
            : (0, messageFactory_1.createPublicMessageSentFromNotUs)(attributesForNotUs);
        await (0, queuedJob_1.handleMessageJob)(msgModel, groupConvo, (0, queuedJob_1.toRegularMessage)(decoded?.dataMessage), lodash_1.noop, sender, '');
    });
}
exports.handleOpenGroupV2Message = handleOpenGroupV2Message;
