"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateOurProfileSync = exports.appendFetchAvatarAndProfileJob = void 0;
const queue_promise_1 = __importDefault(require("queue-promise"));
const bytebuffer_1 = __importDefault(require("bytebuffer"));
const lodash_1 = __importDefault(require("lodash"));
const attachments_1 = require("./attachments");
const Promise_1 = require("../bchat/utils/Promise");
const String_1 = require("../bchat/utils/String");
const MessageAttachment_1 = require("../types/MessageAttachment");
const types_1 = require("../types");
const attachmentsUtil_1 = require("../util/attachmentsUtil");
const profileEncrypter_1 = require("../util/crypto/profileEncrypter");
const conversation_1 = require("../models/conversation");
const conversations_1 = require("../bchat/conversations");
const utils_1 = require("../bchat/utils");
const queue = new queue_promise_1.default({
    concurrent: 1,
    interval: 500,
});
queue.on('reject', error => {
    window.log.warn('[profile-update] task profile image update failed with', error);
});
async function appendFetchAvatarAndProfileJob(conversation, profile, profileKey) {
    if (!conversation?.id) {
        window?.log?.warn('[profile-update] Cannot update profile with empty convoid');
        return;
    }
    const oneAtaTimeStr = `appendFetchAvatarAndProfileJob:${conversation.id}`;
    if ((0, Promise_1.hasAlreadyOneAtaTimeMatching)(oneAtaTimeStr)) {
        return;
    }
    window.log.info(`[profile-update] queuing fetching avatar for ${conversation.id}`);
    const task = (0, Promise_1.allowOnlyOneAtATime)(oneAtaTimeStr, async () => {
        return createOrUpdateProfile(conversation, profile, profileKey);
    });
    queue.enqueue(async () => task);
}
exports.appendFetchAvatarAndProfileJob = appendFetchAvatarAndProfileJob;
async function updateOurProfileSync(profile, profileKey) {
    const ourConvo = (0, conversations_1.getConversationController)().get(utils_1.UserUtils.getOurPubKeyStrFromCache());
    if (!ourConvo?.id) {
        window?.log?.warn('[profile-update] Cannot update our profile with empty convoid');
        return;
    }
    const oneAtaTimeStr = `appendFetchAvatarAndProfileJob:${ourConvo.id}`;
    return (0, Promise_1.allowOnlyOneAtATime)(oneAtaTimeStr, async () => {
        return createOrUpdateProfile(ourConvo, profile, profileKey);
    });
}
exports.updateOurProfileSync = updateOurProfileSync;
async function createOrUpdateProfile(conversation, profile, profileKey) {
    const newProfile = conversation.get('profile') || {};
    let changes = false;
    if (newProfile.displayName !== profile.displayName) {
        changes = true;
    }
    newProfile.displayName = profile.displayName;
    if (profile.profilePicture && profileKey) {
        const prevPointer = conversation.get('avatarPointer');
        const needsUpdate = !prevPointer || !lodash_1.default.isEqual(prevPointer, profile.profilePicture);
        if (needsUpdate) {
            try {
                window.log.debug(`[profile-update] starting downloading task for  ${conversation.id}`);
                const downloaded = await (0, attachments_1.downloadAttachment)({
                    url: profile.profilePicture,
                    isRaw: true,
                });
                let path = null;
                if (profileKey) {
                    const encoding = typeof profileKey === 'string' ? 'base64' : undefined;
                    try {
                        const profileKeyArrayBuffer = bytebuffer_1.default.wrap(profileKey, encoding).toArrayBuffer();
                        const decryptedData = await (0, profileEncrypter_1.decryptProfile)(downloaded.data, profileKeyArrayBuffer);
                        window.log.info(`[profile-update] about to auto scale avatar for convo ${conversation.id}`);
                        const scaledData = await (0, attachmentsUtil_1.autoScaleForIncomingAvatar)(decryptedData);
                        const upgraded = await (0, MessageAttachment_1.processNewAttachment)({
                            data: await scaledData.blob.arrayBuffer(),
                            contentType: types_1.MIME.IMAGE_UNKNOWN,
                        });
                        conversation.set('avatarPointer', profile.profilePicture);
                        conversation.set('profileKey', (0, String_1.toHex)(profileKey));
                        ({ path } = upgraded);
                    }
                    catch (e) {
                        window?.log?.error(`[profile-update] Could not decrypt profile image: ${e}`);
                    }
                }
                newProfile.avatar = path;
                changes = true;
            }
            catch (e) {
                window.log.warn(`[profile-update] Failed to download attachment at ${profile.profilePicture}. Maybe it expired? ${e.message}`);
            }
        }
    }
    else if (profileKey) {
        if (newProfile.avatar !== null) {
            changes = true;
        }
        newProfile.avatar = null;
    }
    const conv = await (0, conversations_1.getConversationController)().getOrCreateAndWait(conversation.id, conversation_1.ConversationTypeEnum.PRIVATE);
    await conv.setBchatProfile(newProfile);
    if (changes) {
        await conv.commit();
    }
}
