"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = __importStar(require("chai"));
const crypto = __importStar(require("crypto"));
const sinon_1 = __importStar(require("sinon")), sinon = sinon_1;
const crypto_1 = require("../../../../bchat/crypto");
const test_utils_1 = require("../../../test-utils");
const protobuf_1 = require("../../../../protobuf");
const utils_1 = require("../../../../bchat/utils");
const chai_bytes_1 = __importDefault(require("chai-bytes"));
const types_1 = require("../../../../bchat/types");
const String_1 = require("../../../../bchat/utils/String");
const BufferPadding_1 = require("../../../../bchat/crypto/BufferPadding");
chai_1.default.use(chai_bytes_1.default);
describe('MessageEncrypter', () => {
    const ourNumber = '0123456789abcdef';
    const ourUserEd25516Keypair = {
        pubKey: '37e1631b002de498caf7c5c1712718bde7f257c6dadeed0c21abf5e939e6c309',
        privKey: 'be1d11154ff9b6de77873f0b6b0bcc460000000000000000000000000000000037e1631b002de498caf7c5c1712718bde7f257c6dadeed0c21abf5e939e6c309',
    };
    const ourIdentityKeypair = {
        pubKey: new Uint8Array([
            5,
            44,
            2,
            168,
            162,
            203,
            50,
            66,
            136,
            81,
            30,
            221,
            57,
            245,
            1,
            148,
            162,
            194,
            255,
            47,
            134,
            104,
            180,
            207,
            188,
            18,
            71,
            62,
            58,
            107,
            23,
            92,
            97,
        ]),
        privKey: new Uint8Array([
            200,
            45,
            226,
            75,
            253,
            235,
            213,
            108,
            187,
            188,
            217,
            9,
            51,
            105,
            65,
            15,
            97,
            36,
            233,
            33,
            21,
            31,
            7,
            90,
            145,
            30,
            52,
            254,
            47,
            162,
            192,
            105,
        ]),
    };
    beforeEach(() => {
        sinon_1.default.stub(utils_1.UserUtils, 'getOurPubKeyStrFromCache').returns(ourNumber);
        sinon_1.default.stub(utils_1.UserUtils, 'getUserED25519KeyPair').resolves(ourUserEd25516Keypair);
    });
    afterEach(() => {
        sinon_1.default.restore();
    });
    describe('EncryptionType', () => {
        describe('ClosedGroup', () => {
            it('should return a CLOSED_GROUP_MESSAGE envelope type for ClosedGroup', async () => {
                const hexKeyPair = {
                    publicHex: `bd${ourUserEd25516Keypair.pubKey}`,
                    privateHex: '0123456789abcdef',
                };
                test_utils_1.TestUtils.stubData('getLatestClosedGroupEncryptionKeyPair').resolves(hexKeyPair);
                const data = crypto.randomBytes(10);
                const result = await crypto_1.MessageEncrypter.encrypt(test_utils_1.TestUtils.generateFakePubKey(), data, protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE);
                chai_1.default
                    .expect(result.envelopeType)
                    .to.deep.equal(protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE);
            });
            it('should return a BCHAT_MESSAGE envelope type for Fallback', async () => {
                const data = crypto.randomBytes(10);
                const result = await crypto_1.MessageEncrypter.encrypt(test_utils_1.TestUtils.generateFakePubKey(), data, protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
                chai_1.default.expect(result.envelopeType).to.deep.equal(protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
            });
            it('should throw an error for anything else than Fallback or ClosedGroup', () => {
                const data = crypto.randomBytes(10);
                return crypto_1.MessageEncrypter.encrypt(test_utils_1.TestUtils.generateFakePubKey(), data, 3).should.eventually.be.rejectedWith(Error);
            });
        });
    });
    describe('BChat Protocol', () => {
        beforeEach(() => {
            sinon_1.default.stub(utils_1.UserUtils, 'getIdentityKeyPair').resolves(ourIdentityKeypair);
        });
        afterEach(() => {
            sinon_1.default.restore();
        });
        it('should pass the padded message body to encrypt', async () => {
            const data = crypto.randomBytes(10);
            const spy = sinon.spy(crypto_1.MessageEncrypter, 'encryptUsingBchatProtocol');
            await crypto_1.MessageEncrypter.encrypt(test_utils_1.TestUtils.generateFakePubKey(), data, protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
            chai_1.default.expect(spy.callCount).to.be.equal(1);
            const paddedData = (0, BufferPadding_1.addMessagePadding)(data);
            const firstArgument = new Uint8Array(spy.args[0][1]);
            chai_1.default.expect(firstArgument).to.deep.equal(paddedData);
            spy.restore();
        });
        it('should pass the correct data for sodium crypto_sign', async () => {
            const keypair = await utils_1.UserUtils.getUserED25519KeyPair();
            const recipient = test_utils_1.TestUtils.generateFakePubKey();
            const sodium = await (0, crypto_1.getSodiumRenderer)();
            const cryptoSignDetachedSpy = sinon_1.default.spy(sodium, 'crypto_sign_detached');
            const plainText = '123456';
            const plainTextBytes = new Uint8Array(utils_1.StringUtils.encode(plainText, 'utf8'));
            const userED25519PubKeyBytes = new Uint8Array(utils_1.StringUtils.fromHex(keypair.pubKey));
            const recipientX25519PublicKeyWithoutPrefix = types_1.PubKey.remove05PrefixIfNeeded(recipient.key);
            const recipientX25519PublicKey = new Uint8Array(utils_1.StringUtils.fromHex(recipientX25519PublicKeyWithoutPrefix));
            await crypto_1.MessageEncrypter.encryptUsingBchatProtocol(recipient, plainTextBytes);
            const [dataForSign, userED25519SecretKeyBytes] = cryptoSignDetachedSpy.args[0];
            const userEdPrivkeyBytes = new Uint8Array(utils_1.StringUtils.fromHex(keypair.privKey));
            (0, chai_1.expect)(userED25519SecretKeyBytes).to.equalBytes(userEdPrivkeyBytes);
            (0, chai_1.expect)(dataForSign.subarray(0, plainTextBytes.length)).to.equalBytes(plainTextBytes);
            (0, chai_1.expect)(dataForSign.subarray(plainTextBytes.length, plainTextBytes.length + userED25519PubKeyBytes.length)).to.equalBytes(userED25519PubKeyBytes);
            (0, chai_1.expect)(dataForSign.subarray(plainTextBytes.length + userED25519PubKeyBytes.length)).to.equalBytes(recipientX25519PublicKey);
        });
        it('should return valid decodable ciphertext', async () => {
            const userX25519KeyPair = await utils_1.UserUtils.getIdentityKeyPair();
            const userEd25519KeyPair = await utils_1.UserUtils.getUserED25519KeyPair();
            const plainTextBytes = new Uint8Array(utils_1.StringUtils.encode('123456789', 'utf8'));
            const sodium = await (0, crypto_1.getSodiumRenderer)();
            const recipientX25519PrivateKey = userX25519KeyPair.privKey;
            const recipientX25519PublicKeyHex = (0, String_1.toHex)(userX25519KeyPair.pubKey);
            const recipientX25519PublicKeyWithoutPrefix = types_1.PubKey.remove05PrefixIfNeeded(recipientX25519PublicKeyHex);
            const recipientX25519PublicKey = new types_1.PubKey(recipientX25519PublicKeyWithoutPrefix);
            const ciphertext = await crypto_1.MessageEncrypter.encryptUsingBchatProtocol(recipientX25519PublicKey, plainTextBytes);
            const plaintextWithMetadata = sodium.crypto_box_seal_open(ciphertext, new Uint8Array((0, String_1.fromHex)(recipientX25519PublicKey.key)), new Uint8Array(recipientX25519PrivateKey));
            const signatureSize = sodium.crypto_sign_BYTES;
            const ed25519PublicKeySize = sodium.crypto_sign_PUBLICKEYBYTES;
            const signatureStart = plaintextWithMetadata.byteLength - signatureSize;
            const signature = plaintextWithMetadata.subarray(signatureStart);
            const pubkeyStart = plaintextWithMetadata.byteLength - (signatureSize + ed25519PublicKeySize);
            const pubkeyEnd = plaintextWithMetadata.byteLength - signatureSize;
            const senderED25519PublicKey = plaintextWithMetadata.subarray(pubkeyStart, pubkeyEnd);
            const plainTextEnd = plaintextWithMetadata.byteLength - (signatureSize + ed25519PublicKeySize);
            const plaintextDecoded = plaintextWithMetadata.subarray(0, plainTextEnd);
            (0, chai_1.expect)(plaintextDecoded).to.equalBytes(plainTextBytes);
            (0, chai_1.expect)(senderED25519PublicKey).to.equalBytes(userEd25519KeyPair.pubKey);
            const dataForVerify = (0, crypto_1.concatUInt8Array)(plaintextDecoded, senderED25519PublicKey, new Uint8Array((0, String_1.fromHex)(recipientX25519PublicKey.key)));
            const isValid = sodium.crypto_sign_verify_detached(signature, dataForVerify, senderED25519PublicKey);
            (0, chai_1.expect)(isValid).to.be.equal(true, 'the signature cannot be verified');
        });
    });
});
