"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const protobuf_1 = require("../../../../protobuf");
const test_utils_1 = require("../../../test-utils");
const utils_1 = require("../../../../bchat/utils");
const types_1 = require("../../../../bchat/types");
const bchat_1 = require("../../../../bchat");
const ClosedGroupVisibleMessage_1 = require("../../../../bchat/messages/outgoing/visibleMessage/ClosedGroupVisibleMessage");
const VisibleMessage_1 = require("../../../../bchat/messages/outgoing/visibleMessage/VisibleMessage");
describe('ClosedGroupVisibleMessage', () => {
    let groupId;
    beforeEach(() => {
        groupId = test_utils_1.TestUtils.generateFakePubKey();
    });
    it('can create empty message with timestamp, groupId and chatMessage', () => {
        const chatMessage = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            body: 'body',
        });
        const message = new ClosedGroupVisibleMessage_1.ClosedGroupVisibleMessage({
            groupId,
            chatMessage,
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        (0, chai_1.expect)(decoded.dataMessage)
            .to.have.property('group')
            .to.have.deep.property('id', new Uint8Array(utils_1.StringUtils.encode(types_1.PubKey.PREFIX_GROUP_TEXTSECURE + groupId.key, 'utf8')));
        (0, chai_1.expect)(decoded.dataMessage)
            .to.have.property('group')
            .to.have.deep.property('type', protobuf_1.SignalService.GroupContext.Type.DELIVER);
        (0, chai_1.expect)(decoded.dataMessage).to.have.deep.property('body', 'body');
        (0, chai_1.expect)(message)
            .to.have.property('timestamp')
            .to.be.equal(chatMessage.timestamp);
    });
    it('correct ttl', () => {
        const chatMessage = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
        });
        const message = new ClosedGroupVisibleMessage_1.ClosedGroupVisibleMessage({
            groupId,
            chatMessage,
        });
        (0, chai_1.expect)(message.ttl()).to.equal(bchat_1.Constants.TTL_DEFAULT.TTL_MAX);
    });
    it('has an identifier', () => {
        const chatMessage = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
        });
        const message = new ClosedGroupVisibleMessage_1.ClosedGroupVisibleMessage({
            groupId,
            chatMessage,
        });
        (0, chai_1.expect)(message.identifier).to.not.equal(null, 'identifier cannot be null');
        (0, chai_1.expect)(message.identifier).to.not.equal(undefined, 'identifier cannot be undefined');
    });
    it('should use the identifier passed into it over the one set in chatMessage', () => {
        const chatMessage = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            body: 'body',
            identifier: 'chatMessage',
        });
        const message = new ClosedGroupVisibleMessage_1.ClosedGroupVisibleMessage({
            groupId,
            chatMessage,
            identifier: 'closedGroupMessage',
        });
        (0, chai_1.expect)(message.identifier).to.be.equal('closedGroupMessage');
    });
    it('should use the identifier of the chatMessage if one is not specified on the closed group message', () => {
        const chatMessage = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            body: 'body',
            identifier: 'chatMessage',
        });
        const message = new ClosedGroupVisibleMessage_1.ClosedGroupVisibleMessage({
            groupId,
            chatMessage,
        });
        (0, chai_1.expect)(message.identifier).to.be.equal('chatMessage');
    });
});
