"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = __importDefault(require("chai"));
const uuid_1 = require("uuid");
const JobQueue_1 = require("../../../../bchat/utils/JobQueue");
const test_utils_1 = require("../../../test-utils");
const chai_as_promised_1 = __importDefault(require("chai-as-promised"));
chai_1.default.use(chai_as_promised_1.default);
const { assert } = chai_1.default;
describe('JobQueue', () => {
    describe('has', () => {
        it('should return the correct value', async () => {
            const queue = new JobQueue_1.JobQueue();
            const id = 'jobId';
            assert.isFalse(queue.has(id));
            const promise = queue.addWithId(id, async () => test_utils_1.TestUtils.timeout(30));
            assert.isTrue(queue.has(id));
            await promise;
            assert.isFalse(queue.has(id));
        });
    });
    describe('addWithId', () => {
        it('should run the jobs concurrently', async () => {
            const input = [
                [10, 10],
                [20, 8],
                [30, 2],
            ];
            const queue = new JobQueue_1.JobQueue();
            const mapper = async ([value, ms]) => queue.addWithId((0, uuid_1.v4)(), async () => {
                await test_utils_1.TestUtils.timeout(ms);
                return value;
            });
            const start = Date.now();
            await assert.eventually.deepEqual(Promise.all(input.map(mapper)), [10, 20, 30]);
            const timeTaken = Date.now() - start;
            assert.isAtLeast(timeTaken, 20, 'Queue should take atleast 100ms to run.');
        });
        it('should return the result of the job', async () => {
            const queue = new JobQueue_1.JobQueue();
            const success = queue.addWithId((0, uuid_1.v4)(), async () => {
                await test_utils_1.TestUtils.timeout(10);
                return 'success';
            });
            const failure = queue.addWithId((0, uuid_1.v4)(), async () => {
                await test_utils_1.TestUtils.timeout(10);
                throw new Error('failed');
            });
            await assert.eventually.equal(success, 'success');
            await assert.isRejected(failure, /failed/);
        });
        it('should handle sync and async tasks', async () => {
            const queue = new JobQueue_1.JobQueue();
            const first = queue.addWithId((0, uuid_1.v4)(), () => 'first');
            const second = queue.addWithId((0, uuid_1.v4)(), async () => {
                await test_utils_1.TestUtils.timeout(10);
                return 'second';
            });
            const third = queue.addWithId((0, uuid_1.v4)(), () => 'third');
            await assert.eventually.deepEqual(Promise.all([first, second, third]), [
                'first',
                'second',
                'third',
            ]);
        });
        it('should return the previous job if same id was passed', async () => {
            const queue = new JobQueue_1.JobQueue();
            const id = (0, uuid_1.v4)();
            const job = async () => {
                await test_utils_1.TestUtils.timeout(10);
                return 'job1';
            };
            const promise = queue.addWithId(id, job);
            const otherPromise = queue.addWithId(id, () => 'job2');
            await assert.eventually.equal(promise, 'job1');
            await assert.eventually.equal(otherPromise, 'job1');
        });
        it('should remove completed jobs', async () => {
            const queue = new JobQueue_1.JobQueue();
            const id = (0, uuid_1.v4)();
            const successfullJob = queue.addWithId(id, async () => test_utils_1.TestUtils.timeout(10));
            assert.isTrue(queue.has(id));
            await successfullJob;
            assert.isFalse(queue.has(id));
            const failJob = queue.addWithId(id, async () => {
                await test_utils_1.TestUtils.timeout(10);
                throw new Error('failed');
            });
            assert.isTrue(queue.has(id));
            await assert.isRejected(failJob, /failed/);
            assert.isFalse(queue.has(id));
        });
    });
});
