"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = __importDefault(require("chai"));
const test_utils_1 = require("../../../test-utils");
const utils_1 = require("../../../../bchat/utils");
const types_1 = require("../../../../bchat/types");
const ClosedGroupVisibleMessage_1 = require("../../../../bchat/messages/outgoing/visibleMessage/ClosedGroupVisibleMessage");
const ConfigurationMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/ConfigurationMessage");
const chai_as_promised_1 = __importDefault(require("chai-as-promised"));
chai_1.default.use(chai_as_promised_1.default);
const ClosedGroupEncryptionPairReplyMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/group/ClosedGroupEncryptionPairReplyMessage");
const protobuf_1 = require("../../../../protobuf");
const ClosedGroupAddedMembersMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/group/ClosedGroupAddedMembersMessage");
const ClosedGroupEncryptionPairMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/group/ClosedGroupEncryptionPairMessage");
const ClosedGroupNameChangeMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/group/ClosedGroupNameChangeMessage");
const ClosedGroupNewMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/group/ClosedGroupNewMessage");
const ClosedGroupRemovedMembersMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/group/ClosedGroupRemovedMembersMessage");
const sinon_1 = __importDefault(require("sinon"));
const { expect } = chai_1.default;
describe('Message Utils', () => {
    afterEach(() => {
        sinon_1.default.restore();
    });
    describe('toRawMessage', () => {
        it('can convert to raw message', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const message = test_utils_1.TestUtils.generateVisibleMessage();
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
            expect(Object.keys(rawMessage)).to.have.length(5);
            expect(rawMessage.identifier).to.exist;
            expect(rawMessage.device).to.exist;
            expect(rawMessage.encryption).to.exist;
            expect(rawMessage.plainTextBuffer).to.exist;
            expect(rawMessage.ttl).to.exist;
            expect(rawMessage.identifier).to.equal(message.identifier);
            expect(rawMessage.device).to.equal(device.key);
            expect(rawMessage.plainTextBuffer).to.deep.equal(message.plainTextBuffer());
            expect(rawMessage.ttl).to.equal(message.ttl());
        });
        it('should generate valid plainTextBuffer', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const message = test_utils_1.TestUtils.generateVisibleMessage();
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
            const rawBuffer = rawMessage.plainTextBuffer;
            const rawBufferJSON = JSON.stringify(rawBuffer);
            const messageBufferJSON = JSON.stringify(message.plainTextBuffer());
            expect(rawBuffer instanceof Uint8Array).to.equal(true, 'raw message did not contain a plainTextBuffer');
            expect(rawBufferJSON).to.equal(messageBufferJSON, 'plainTextBuffer was not converted correctly');
        });
        it('should maintain pubkey', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const message = test_utils_1.TestUtils.generateVisibleMessage();
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
            const derivedPubKey = types_1.PubKey.from(rawMessage.device);
            expect(derivedPubKey).to.not.be.eq(undefined, 'should maintain pubkey');
            expect(derivedPubKey?.isEqual(device)).to.equal(true, 'pubkey of message was not converted correctly');
        });
        it('should set encryption to ClosedGroup if a ClosedGroupVisibleMessage is passed in', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const groupId = test_utils_1.TestUtils.generateFakePubKey();
            const chatMessage = test_utils_1.TestUtils.generateVisibleMessage();
            const message = new ClosedGroupVisibleMessage_1.ClosedGroupVisibleMessage({ chatMessage, groupId });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE);
        });
        it('should set encryption to Fallback on other messages', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const message = test_utils_1.TestUtils.generateVisibleMessage();
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
        });
        it('passing ClosedGroupNewMessage returns Fallback', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const msg = new ClosedGroupNewMessage_1.ClosedGroupNewMessage({
                timestamp: Date.now(),
                name: 'df',
                members: [member],
                admins: [member],
                groupId: test_utils_1.TestUtils.generateFakePubKey().key,
                keypair: test_utils_1.TestUtils.generateFakeECKeyPair(),
                expireTimer: 0,
            });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, msg);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
        });
        it('passing ClosedGroupNameChangeMessage returns ClosedGroup', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const msg = new ClosedGroupNameChangeMessage_1.ClosedGroupNameChangeMessage({
                timestamp: Date.now(),
                name: 'df',
                groupId: test_utils_1.TestUtils.generateFakePubKey().key,
            });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, msg);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE);
        });
        it('passing ClosedGroupAddedMembersMessage returns ClosedGroup', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const msg = new ClosedGroupAddedMembersMessage_1.ClosedGroupAddedMembersMessage({
                timestamp: Date.now(),
                addedMembers: [test_utils_1.TestUtils.generateFakePubKey().key],
                groupId: test_utils_1.TestUtils.generateFakePubKey().key,
            });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, msg);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE);
        });
        it('passing ClosedGroupRemovedMembersMessage returns ClosedGroup', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const msg = new ClosedGroupRemovedMembersMessage_1.ClosedGroupRemovedMembersMessage({
                timestamp: Date.now(),
                removedMembers: [test_utils_1.TestUtils.generateFakePubKey().key],
                groupId: test_utils_1.TestUtils.generateFakePubKey().key,
            });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, msg);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE);
        });
        it('passing ClosedGroupEncryptionPairMessage returns ClosedGroup', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const fakeWrappers = new Array();
            fakeWrappers.push(new protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage.KeyPairWrapper({
                publicKey: new Uint8Array(8),
                encryptedKeyPair: new Uint8Array(8),
            }));
            const msg = new ClosedGroupEncryptionPairMessage_1.ClosedGroupEncryptionPairMessage({
                timestamp: Date.now(),
                groupId: test_utils_1.TestUtils.generateFakePubKey().key,
                encryptedKeyPairs: fakeWrappers,
            });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, msg);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE);
        });
        it('passing ClosedGroupEncryptionKeyPairReply returns Fallback', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const fakeWrappers = new Array();
            fakeWrappers.push(new protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage.KeyPairWrapper({
                publicKey: new Uint8Array(8),
                encryptedKeyPair: new Uint8Array(8),
            }));
            const msg = new ClosedGroupEncryptionPairReplyMessage_1.ClosedGroupEncryptionPairReplyMessage({
                timestamp: Date.now(),
                groupId: test_utils_1.TestUtils.generateFakePubKey().key,
                encryptedKeyPairs: fakeWrappers,
            });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, msg);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
        });
        it('passing a ConfigurationMessage returns Fallback', async () => {
            const device = test_utils_1.TestUtils.generateFakePubKey();
            const msg = new ConfigurationMessage_1.ConfigurationMessage({
                timestamp: Date.now(),
                activeOpenGroups: [],
                activeClosedGroups: [],
                displayName: 'displayName',
                contacts: [],
            });
            const rawMessage = await utils_1.MessageUtils.toRawMessage(device, msg);
            expect(rawMessage.encryption).to.equal(protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
        });
    });
    describe('getCurrentConfigurationMessage', () => {
        const ourNumber = test_utils_1.TestUtils.generateFakePubKey().key;
        beforeEach(() => {
            sinon_1.default.stub(utils_1.UserUtils, 'getOurPubKeyStrFromCache').resolves(ourNumber);
            sinon_1.default.stub(utils_1.UserUtils, 'getOurPubKeyFromCache').resolves(types_1.PubKey.cast(ourNumber));
        });
        beforeEach(() => {
            sinon_1.default.restore();
        });
    });
});
