"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = __importDefault(require("chai"));
const bytebuffer_1 = __importDefault(require("bytebuffer"));
const utils_1 = require("../../../../bchat/utils");
const chai_as_promised_1 = __importDefault(require("chai-as-promised"));
chai_1.default.use(chai_as_promised_1.default);
const { expect } = chai_1.default;
describe('String Utils', () => {
    describe('encode', () => {
        it('can encode to base64', () => {
            const testString = 'AAAAAAAAAA';
            const encoded = utils_1.StringUtils.encode(testString, 'base64');
            expect(encoded instanceof ArrayBuffer).to.equal(true, 'a buffer was not returned from `encode`');
            expect(encoded.byteLength).to.be.greaterThan(0);
        });
        it('can encode to hex', () => {
            const testString = 'AAAAAAAAAA';
            const encoded = utils_1.StringUtils.encode(testString, 'hex');
            expect(encoded instanceof ArrayBuffer).to.equal(true, 'a buffer was not returned from `encode`');
            expect(encoded.byteLength).to.be.greaterThan(0);
        });
        it('wont encode invalid hex', () => {
            const testString = 'ZZZZZZZZZZ';
            const encoded = utils_1.StringUtils.encode(testString, 'hex');
            expect(encoded.byteLength).to.equal(0);
        });
        it('can encode to binary', () => {
            const testString = 'AAAAAAAAAA';
            const encoded = utils_1.StringUtils.encode(testString, 'binary');
            expect(encoded instanceof ArrayBuffer).to.equal(true, 'a buffer was not returned from `encode`');
            expect(encoded.byteLength).to.be.greaterThan(0);
        });
        it('can encode to utf8', () => {
            const testString = 'AAAAAAAAAA';
            const encoded = utils_1.StringUtils.encode(testString, 'binary');
            expect(encoded instanceof ArrayBuffer).to.equal(true, 'a buffer was not returned from `encode`');
            expect(encoded.byteLength).to.be.greaterThan(0);
        });
        it('can encode empty string', () => {
            const testString = '';
            expect(testString).to.have.length(0);
            const allEncodedings = ['base64', 'hex', 'binary', 'utf8'].map(e => utils_1.StringUtils.encode(testString, e));
            allEncodedings.forEach(encoded => {
                expect(encoded instanceof ArrayBuffer).to.equal(true, 'a buffer was not returned from `encode`');
                expect(encoded.byteLength).to.equal(0);
            });
        });
        it('can encode huge string', () => {
            const stringSize = Math.pow(2, 16);
            const testString = Array(stringSize)
                .fill('0')
                .join('');
            const allEncodedings = ['base64', 'hex', 'binary', 'utf8'].map(e => utils_1.StringUtils.encode(testString, e));
            allEncodedings.forEach(encoded => {
                expect(encoded instanceof ArrayBuffer).to.equal(true, 'a buffer was not returned from `encode`');
                expect(encoded.byteLength).to.be.greaterThan(0);
            });
        });
        it("won't encode illegal string length in hex", () => {
            const testString = 'A';
            const encode = () => utils_1.StringUtils.encode(testString, 'hex');
            expect(testString.length % 2).to.equal(1);
            expect(encode).to.throw('Illegal str: Length not a multiple of 2');
        });
        it('can encode obscure string', () => {
            const testString = '↓←¶ᶑᵶ⅑⏕→⅓‎ᵹ⅙ᵰᶎ⅔⅗↔‌ᶈ⅞⁯⸜ᶊ⁬ᵴᶉ↉⁭¥ᶖᶋᶃᶓ⏦ᵾᶂᶆ↕⸝ᶔᶐ⏔£⏙⅐⅒ᶌ⁁ᶘᶄᶒ⁪ᶸ⅘‏⁮⅚⅛ᶙᶇᶕᶀ↑ᵿ⏠ᶍᵯ⏖⏗⅜ᶚᶏ⁊‍ᶁᶗᵽ⁫ᵼ⅝⏘⅖⅕⏡';
            const encodings = ['base64', 'binary', 'utf8'];
            encodings.forEach(encoding => {
                const encoded = utils_1.StringUtils.encode(testString, encoding);
                expect(encoded instanceof ArrayBuffer).to.equal(true, `a buffer was not returned using encoding: '${encoding}'`);
                expect(encoded.byteLength).to.be.greaterThan(0);
            });
        });
    });
    describe('decode', () => {
        it('can decode empty buffer', () => {
            const buffer = new bytebuffer_1.default(0);
            const encodings = ['base64', 'hex', 'binary', 'utf8'];
            encodings.forEach(encoding => {
                const decoded = utils_1.StringUtils.decode(buffer, encoding);
                expect(decoded).to.exist;
                expect(typeof decoded === String.name.toLowerCase());
                expect(decoded).to.have.length(0);
            });
        });
        it('can decode huge buffer', () => {
            const bytes = Math.pow(2, 16);
            const bufferString = Array(bytes)
                .fill('A')
                .join('');
            const buffer = bytebuffer_1.default.fromUTF8(bufferString);
            const encodings = ['base64', 'hex', 'binary', 'utf8'];
            encodings.forEach(encoding => {
                const decoded = utils_1.StringUtils.decode(buffer, encoding);
                expect(decoded).to.exist;
                expect(typeof decoded === String.name.toLowerCase());
                expect(decoded).to.have.length.greaterThan(0);
            });
        });
        it('can decode from ByteBuffer', () => {
            const buffer = bytebuffer_1.default.fromUTF8('AAAAAAAAAA');
            const encodings = ['base64', 'hex', 'binary', 'utf8'];
            encodings.forEach(encoding => {
                const decoded = utils_1.StringUtils.decode(buffer, encoding);
                expect(decoded).to.exist;
                expect(typeof decoded === String.name.toLowerCase());
                expect(decoded).to.have.length.greaterThan(0);
            });
        });
        it('can decode from Buffer', () => {
            const arrayBuffer = new ArrayBuffer(10);
            const buffer = Buffer.from(arrayBuffer);
            buffer.writeUInt8(0, 0);
            const encodings = ['base64', 'hex', 'binary', 'utf8'];
            encodings.forEach(encoding => {
                const decoded = utils_1.StringUtils.decode(buffer, encoding);
                expect(decoded).to.exist;
                expect(typeof decoded === String.name.toLowerCase());
                expect(decoded).to.have.length.greaterThan(0);
            });
        });
        it('can decode from ArrayBuffer', () => {
            const buffer = new ArrayBuffer(10);
            const encodings = ['base64', 'hex', 'binary', 'utf8'];
            encodings.forEach(encoding => {
                const decoded = utils_1.StringUtils.decode(buffer, encoding);
                expect(decoded).to.exist;
                expect(typeof decoded === String.name.toLowerCase());
                expect(decoded).to.have.length.greaterThan(0);
            });
        });
        it('can decode from Uint8Array', () => {
            const buffer = new Uint8Array(10);
            const encodings = ['base64', 'hex', 'binary', 'utf8'];
            encodings.forEach(encoding => {
                const decoded = utils_1.StringUtils.decode(buffer, encoding);
                expect(decoded).to.exist;
                expect(typeof decoded === String.name.toLowerCase());
                expect(decoded).to.have.length.greaterThan(0);
            });
        });
    });
});
