"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const lodash_1 = require("lodash");
const MIME_1 = require("../../../types/MIME");
const groupMediaItemsByDate_1 = require("../../../components/conversation/media-gallery/groupMediaItemsByDate");
const test_utils_1 = require("../../test-utils");
const generatedMessageSenderKey = test_utils_1.TestUtils.generateFakePubKey().key;
const toMediaItem = (date) => ({
    objectURL: date.toUTCString(),
    index: 0,
    attachment: {
        fileName: 'fileName',
        contentType: MIME_1.IMAGE_JPEG,
        url: 'url',
        fileSize: null,
        screenshot: null,
        thumbnail: null,
        path: '123456',
        id: 123456,
    },
    contentType: MIME_1.IMAGE_JPEG,
    messageSender: generatedMessageSenderKey,
    messageTimestamp: date.getTime(),
    messageId: '123456',
});
describe('groupMediaItemsByDate', () => {
    it('should group mediaItems', () => {
        const referenceTime = new Date('2018-04-12T18:00Z').getTime();
        const input = (0, lodash_1.shuffle)([
            toMediaItem(new Date('2018-04-12T12:00Z')),
            toMediaItem(new Date('2018-04-12T00:01Z')),
            toMediaItem(new Date('2018-04-11T23:59Z')),
            toMediaItem(new Date('2018-04-09T00:01Z')),
            toMediaItem(new Date('2018-04-08T23:59Z')),
            toMediaItem(new Date('2018-04-01T00:01Z')),
            toMediaItem(new Date('2018-03-31T23:59Z')),
            toMediaItem(new Date('2018-03-01T14:00Z')),
            toMediaItem(new Date('2011-02-28T23:59Z')),
            toMediaItem(new Date('2011-02-01T10:00Z')),
        ]);
        const expected = [
            {
                type: 'today',
                mediaItems: [
                    {
                        objectURL: 'Thu, 12 Apr 2018 12:00:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1523534400000,
                        messageId: '123456',
                    },
                    {
                        objectURL: 'Thu, 12 Apr 2018 00:01:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1523491260000,
                        messageId: '123456',
                    },
                ],
            },
            {
                type: 'yesterday',
                mediaItems: [
                    {
                        objectURL: 'Wed, 11 Apr 2018 23:59:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1523491140000,
                        messageId: '123456',
                    },
                ],
            },
            {
                type: 'thisWeek',
                mediaItems: [
                    {
                        objectURL: 'Mon, 09 Apr 2018 00:01:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1523232060000,
                        messageId: '123456',
                    },
                ],
            },
            {
                type: 'thisMonth',
                mediaItems: [
                    {
                        objectURL: 'Sun, 08 Apr 2018 23:59:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1523231940000,
                        messageId: '123456',
                    },
                    {
                        objectURL: 'Sun, 01 Apr 2018 00:01:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1522540860000,
                        messageId: '123456',
                    },
                ],
            },
            {
                type: 'yearMonth',
                year: 2018,
                month: 2,
                mediaItems: [
                    {
                        objectURL: 'Sat, 31 Mar 2018 23:59:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1522540740000,
                        messageId: '123456',
                    },
                    {
                        objectURL: 'Thu, 01 Mar 2018 14:00:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1519912800000,
                        messageId: '123456',
                    },
                ],
            },
            {
                type: 'yearMonth',
                year: 2011,
                month: 1,
                mediaItems: [
                    {
                        objectURL: 'Mon, 28 Feb 2011 23:59:00 GMT',
                        index: 0,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        contentType: MIME_1.IMAGE_JPEG,
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1298937540000,
                        messageId: '123456',
                    },
                    {
                        objectURL: 'Tue, 01 Feb 2011 10:00:00 GMT',
                        index: 0,
                        contentType: MIME_1.IMAGE_JPEG,
                        messageSender: generatedMessageSenderKey,
                        messageTimestamp: 1296554400000,
                        attachment: {
                            fileName: 'fileName',
                            contentType: MIME_1.IMAGE_JPEG,
                            url: 'url',
                            fileSize: null,
                            screenshot: null,
                            thumbnail: null,
                            path: '123456',
                            id: 123456,
                        },
                        messageId: '123456',
                    },
                ],
            },
        ];
        const actual = (0, groupMediaItemsByDate_1.groupMediaItemsByDate)(referenceTime, input);
        chai_1.assert.deepEqual(actual, expected);
    });
});
