"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const protobuf_1 = require("../../protobuf");
const initializeAttachmentMetadata_1 = require("../../types/message/initializeAttachmentMetadata");
const utils_1 = require("../test-utils/utils");
describe('initializeAttachmentMetadata', () => {
    beforeEach(() => {
        (0, utils_1.stubWindowLog)();
    });
    describe('hasAttachmentInMessage', () => {
        it('no attachments should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.eq(false);
        });
        it('empty list attachments should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment has undefined content type should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: undefined }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment has null content type should return true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: null }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('first attachment is gif should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'image/gif' }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is gif should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'image/gif' }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is jpeg should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'image/jpeg' }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is png should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'image/png' }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is JPG should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'image/JPG' }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is PNG should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'image/PNG' }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is audio should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'audio/mp3' }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is flagged as voice message should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [
                    { contentType: 'audio/mp3', flags: protobuf_1.SignalService.AttachmentPointer.Flags.VOICE_MESSAGE },
                ],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment is flagged as voice message but no content type is false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [
                    { contentType: undefined, flags: protobuf_1.SignalService.AttachmentPointer.Flags.VOICE_MESSAGE },
                ],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment content type is audio and other is null should return true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({ attachments: [{ contentType: 'audio/mp3' }, { contentType: null }] });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('first attachment content type is audio and other is null should return true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'audio/mp3' }, { contentType: 'file/whatever' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasFileAttachmentInMessage)(msgModel)).to.be.true;
        });
    });
    describe('hasVisualMediaAttachmentInMessage', () => {
        it('no attachments should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('empty attachments list should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment type is undefined should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: undefined }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment type is null should return false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: null }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.false;
        });
        it('first attachment type is image/whatever should return true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'image/whatever' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('first attachment type is jpeg should return true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'image/jpeg' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('first attachment type is png should return true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'image/png' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('first attachment type is JPG should return true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'image/JPG' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('multiple attachments where one is not image and one is returns true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'audio/whatever' }, { contentType: 'image/JPG' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('multiple attachments where both are images returns true', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'image/whatever' }, { contentType: 'image/JPG' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.true;
        });
        it('multiple attachments  where none are images returns false', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set({
                attachments: [{ contentType: 'audio/whatever' }, { contentType: 'audio/JPG' }],
            });
            (0, chai_1.expect)((0, initializeAttachmentMetadata_1.hasVisualMediaAttachmentInMessage)(msgModel)).to.be.false;
        });
    });
    describe('getAttachmentMetadata', () => {
        it('no attachments should return false x3', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(0);
        });
        it('empty attachments [] should return false x3', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', []);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(0);
        });
        it('has one image attachment only', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', [{ contentType: 'image/jpeg' }]);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(1);
        });
        it('has two image attachment only', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', [{ contentType: 'image/jpeg' }, { contentType: 'image/jpeg' }]);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(1);
        });
        it('has one audio attachment only', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', [{ contentType: 'audio/mp3' }]);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(0);
        });
        it('has one file attachment only', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', [{ contentType: 'whatever' }]);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(0);
        });
        it('has two file attachment only', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', [{ contentType: 'whatever' }, { contentType: 'whatever' }]);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(0);
        });
        it('has two attachments with undefined contenttype', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', [{ contentType: undefined }, { contentType: undefined }]);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(0);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(0);
        });
        it('has two attachments with null contenttype', () => {
            const msgModel = (0, utils_1.generateFakeIncomingPrivateMessage)();
            msgModel.set('attachments', [{ contentType: null }, { contentType: null }]);
            const mt = (0, initializeAttachmentMetadata_1.getAttachmentMetadata)(msgModel);
            (0, chai_1.expect)(mt.hasAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasFileAttachments).to.be.eq(1);
            (0, chai_1.expect)(mt.hasVisualMediaAttachments).to.be.eq(0);
        });
    });
});
