"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const blockedNumberController_1 = require("../../util/blockedNumberController");
const test_utils_1 = require("../test-utils");
const utils_1 = require("../../bchat/utils");
const sinon_1 = __importDefault(require("sinon"));
describe('BlockedNumberController', () => {
    let memoryDB;
    beforeEach(() => {
        memoryDB = {};
        test_utils_1.TestUtils.stubDataItem('createOrUpdateItem').callsFake(data => {
            memoryDB[data.id] = data.value;
        });
        test_utils_1.TestUtils.stubDataItem('getItemById').callsFake(id => {
            if (!memoryDB[id]) {
                return undefined;
            }
            const value = memoryDB[id];
            return {
                id,
                value,
            };
        });
        blockedNumberController_1.BlockedNumberController.reset();
    });
    afterEach(() => {
        sinon_1.default.restore();
    });
    describe('load', () => {
        it('should load data from the database', async () => {
            const normal = test_utils_1.TestUtils.generateFakePubKey();
            const group = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [normal.key];
            memoryDB['blocked-groups'] = [group.key];
            await blockedNumberController_1.BlockedNumberController.load();
            const blockedNumbers = blockedNumberController_1.BlockedNumberController.getBlockedNumbers();
            const blockedGroups = blockedNumberController_1.BlockedNumberController.getBlockedGroups();
            (0, chai_1.expect)(blockedNumbers).to.have.lengthOf(1);
            (0, chai_1.expect)(blockedNumbers).to.include(normal.key);
            (0, chai_1.expect)(blockedGroups).to.have.lengthOf(1);
            (0, chai_1.expect)(blockedGroups).to.include(group.key);
        });
        it('should return empty if nothing in the db exists', async () => {
            await blockedNumberController_1.BlockedNumberController.load();
            const blockedNumbers = blockedNumberController_1.BlockedNumberController.getBlockedNumbers();
            const blockedGroups = blockedNumberController_1.BlockedNumberController.getBlockedGroups();
            (0, chai_1.expect)(blockedNumbers).to.be.empty;
            (0, chai_1.expect)(blockedGroups).to.be.empty;
        });
    });
    describe('block', () => {
        it('should block the user', async () => {
            const other = test_utils_1.TestUtils.generateFakePubKey();
            await blockedNumberController_1.BlockedNumberController.block(other);
            const blockedNumbers = blockedNumberController_1.BlockedNumberController.getBlockedNumbers();
            (0, chai_1.expect)(blockedNumbers).to.have.lengthOf(1);
            (0, chai_1.expect)(blockedNumbers).to.include(other.key);
            (0, chai_1.expect)(memoryDB.blocked).to.include(other.key);
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.getBlockedGroups()).to.be.empty;
        });
    });
    describe('unblock', () => {
        it('should unblock the user', async () => {
            const primary = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [primary.key];
            await blockedNumberController_1.BlockedNumberController.unblock(primary);
            const blockedNumbers = blockedNumberController_1.BlockedNumberController.getBlockedNumbers();
            (0, chai_1.expect)(blockedNumbers).to.be.empty;
            (0, chai_1.expect)(memoryDB.blocked).to.be.empty;
        });
        it('should only unblock if a device was blocked', async () => {
            const pubKey = test_utils_1.TestUtils.generateFakePubKey();
            const another = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [pubKey.key, another.key];
            await blockedNumberController_1.BlockedNumberController.unblock(pubKey);
            const blockedNumbers = blockedNumberController_1.BlockedNumberController.getBlockedNumbers();
            (0, chai_1.expect)(blockedNumbers).to.have.lengthOf(1);
            (0, chai_1.expect)(blockedNumbers).to.include(another.key);
            (0, chai_1.expect)(memoryDB.blocked).to.have.lengthOf(1);
            (0, chai_1.expect)(memoryDB.blocked).to.include(another.key);
        });
    });
    describe('blockGroup', () => {
        it('should block a group', async () => {
            const group = test_utils_1.TestUtils.generateFakePubKey();
            await blockedNumberController_1.BlockedNumberController.blockGroup(group);
            const blockedGroups = blockedNumberController_1.BlockedNumberController.getBlockedGroups();
            (0, chai_1.expect)(blockedGroups).to.have.lengthOf(1);
            (0, chai_1.expect)(blockedGroups).to.include(group.key);
            (0, chai_1.expect)(memoryDB['blocked-groups']).to.have.lengthOf(1);
            (0, chai_1.expect)(memoryDB['blocked-groups']).to.include(group.key);
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.getBlockedNumbers()).to.be.empty;
        });
    });
    describe('unblockGroup', () => {
        it('should unblock a group', async () => {
            const group = test_utils_1.TestUtils.generateFakePubKey();
            const another = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB['blocked-groups'] = [group.key, another.key];
            await blockedNumberController_1.BlockedNumberController.unblockGroup(group);
            const blockedGroups = blockedNumberController_1.BlockedNumberController.getBlockedGroups();
            (0, chai_1.expect)(blockedGroups).to.have.lengthOf(1);
            (0, chai_1.expect)(blockedGroups).to.include(another.key);
            (0, chai_1.expect)(memoryDB['blocked-groups']).to.have.lengthOf(1);
            (0, chai_1.expect)(memoryDB['blocked-groups']).to.include(another.key);
        });
    });
    describe('isBlocked', () => {
        it('should return true if number is blocked', async () => {
            const pubKey = test_utils_1.TestUtils.generateFakePubKey();
            const groupPubKey = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [pubKey.key];
            memoryDB['blocked-groups'] = [groupPubKey.key];
            await blockedNumberController_1.BlockedNumberController.load();
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.isBlocked(pubKey.key)).to.equal(true, 'Expected isBlocked to return true for user pubkey');
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.isBlocked(groupPubKey.key)).to.equal(false, 'Expected isBlocked to return false for a group pubkey');
        });
        it('should return false if number is not blocked', async () => {
            const pubKey = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [];
            await blockedNumberController_1.BlockedNumberController.load();
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.isBlocked(pubKey.key)).to.equal(false, 'Expected isBlocked to return false');
        });
    });
    describe('isBlockedAsync', () => {
        let ourDevice;
        beforeEach(() => {
            ourDevice = test_utils_1.TestUtils.generateFakePubKey();
            sinon_1.default.stub(utils_1.UserUtils, 'getOurPubKeyStrFromCache').returns(ourDevice.key);
        });
        it('should return false for our device', async () => {
            const isBlocked = await blockedNumberController_1.BlockedNumberController.isBlockedAsync(ourDevice);
            (0, chai_1.expect)(isBlocked).to.equal(false, 'Expected our device to return false');
        });
        it('should return true if the device is blocked', async () => {
            const other = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [other.key];
            const isBlocked = await blockedNumberController_1.BlockedNumberController.isBlockedAsync(other);
            (0, chai_1.expect)(isBlocked).to.equal(true, 'Expected isBlockedAsync to return true.');
        });
        it('should return false if device is not blocked', async () => {
            const other = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [];
            const isBlocked = await blockedNumberController_1.BlockedNumberController.isBlockedAsync(other);
            (0, chai_1.expect)(isBlocked).to.equal(false, 'Expected isBlockedAsync to return false.');
        });
    });
    describe('isGroupBlocked', () => {
        it('should return true if group is blocked', async () => {
            const pubKey = test_utils_1.TestUtils.generateFakePubKey();
            const groupPubKey = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB.blocked = [pubKey.key];
            memoryDB['blocked-groups'] = [groupPubKey.key];
            await blockedNumberController_1.BlockedNumberController.load();
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.isGroupBlocked(pubKey.key)).to.equal(false, 'Expected isGroupBlocked to return false for user pubkey');
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.isGroupBlocked(groupPubKey.key)).to.equal(true, 'Expected isGroupBlocked to return true for a group pubkey');
        });
        it('should return false if group is not blocked', async () => {
            const groupPubKey = test_utils_1.TestUtils.generateFakePubKey();
            memoryDB['blocked-groups'] = [];
            await blockedNumberController_1.BlockedNumberController.load();
            (0, chai_1.expect)(blockedNumberController_1.BlockedNumberController.isGroupBlocked(groupPubKey.key)).to.equal(false, 'Expected isGroupBlocked to return false');
        });
    });
});
