"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.registerSingleDevice = exports.signInByLinkingDevice = exports.signInWithRecovery = exports.bchatGenerateKeyPair = void 0;
const conversations_1 = require("../bchat/conversations");
const crypto_1 = require("../bchat/crypto");
const String_1 = require("../bchat/utils/String");
const User_1 = require("../bchat/utils/User");
const events_1 = require("../shims/events");
const user_1 = require("../state/ducks/user");
const mnemonic_1 = require("../bchat/crypto/mnemonic");
const conversation_1 = require("../models/conversation");
const settings_key_1 = require("../data/settings-key");
const storage_1 = require("./storage");
const registration_1 = require("./registration");
async function bchatGenerateKeyPair(seed) {
    const sodium = await (0, crypto_1.getSodiumRenderer)();
    const ed25519KeyPair = sodium.crypto_sign_seed_keypair(new Uint8Array(seed));
    const x25519PublicKey = sodium.crypto_sign_ed25519_pk_to_curve25519(ed25519KeyPair.publicKey);
    const origPub = new Uint8Array(x25519PublicKey);
    const prependedX25519PublicKey = new Uint8Array(33);
    prependedX25519PublicKey.set(origPub, 1);
    prependedX25519PublicKey[0] = 189;
    const x25519SecretKey = sodium.crypto_sign_ed25519_sk_to_curve25519(ed25519KeyPair.privateKey);
    const x25519KeyPair = {
        pubKey: prependedX25519PublicKey.buffer,
        privKey: x25519SecretKey.buffer,
        ed25519KeyPair,
    };
    return x25519KeyPair;
}
exports.bchatGenerateKeyPair = bchatGenerateKeyPair;
const generateKeypair = async (mnemonic, mnemonicLanguage) => {
    let seedHex = (0, mnemonic_1.mn_decode)(mnemonic, mnemonicLanguage);
    const privKeyHexLength = 32 * 2;
    if (seedHex.length !== privKeyHexLength) {
        seedHex = seedHex.concat('0'.repeat(32));
        seedHex = seedHex.substring(0, privKeyHexLength);
    }
    const seed = (0, String_1.fromHex)(seedHex);
    return bchatGenerateKeyPair(seed);
};
async function signInWithRecovery(mnemonic, mnemonicLanguage, profileName) {
    return registerSingleDevice(mnemonic, mnemonicLanguage, profileName, 0);
}
exports.signInWithRecovery = signInWithRecovery;
async function signInByLinkingDevice(mnemonic, mnemonicLanguage) {
    if (!mnemonic) {
        throw new Error('BChat always needs a mnemonic. Either generated or given by the user');
    }
    if (!mnemonicLanguage) {
        throw new Error('We always needs a mnemonicLanguage');
    }
    const identityKeyPair = await generateKeypair(mnemonic, mnemonicLanguage);
    await (0, storage_1.setSignInByLinking)(true);
    await createAccount(identityKeyPair);
    await (0, storage_1.saveRecoveryPhrase)(mnemonic);
    const pubKeyString = (0, String_1.toHex)(identityKeyPair.pubKey);
    await registrationDone(pubKeyString, '', 0);
    return pubKeyString;
}
exports.signInByLinkingDevice = signInByLinkingDevice;
async function registerSingleDevice(generatedMnemonic, mnemonicLanguage, profileName, deamonHeight) {
    if (!generatedMnemonic) {
        throw new Error('BChat always needs a mnemonic. Either generated or given by the user');
    }
    if (!profileName) {
        throw new Error('We always needs a profileName');
    }
    if (!mnemonicLanguage) {
        throw new Error('We always needs a mnemonicLanguage');
    }
    const identityKeyPair = await generateKeypair(generatedMnemonic, mnemonicLanguage);
    await createAccount(identityKeyPair);
    await (0, storage_1.saveRecoveryPhrase)(generatedMnemonic);
    await (0, storage_1.setLastProfileUpdateTimestamp)(Date.now());
    const pubKeyString = (0, String_1.toHex)(identityKeyPair.pubKey);
    await registrationDone(pubKeyString, profileName, deamonHeight);
}
exports.registerSingleDevice = registerSingleDevice;
async function createAccount(identityKeyPair) {
    const sodium = await (0, crypto_1.getSodiumRenderer)();
    let password = (0, String_1.fromArrayBufferToBase64)(sodium.randombytes_buf(16));
    password = password.substring(0, password.length - 2);
    await Promise.all([
        storage_1.Storage.remove('identityKey'),
        storage_1.Storage.remove('signaling_key'),
        storage_1.Storage.remove('password'),
        storage_1.Storage.remove('registrationId'),
        storage_1.Storage.remove('number_id'),
        storage_1.Storage.remove('device_name'),
        storage_1.Storage.remove('userAgent'),
        storage_1.Storage.remove(settings_key_1.SettingsKey.settingsReadReceipt),
        storage_1.Storage.remove(settings_key_1.SettingsKey.settingsTypingIndicator),
        storage_1.Storage.remove('regionCode'),
        storage_1.Storage.remove('local_attachment_encrypted_key'),
    ]);
    const pubKeyString = (0, String_1.toHex)(identityKeyPair.pubKey);
    await storage_1.Storage.put('identityKey', identityKeyPair);
    await storage_1.Storage.put('password', password);
    await storage_1.Storage.put(settings_key_1.SettingsKey.settingsReadReceipt, false);
    await storage_1.Storage.put(settings_key_1.SettingsKey.settingsTypingIndicator, false);
    await storage_1.Storage.put(settings_key_1.SettingsKey.settingsOpengroupPruning, true);
    await window.setOpengroupPruning(true);
    await (0, storage_1.setLocalPubKey)(pubKeyString);
}
async function registrationDone(ourPubkey, displayName, deamonHeight) {
    window?.log?.info('registration done');
    await storage_1.Storage.put('primaryDevicePubKey', ourPubkey);
    const conversation = await (0, conversations_1.getConversationController)().getOrCreateAndWait(ourPubkey, conversation_1.ConversationTypeEnum.PRIVATE);
    await conversation.setBchatProfile({ displayName });
    await conversation.setIsApproved(true);
    await conversation.setDidApproveMe(true);
    await conversation.setwalletCreatedDaemonHeight(deamonHeight);
    await conversation.commit();
    const user = {
        ourNumber: (0, User_1.getOurPubKeyStrFromCache)(),
        ourPrimary: ourPubkey,
    };
    window.inboxStore?.dispatch(user_1.actions.userChanged(user));
    await registration_1.Registration.markDone();
    window?.log?.info('dispatching registration event');
    (0, events_1.trigger)('registration_done');
}
