"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.joinOpenGroupV2WithUIEvents = exports.parseOpenGroupV2 = void 0;
const opengroups_1 = require("../../../../data/opengroups");
const conversations_1 = require("../../../../state/ducks/conversations");
const conversations_2 = require("../../../conversations");
const utils_1 = require("../../../utils");
const syncUtils_1 = require("../../../utils/syncUtils");
const OpenGroupUtils_1 = require("../utils/OpenGroupUtils");
const OpenGroupManagerV2_1 = require("./OpenGroupManagerV2");
function parseOpenGroupV2(urlWithPubkey) {
    const lowerCased = urlWithPubkey.trim().toLowerCase();
    try {
        if (!OpenGroupUtils_1.openGroupV2CompleteURLRegex.test(lowerCased)) {
            throw new Error('regex fail');
        }
        const prefixedUrl = (0, OpenGroupUtils_1.prefixify)(lowerCased);
        const url = new URL(prefixedUrl);
        const serverUrl = `${url.protocol}//${url.host}`;
        const room = {
            serverUrl,
            roomId: url.pathname.slice(1),
            serverPublicKey: url.search.slice(OpenGroupUtils_1.publicKeyParam.length + 1),
        };
        return room;
    }
    catch (e) {
        window?.log?.error('Invalid Opengroup v2 join URL:', lowerCased, e);
    }
    return undefined;
}
exports.parseOpenGroupV2 = parseOpenGroupV2;
async function joinOpenGroupV2(room, fromConfigMessage) {
    if (!room.serverUrl || !room.roomId || room.roomId.length < 2 || !room.serverPublicKey) {
        return;
    }
    const serverUrl = room.serverUrl.toLowerCase();
    const roomId = room.roomId.toLowerCase();
    const publicKey = room.serverPublicKey.toLowerCase();
    const prefixedServer = (0, OpenGroupUtils_1.prefixify)(serverUrl);
    const alreadyExist = await (0, opengroups_1.getV2OpenGroupRoomByRoomId)({ serverUrl, roomId });
    const conversationId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(serverUrl, roomId);
    const existingConvo = (0, conversations_2.getConversationController)().get(conversationId);
    if (alreadyExist && existingConvo) {
        window?.log?.warn('Skipping join opengroupv2: already exists');
        return;
    }
    else if (existingConvo) {
        window?.log?.warn('leaving before rejoining Social group v2 room', conversationId);
        await (0, conversations_2.getConversationController)().deleteContact(conversationId);
    }
    try {
        const conversation = await utils_1.PromiseUtils.timeout((0, OpenGroupManagerV2_1.getOpenGroupManager)().attemptConnectionV2OneAtATime(prefixedServer, roomId, publicKey), 20000);
        if (!conversation) {
            window?.log?.warn('Failed to join Social group v2');
            throw new Error(window.i18n('connectToServerFail'));
        }
        if (!fromConfigMessage) {
            await (0, syncUtils_1.forceSyncConfigurationNowIfNeeded)();
        }
    }
    catch (e) {
        window?.log?.error('Could not join Social group v2', e.message);
        throw e;
    }
}
async function joinOpenGroupV2WithUIEvents(completeUrl, showToasts, fromConfigMessage, uiCallback) {
    try {
        const parsedRoom = parseOpenGroupV2(completeUrl);
        if (!parsedRoom) {
            if (showToasts) {
                utils_1.ToastUtils.pushToastError('connectToServer', window.i18n('invalidSocialGroupUrl'));
            }
            return false;
        }
        const conversationID = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(parsedRoom.serverUrl, parsedRoom.roomId);
        if ((0, conversations_2.getConversationController)().get(conversationID)) {
            if (showToasts) {
                utils_1.ToastUtils.pushToastWarning('publicChatExists', window.i18n('publicChatExists'));
            }
            return false;
        }
        if (showToasts) {
            utils_1.ToastUtils.pushToastInfo('connectingToServer', window.i18n('connectingToServer'));
        }
        if (uiCallback) {
            uiCallback(true);
        }
        await joinOpenGroupV2(parsedRoom, fromConfigMessage);
        const isConvoCreated = (0, conversations_2.getConversationController)().get(conversationID);
        if (isConvoCreated) {
            await (0, conversations_1.openConversationWithMessages)({ conversationKey: conversationID, messageId: null });
            if (showToasts) {
                utils_1.ToastUtils.pushToastSuccess('connectToServerSuccess', window.i18n('connectToServerSuccess'));
            }
            return true;
        }
        else {
            if (showToasts) {
                utils_1.ToastUtils.pushToastError('connectToServerFail', window.i18n('connectToServerFail'));
            }
        }
    }
    catch (error) {
        window?.log?.warn('got error while joining Social group:', error.message);
        if (showToasts) {
            utils_1.ToastUtils.pushToastError('connectToServerFail', window.i18n('connectToServerFail'));
        }
    }
    finally {
        if (uiCallback) {
            uiCallback(false);
        }
    }
    return false;
}
exports.joinOpenGroupV2WithUIEvents = joinOpenGroupV2WithUIEvents;
