"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.removeModerator = exports.addModerator = exports.uploadImageForRoomOpenGroupV2 = exports.uploadFileOpenGroupV2 = exports.downloadPreviewOpenGroupV2 = exports.downloadFileOpenGroupV2ByUrl = exports.downloadFileOpenGroupV2 = exports.getMemberCount = exports.getAllRoomInfos = exports.deleteMessageByServerIds = exports.unbanUser = exports.banUser = exports.postMessage = exports.postMessageRetryable = exports.openGroupV2GetRoomInfo = exports.sendApiV2Request = exports.getMinTimeout = void 0;
const opengroups_1 = require("../../../../data/opengroups");
const file_server_api_1 = require("../../file_server_api");
const onionSend_1 = require("../../../onions/onionSend");
const OpenGroupAPIV2Parser_1 = require("./OpenGroupAPIV2Parser");
const OpenGroupMessageV2_1 = require("./OpenGroupMessageV2");
const FileServerApiV2_1 = require("../../file_server_api/FileServerApiV2");
const ApiAuth_1 = require("./ApiAuth");
const p_retry_1 = __importDefault(require("p-retry"));
const util_worker_interface_1 = require("../../../../webworker/workers/util_worker_interface");
const utils_1 = require("../../../utils");
const getMinTimeout = () => 1000;
exports.getMinTimeout = getMinTimeout;
function getCompleteEndpointUrl(roomInfos, endpoint, hasRoomInEndpoint) {
    if (!hasRoomInEndpoint) {
        return `${roomInfos.serverUrl}/${roomInfos.roomId}/${endpoint}`;
    }
    return `${roomInfos.serverUrl}/${endpoint}`;
}
const getDestinationPubKey = async (request) => {
    if (file_server_api_1.FSv2.isOpenGroupV2Request(request)) {
        if (!request.serverPublicKey) {
            const roomDetails = await (0, opengroups_1.getV2OpenGroupRoomByRoomId)({
                serverUrl: request.server,
                roomId: request.room,
            });
            if (!roomDetails?.serverPublicKey) {
                throw new Error('PublicKey not found for this server.');
            }
            return roomDetails.serverPublicKey;
        }
        else {
            return request.serverPublicKey;
        }
    }
    else {
        return file_server_api_1.FSv2.fileServerV2PubKey;
    }
};
async function sendApiV2Request(request) {
    const builtUrl = file_server_api_1.FSv2.buildUrl(request);
    if (!builtUrl) {
        throw new Error('Invalid request');
    }
    if (!window.getGlobalOnlineStatus()) {
        throw new p_retry_1.default.AbortError('Network is not available');
    }
    const headers = request.headers || {};
    if (file_server_api_1.FSv2.isOpenGroupV2Request(request)) {
        headers.Room = request.room;
    }
    let body = '';
    if (request.method !== 'GET') {
        body = JSON.stringify(request.queryParams);
    }
    const destinationX25519Key = await getDestinationPubKey(request);
    if ((0, FileServerApiV2_1.isOpenGroupV2Request)(request) && request.isAuthRequired && request.room) {
        if (request.forcedTokenToUse) {
            window?.log?.info('sendV2Request. Forcing token to use for room:', request.room);
        }
        const token = request.forcedTokenToUse ||
            (await (0, ApiAuth_1.getAuthToken)({
                roomId: request.room,
                serverUrl: request.server,
            }));
        if (!token) {
            window?.log?.error('Failed to get token for Social group v2');
            return null;
        }
        headers.Authorization = token;
        const res = await (0, onionSend_1.sendViaOnionToNonSnode)(destinationX25519Key, builtUrl, {
            method: request.method,
            headers,
            body,
        }, { noJson: true });
        const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(res);
        if (!statusCode) {
            window?.log?.warn('sendOpenGroupV2Request Got unknown status code; res:', res);
            return res;
        }
        if (statusCode === 401) {
            const roomDetails = await (0, opengroups_1.getV2OpenGroupRoomByRoomId)({
                serverUrl: request.server,
                roomId: request.room,
            });
            if (!roomDetails) {
                window?.log?.warn('Got 401, but this room does not exist');
                return null;
            }
            roomDetails.token = undefined;
            await (0, opengroups_1.saveV2OpenGroupRoom)(roomDetails);
        }
        return res;
    }
    else {
        const res = await (0, onionSend_1.sendViaOnionToNonSnode)(destinationX25519Key, builtUrl, {
            method: request.method,
            headers,
            body,
        });
        return res;
    }
}
exports.sendApiV2Request = sendApiV2Request;
async function openGroupV2GetRoomInfo({ serverUrl, roomId, }) {
    const request = {
        method: 'GET',
        server: serverUrl,
        room: roomId,
        isAuthRequired: false,
        endpoint: `rooms/${roomId}`,
    };
    const result = await exports.sendApiV2Request(request);
    if (result?.result?.room) {
        const { id, name, image_id: imageId } = result?.result?.room;
        if (!id || !name) {
            window?.log?.warn('getRoominfo Parsing failed');
            return null;
        }
        const info = {
            id,
            name,
            imageId,
        };
        return info;
    }
    window?.log?.warn('getInfo failed');
    return null;
}
exports.openGroupV2GetRoomInfo = openGroupV2GetRoomInfo;
const postMessageRetryable = async (message, room) => {
    const ourKeyPair = await utils_1.UserUtils.getIdentityKeyPair();
    const signedMessage = await message.sign(ourKeyPair);
    const json = signedMessage.toJson();
    const request = {
        method: 'POST',
        room: room.roomId,
        server: room.serverUrl,
        queryParams: json,
        isAuthRequired: true,
        endpoint: 'messages',
    };
    const result = await exports.sendApiV2Request(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        throw new Error(`Could not postMessage, status code: ${statusCode}`);
    }
    const rawMessage = result?.result?.message;
    if (!rawMessage) {
        throw new Error('postMessage parsing failed');
    }
    return OpenGroupMessageV2_1.OpenGroupMessageV2.fromJson(rawMessage);
};
exports.postMessageRetryable = postMessageRetryable;
const postMessage = async (message, room) => {
    const result = await (0, p_retry_1.default)(async () => {
        return exports.postMessageRetryable(message, room);
    }, {
        retries: 3,
        factor: 2,
        minTimeout: exports.getMinTimeout(),
        maxTimeout: 4000,
        onFailedAttempt: e => {
            window?.log?.warn(`postMessageRetryable attempt #${e.attemptNumber} failed. ${e.retriesLeft} retries left...`);
        },
    });
    return result;
};
exports.postMessage = postMessage;
const banUser = async (userToBan, roomInfos, deleteAllMessages) => {
    const queryParams = { public_key: userToBan.key };
    const endpoint = deleteAllMessages ? 'ban_and_delete_all' : 'block_list';
    const request = {
        method: 'POST',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        queryParams,
        endpoint,
    };
    const banResult = await exports.sendApiV2Request(request);
    const isOk = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(banResult) === 200;
    return isOk;
};
exports.banUser = banUser;
const unbanUser = async (userToBan, roomInfos) => {
    const request = {
        method: 'DELETE',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        endpoint: `block_list/${userToBan.key}`,
    };
    const unbanResult = await exports.sendApiV2Request(request);
    const isOk = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(unbanResult) === 200;
    return isOk;
};
exports.unbanUser = unbanUser;
const deleteMessageByServerIds = async (idsToRemove, roomInfos) => {
    const request = {
        method: 'POST',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        endpoint: 'delete_messages',
        queryParams: { ids: idsToRemove },
    };
    const messageDeletedResult = await exports.sendApiV2Request(request);
    const isOk = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(messageDeletedResult) === 200;
    return isOk;
};
exports.deleteMessageByServerIds = deleteMessageByServerIds;
const getAllRoomInfos = async (roomInfos) => {
    const request = {
        method: 'GET',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: false,
        endpoint: 'rooms',
        serverPublicKey: roomInfos.serverPublicKey,
    };
    const result = await exports.sendApiV2Request(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        window?.log?.warn('getAllRoomInfos failed invalid status code');
        return;
    }
    return (0, OpenGroupAPIV2Parser_1.parseRooms)(result);
};
exports.getAllRoomInfos = getAllRoomInfos;
const getMemberCount = async (roomInfos) => {
    const request = {
        method: 'GET',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        endpoint: 'member_count',
    };
    const result = await exports.sendApiV2Request(request);
    if ((0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result) !== 200) {
        window?.log?.warn(`getMemberCount failed invalid status code for serverUrl:'${roomInfos.serverUrl}' roomId:'${roomInfos.roomId}; '`, result);
        return;
    }
    const count = (0, OpenGroupAPIV2Parser_1.parseMemberCount)(result);
    if (count === undefined) {
        window?.log?.warn(`getMemberCount failed invalid count for serverUrl:'${roomInfos.serverUrl}' roomId:'${roomInfos.roomId}'`);
        return;
    }
    return count;
};
exports.getMemberCount = getMemberCount;
const downloadFileOpenGroupV2 = async (fileId, roomInfos) => {
    if (!fileId) {
        window?.log?.warn('downloadFileOpenGroupV2: FileId cannot be unset. returning null');
        return null;
    }
    const request = {
        method: 'GET',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        endpoint: `files/${fileId}`,
    };
    const result = await exports.sendApiV2Request(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        return null;
    }
    const base64Data = result?.result?.result;
    if (!base64Data) {
        return null;
    }
    return new Uint8Array(await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', base64Data));
};
exports.downloadFileOpenGroupV2 = downloadFileOpenGroupV2;
const downloadFileOpenGroupV2ByUrl = async (pathName, roomInfos) => {
    const request = {
        method: 'GET',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: false,
        endpoint: pathName,
    };
    const result = await exports.sendApiV2Request(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        return null;
    }
    const base64Data = result?.result?.result;
    if (!base64Data) {
        return null;
    }
    return new Uint8Array(await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', base64Data));
};
exports.downloadFileOpenGroupV2ByUrl = downloadFileOpenGroupV2ByUrl;
const downloadPreviewOpenGroupV2 = async (roomInfos) => {
    const request = {
        method: 'GET',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: false,
        endpoint: `rooms/${roomInfos.roomId}/image`,
        serverPublicKey: roomInfos.serverPublicKey,
    };
    const result = await exports.sendApiV2Request(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        return null;
    }
    const base64Data = result?.result?.result;
    if (!base64Data) {
        return null;
    }
    return base64Data;
};
exports.downloadPreviewOpenGroupV2 = downloadPreviewOpenGroupV2;
const uploadFileOpenGroupV2 = async (fileContent, roomInfos) => {
    if (!fileContent || !fileContent.length) {
        return null;
    }
    const queryParams = {
        file: await (0, util_worker_interface_1.callUtilsWorker)('arrayBufferToStringBase64', fileContent),
    };
    const filesEndpoint = 'files';
    const request = {
        method: 'POST',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        endpoint: filesEndpoint,
        queryParams,
    };
    const result = await exports.sendApiV2Request(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        return null;
    }
    const fileId = result?.result?.result;
    if (!fileId) {
        return null;
    }
    const fileUrl = getCompleteEndpointUrl(roomInfos, `${filesEndpoint}/${fileId}`, false);
    return {
        fileId: fileId,
        fileUrl,
    };
};
exports.uploadFileOpenGroupV2 = uploadFileOpenGroupV2;
const uploadImageForRoomOpenGroupV2 = async (fileContent, roomInfos) => {
    if (!fileContent || !fileContent.length) {
        return null;
    }
    const queryParams = {
        file: await (0, util_worker_interface_1.callUtilsWorker)('arrayBufferToStringBase64', fileContent),
    };
    const imageEndpoint = `rooms/${roomInfos.roomId}/image`;
    const request = {
        method: 'POST',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        endpoint: imageEndpoint,
        queryParams,
    };
    const result = await exports.sendApiV2Request(request);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(result);
    if (statusCode !== 200) {
        return null;
    }
    const fileUrl = getCompleteEndpointUrl(roomInfos, `${imageEndpoint}`, true);
    return {
        fileUrl,
    };
};
exports.uploadImageForRoomOpenGroupV2 = uploadImageForRoomOpenGroupV2;
const addModerator = async (userToAddAsMods, roomInfos) => {
    const request = {
        method: 'POST',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        queryParams: { public_key: userToAddAsMods.key, room_id: roomInfos.roomId },
        endpoint: 'moderators',
    };
    const addModResult = await exports.sendApiV2Request(request);
    const isOk = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(addModResult) === 200;
    return isOk;
};
exports.addModerator = addModerator;
const removeModerator = async (userToAddAsMods, roomInfos) => {
    const request = {
        method: 'DELETE',
        room: roomInfos.roomId,
        server: roomInfos.serverUrl,
        isAuthRequired: true,
        endpoint: `moderators/${userToAddAsMods.key}`,
    };
    const removeModResult = await exports.sendApiV2Request(request);
    const isOk = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(removeModResult) === 200;
    return isOk;
};
exports.removeModerator = removeModerator;
