"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAllMemberCount = exports.getAllBase64AvatarForRooms = exports.compactFetchEverything = void 0;
const opengroups_1 = require("../../../../data/opengroups");
const ApiUtil_1 = require("./ApiUtil");
const OpenGroupAPIV2Parser_1 = require("./OpenGroupAPIV2Parser");
const lodash_1 = __importDefault(require("lodash"));
const onionSend_1 = require("../../../onions/onionSend");
const OpenGroupAPIV2_1 = require("./OpenGroupAPIV2");
const ApiAuth_1 = require("./ApiAuth");
const constants_1 = require("../../../constants");
const COMPACT_POLL_ENDPOINT = 'compact_poll';
const compactFetchEverything = async (serverUrl, rooms, abortSignal) => {
    const compactPollRequest = await getCompactPollRequest(serverUrl, rooms);
    if (!compactPollRequest) {
        window?.log?.info('Nothing found to be fetched. returning');
        return null;
    }
    const result = await sendOpenGroupV2RequestCompactPoll(compactPollRequest, abortSignal);
    return result ? result : null;
};
exports.compactFetchEverything = compactFetchEverything;
const getAllBase64AvatarForRooms = async (serverUrl, rooms, abortSignal) => {
    const allValidRoomInfos = await getAllValidRoomInfos(serverUrl, rooms);
    if (!allValidRoomInfos?.length) {
        window?.log?.info('getAllBase64AvatarForRooms: no valid roominfos got.');
        return null;
    }
    if (abortSignal.aborted) {
        window?.log?.info('preview download aborted, returning null');
        return null;
    }
    const validPreviewBase64 = lodash_1.default.compact(await Promise.all(allValidRoomInfos.map(async (room) => {
        try {
            const base64 = await (0, OpenGroupAPIV2_1.downloadPreviewOpenGroupV2)(room);
            if (base64) {
                return {
                    roomId: room.roomId,
                    base64,
                };
            }
        }
        catch (e) {
            window?.log?.warn('getPreview failed for room', room);
        }
        return null;
    })));
    if (abortSignal.aborted) {
        window?.log?.info('preview download aborted, returning null');
        return null;
    }
    return validPreviewBase64 ? validPreviewBase64 : null;
};
exports.getAllBase64AvatarForRooms = getAllBase64AvatarForRooms;
const getAllMemberCount = async (serverUrl, rooms, abortSignal) => {
    const allValidRoomInfos = await getAllValidRoomInfos(serverUrl, rooms);
    if (!allValidRoomInfos?.length) {
        window?.log?.info('getAllMemberCount: no valid roominfos got.');
        return null;
    }
    if (abortSignal.aborted) {
        window?.log?.info('memberCount aborted, returning null');
        return null;
    }
    const validMemberCount = lodash_1.default.compact(await Promise.all(allValidRoomInfos.map(async (room) => {
        try {
            const memberCount = await (0, OpenGroupAPIV2_1.getMemberCount)(room);
            if (memberCount !== undefined) {
                return {
                    roomId: room.roomId,
                    memberCount,
                };
            }
        }
        catch (e) {
            window?.log?.warn('getPreview failed for room', room);
        }
        return null;
    })));
    if (abortSignal.aborted) {
        window?.log?.info('getMemberCount aborted, returning null');
        return null;
    }
    return validMemberCount ? validMemberCount : null;
};
exports.getAllMemberCount = getAllMemberCount;
const getAllValidRoomInfos = async (serverUrl, rooms) => {
    const allServerPubKeys = [];
    const validRoomInfos = lodash_1.default.compact(await Promise.all([...rooms].map(async (roomId) => {
        try {
            const fetchedInfo = await (0, opengroups_1.getV2OpenGroupRoomByRoomId)({
                serverUrl,
                roomId,
            });
            if (!fetchedInfo) {
                window?.log?.warn('Could not find this room getMessages');
                return null;
            }
            allServerPubKeys.push(fetchedInfo.serverPublicKey);
            const tokenInProgress = await (0, ApiAuth_1.getAuthToken)({ serverUrl, roomId });
            return { ...fetchedInfo, token: tokenInProgress || undefined };
        }
        catch (e) {
            window?.log?.warn('failed to fetch roominfos for room', roomId);
            return null;
        }
    })));
    if (!validRoomInfos?.length) {
        return null;
    }
    let firstPubkey;
    if (allServerPubKeys?.length) {
        firstPubkey = allServerPubKeys[0];
        const allMatch = allServerPubKeys.every(p => p === firstPubkey);
        if (!allMatch) {
            window?.log?.warn('All pubkeys do not match:', allServerPubKeys);
            return null;
        }
    }
    else {
        window?.log?.warn('No pubkeys found:', allServerPubKeys);
        return null;
    }
    return validRoomInfos;
};
const getCompactPollRequest = async (serverUrl, rooms) => {
    const allValidRoomInfos = await getAllValidRoomInfos(serverUrl, rooms);
    if (!allValidRoomInfos?.length) {
        window?.log?.info('compactPoll: no valid roominfos got.');
        return null;
    }
    const roomsRequestInfos = lodash_1.default.compact(allValidRoomInfos.map(validRoomInfos => {
        try {
            const { lastMessageFetchedServerID, lastFetchTimestamp, lastMessageDeletedServerID, token, roomId, } = validRoomInfos;
            const roomRequestContent = {
                room_id: roomId,
                auth_token: token || '',
            };
            roomRequestContent.from_deletion_server_id = lastMessageDeletedServerID;
            if (Date.now() - (lastFetchTimestamp || 0) <= constants_1.DURATION.DAYS * 14) {
                roomRequestContent.from_message_server_id = lastMessageFetchedServerID;
            }
            else {
                window?.log?.info(`We've been away for a long time... Only fetching last messages of room '${roomId}'`);
            }
            return roomRequestContent;
        }
        catch (e) {
            window?.log?.warn('failed to fetch roominfos for room', validRoomInfos.roomId);
            return null;
        }
    }));
    if (!roomsRequestInfos?.length) {
        return null;
    }
    const body = JSON.stringify({
        requests: roomsRequestInfos,
    });
    return {
        body,
        server: serverUrl,
        serverPubKey: allValidRoomInfos[0].serverPublicKey,
        endpoint: COMPACT_POLL_ENDPOINT,
    };
};
async function sendOpenGroupV2RequestCompactPoll(request, abortSignal) {
    const { server: serverUrl, endpoint, body, serverPubKey } = request;
    const builtUrl = new URL(`${serverUrl}/${endpoint}`);
    const res = await (0, onionSend_1.sendViaOnionToNonSnode)(serverPubKey, builtUrl, {
        method: 'POST',
        body,
    }, {}, abortSignal);
    const statusCode = (0, OpenGroupAPIV2Parser_1.parseStatusCodeFromOnionRequest)(res);
    if (!statusCode) {
        window?.log?.warn('sendSocialGroupV2RequestCompactPoll Got unknown status code; res:', res);
        return null;
    }
    const results = await parseCompactPollResults(res, serverUrl);
    if (!results) {
        window?.log?.info('got empty compactPollResults');
        return null;
    }
    const roomWithTokensToRefresh = results.filter(ret => ret.statusCode === 401).map(r => r.roomId);
    const roomPollValidResults = results.filter(ret => ret.statusCode === 200);
    if (roomWithTokensToRefresh?.length) {
        window.log.info('We got those rooms to refresh the token with:', roomWithTokensToRefresh);
        await Promise.all(roomWithTokensToRefresh.map(async (roomId) => {
            const roomDetails = await (0, opengroups_1.getV2OpenGroupRoomByRoomId)({
                serverUrl,
                roomId,
            });
            if (!roomDetails) {
                return;
            }
            roomDetails.token = undefined;
            await (0, opengroups_1.saveV2OpenGroupRoom)(roomDetails);
            await (0, ApiAuth_1.getAuthToken)({ serverUrl, roomId });
        }));
    }
    return roomPollValidResults;
}
const parseCompactPollResult = async (singleRoomResult, _serverUrl) => {
    const { room_id, deletions: rawDeletions, messages: rawMessages, moderators: rawMods, status_code: rawStatusCode, } = singleRoomResult;
    if (!room_id ||
        rawDeletions === undefined ||
        rawMessages === undefined ||
        rawMods === undefined ||
        !rawStatusCode) {
        window?.log?.warn('Invalid compactPoll result', singleRoomResult);
        return null;
    }
    const validMessages = await (0, ApiUtil_1.parseMessages)(rawMessages);
    const moderators = rawMods.sort();
    const deletions = rawDeletions;
    const statusCode = rawStatusCode;
    return {
        roomId: room_id,
        deletions,
        messages: validMessages,
        moderators,
        statusCode,
    };
};
const parseCompactPollResults = async (res, serverUrl) => {
    if (!res || !res.result || !res.result.results || !res.result.results.length) {
        return null;
    }
    const arrayOfResults = res.result.results;
    const parsedResults = lodash_1.default.compact(await Promise.all(arrayOfResults.map(async (m) => {
        return parseCompactPollResult(m, serverUrl);
    })));
    if (!parsedResults || !parsedResults.length) {
        return null;
    }
    return parsedResults;
};
