"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenGroupMessageV2 = void 0;
const curve25519_js_1 = require("curve25519-js");
const util_worker_interface_1 = require("../../../../webworker/workers/util_worker_interface");
const String_1 = require("../../../utils/String");
class OpenGroupMessageV2 {
    serverId;
    sender;
    sentTimestamp;
    base64EncodedData;
    base64EncodedSignature;
    constructor(messageData) {
        const { base64EncodedData, sentTimestamp, base64EncodedSignature, sender, serverId, } = messageData;
        this.base64EncodedData = base64EncodedData;
        this.sentTimestamp = sentTimestamp;
        this.base64EncodedSignature = base64EncodedSignature;
        this.sender = sender;
        this.serverId = serverId;
    }
    static fromJson(json) {
        const { data: base64EncodedData, timestamp: sentTimestamp, server_id: serverId, public_key: sender, signature: base64EncodedSignature, } = json;
        if (!base64EncodedData || !sentTimestamp) {
            window?.log?.info('invalid json to build OpenGroupMessageV2');
            throw new Error('OpengroupV2Message fromJson() failed');
        }
        return new OpenGroupMessageV2({
            base64EncodedData,
            base64EncodedSignature,
            sentTimestamp,
            serverId,
            sender,
        });
    }
    async sign(ourKeyPair) {
        if (!ourKeyPair) {
            window?.log?.warn("Couldn't find user X25519 key pair.");
            throw new Error("Couldn't sign message");
        }
        const data = (0, String_1.fromBase64ToArray)(this.base64EncodedData);
        const signature = (0, curve25519_js_1.sign)(new Uint8Array(ourKeyPair.privKey), data, null);
        if (!signature || signature.length === 0) {
            throw new Error("Couldn't sign message");
        }
        const base64Sig = await (0, util_worker_interface_1.callUtilsWorker)('arrayBufferToStringBase64', signature);
        return new OpenGroupMessageV2({
            base64EncodedData: this.base64EncodedData,
            sentTimestamp: this.sentTimestamp,
            base64EncodedSignature: base64Sig,
            sender: this.sender,
            serverId: this.serverId,
        });
    }
    toJson() {
        const json = {
            data: this.base64EncodedData,
            timestamp: this.sentTimestamp,
        };
        if (this.serverId) {
            json.server_id = this.serverId;
        }
        if (this.sender) {
            json.public_key = this.sender;
        }
        if (this.base64EncodedSignature) {
            json.signature = this.base64EncodedSignature;
        }
        return json;
    }
}
exports.OpenGroupMessageV2 = OpenGroupMessageV2;
