"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenGroupServerPoller = void 0;
const abort_controller_1 = require("abort-controller");
const conversations_1 = require("../../../conversations");
const OpenGroupUtils_1 = require("../utils/OpenGroupUtils");
const OpenGroupAPIV2CompactPoll_1 = require("./OpenGroupAPIV2CompactPoll");
const lodash_1 = __importStar(require("lodash"));
const data_1 = require("../../../../data/data");
const opengroups_1 = require("../../../../data/opengroups");
const auto_bind_1 = __importDefault(require("auto-bind"));
const crypto_1 = require("../../../crypto");
const constants_1 = require("../../../constants");
const MessageAttachment_1 = require("../../../../types/MessageAttachment");
const types_1 = require("../../../../types");
const opengroup_1 = require("../../../../receiver/opengroup");
const util_worker_interface_1 = require("../../../../webworker/workers/util_worker_interface");
const SogsFilterDuplicate_1 = require("./SogsFilterDuplicate");
const pollForEverythingInterval = constants_1.DURATION.SECONDS * 10;
const pollForRoomAvatarInterval = constants_1.DURATION.DAYS * 1;
const pollForMemberCountInterval = constants_1.DURATION.MINUTES * 10;
class OpenGroupServerPoller {
    serverUrl;
    roomIdsToPoll = new Set();
    pollForEverythingTimer;
    pollForRoomAvatarTimer;
    pollForMemberCountTimer;
    abortController;
    isPolling = false;
    isPreviewPolling = false;
    isMemberCountPolling = false;
    wasStopped = false;
    constructor(roomInfos) {
        (0, auto_bind_1.default)(this);
        if (!roomInfos?.length) {
            throw new Error('Empty roomInfos list');
        }
        const firstUrl = roomInfos[0].serverUrl;
        const every = roomInfos.every(r => r.serverUrl === firstUrl);
        if (!every) {
            throw new Error('All rooms must be for the same serverUrl');
        }
        window?.log?.info(`Creating a new OpenGroupServerPoller for url ${firstUrl}`);
        this.serverUrl = firstUrl;
        roomInfos.forEach(r => {
            window?.log?.info(`Adding room on construct for url serverUrl: ${firstUrl}, roomId:'${r.roomId}' to poller:${this.serverUrl}`);
            this.roomIdsToPoll.add(r.roomId);
        });
        this.abortController = new abort_controller_1.AbortController();
        this.pollForEverythingTimer = global.setInterval(this.compactPoll, pollForEverythingInterval);
        this.pollForRoomAvatarTimer = global.setInterval(this.previewPerRoomPoll, pollForRoomAvatarInterval);
        this.pollForMemberCountTimer = global.setInterval(this.pollForAllMemberCount, pollForMemberCountInterval);
        if (this.roomIdsToPoll.size) {
            void this.triggerPollAfterAdd();
        }
    }
    addRoomToPoll(room) {
        if (room.serverUrl !== this.serverUrl) {
            throw new Error('All rooms must be for the same serverUrl');
        }
        if (this.roomIdsToPoll.has(room.roomId)) {
            window?.log?.info('skipping addRoomToPoll of already polled room:', room);
            return;
        }
        window?.log?.info(`Adding room on addRoomToPoll for url serverUrl: ${this.serverUrl}, roomId:'${room.roomId}' to poller:${this.serverUrl}`);
        this.roomIdsToPoll.add(room.roomId);
        void this.triggerPollAfterAdd(room);
    }
    removeRoomFromPoll(room) {
        if (room.serverUrl !== this.serverUrl) {
            window?.log?.info('this is not the correct ServerPoller');
            return;
        }
        if (this.roomIdsToPoll.has(room.roomId)) {
            window?.log?.info(`Removing ${room.roomId} from polling for ${this.serverUrl}`);
            this.roomIdsToPoll.delete(room.roomId);
        }
        else {
            window?.log?.info(`Cannot remove polling of ${room.roomId} as it is not polled on ${this.serverUrl}`);
        }
    }
    getPolledRoomsCount() {
        return this.roomIdsToPoll.size;
    }
    stop() {
        if (this.pollForRoomAvatarTimer) {
            global.clearInterval(this.pollForRoomAvatarTimer);
        }
        if (this.pollForMemberCountTimer) {
            global.clearInterval(this.pollForMemberCountTimer);
        }
        if (this.pollForEverythingTimer) {
            global.clearInterval(this.pollForEverythingTimer);
            this.abortController?.abort();
            this.pollForEverythingTimer = undefined;
            this.pollForRoomAvatarTimer = undefined;
            this.pollForMemberCountTimer = undefined;
            this.wasStopped = true;
        }
    }
    async triggerPollAfterAdd(_room) {
        await this.compactPoll();
        await this.previewPerRoomPoll();
        await this.pollForAllMemberCount();
    }
    shouldPoll() {
        if (this.wasStopped) {
            window?.log?.error('Serverpoller was stopped. CompactPoll should not happen');
            return false;
        }
        if (!this.roomIdsToPoll.size) {
            return false;
        }
        if (this.isPolling) {
            return false;
        }
        if (!window.getGlobalOnlineStatus()) {
            return false;
        }
        return true;
    }
    shouldPollPreview() {
        if (this.wasStopped) {
            window?.log?.error('Serverpoller was stopped. PollPreview should not happen');
            return false;
        }
        if (!this.roomIdsToPoll.size) {
            return false;
        }
        if (this.isPreviewPolling) {
            return false;
        }
        if (!window.getGlobalOnlineStatus()) {
            return false;
        }
        return true;
    }
    shouldPollForMemberCount() {
        if (this.wasStopped) {
            window?.log?.error('Serverpoller was stopped. PolLForMemberCount should not happen');
            return false;
        }
        if (!this.roomIdsToPoll.size) {
            return false;
        }
        if (this.isMemberCountPolling) {
            return false;
        }
        if (!window.getGlobalOnlineStatus()) {
            return false;
        }
        return true;
    }
    async previewPerRoomPoll() {
        if (!this.shouldPollPreview()) {
            return;
        }
        try {
            this.isPreviewPolling = true;
            if (this.abortController.signal.aborted) {
                throw new Error('Poller aborted');
            }
            let previewGotResults = await (0, OpenGroupAPIV2CompactPoll_1.getAllBase64AvatarForRooms)(this.serverUrl, this.roomIdsToPoll, this.abortController.signal);
            if (this.abortController.signal.aborted) {
                throw new Error('Abort controller was canceled. Dropping preview request');
            }
            if (!previewGotResults) {
                throw new Error('getPreview: no results');
            }
            previewGotResults = previewGotResults.filter(result => this.roomIdsToPoll.has(result.roomId));
            await handleBase64AvatarUpdate(this.serverUrl, previewGotResults);
        }
        catch (e) {
            window?.log?.warn('Got error while preview fetch:', e);
        }
        finally {
            this.isPreviewPolling = false;
        }
    }
    async pollForAllMemberCount() {
        if (!this.shouldPollForMemberCount()) {
            return;
        }
        try {
            this.isMemberCountPolling = true;
            if (this.abortController.signal.aborted) {
                throw new Error('Poller aborted');
            }
            let memberCountGotResults = await (0, OpenGroupAPIV2CompactPoll_1.getAllMemberCount)(this.serverUrl, this.roomIdsToPoll, this.abortController.signal);
            if (this.abortController.signal.aborted) {
                throw new Error('Abort controller was canceled. Dropping memberCount request');
            }
            if (!memberCountGotResults) {
                throw new Error('MemberCount: no results');
            }
            memberCountGotResults = memberCountGotResults.filter(result => this.roomIdsToPoll.has(result.roomId));
            await handleAllMemberCount(this.serverUrl, memberCountGotResults);
        }
        catch (e) {
            window?.log?.warn('Got error while memberCount fetch:', e);
        }
        finally {
            this.isMemberCountPolling = false;
        }
    }
    async compactPoll() {
        if (!this.shouldPoll()) {
            return;
        }
        try {
            this.isPolling = true;
            if (this.abortController.signal.aborted) {
                throw new Error('Poller aborted');
            }
            let compactFetchResults = await (0, OpenGroupAPIV2CompactPoll_1.compactFetchEverything)(this.serverUrl, this.roomIdsToPoll, this.abortController.signal);
            if (this.abortController.signal.aborted) {
                throw new Error('Abort controller was canceled. dropping request');
            }
            if (!compactFetchResults) {
                throw new Error('compactFetch: no results');
            }
            compactFetchResults = compactFetchResults.filter(result => this.roomIdsToPoll.has(result.roomId));
            await handleCompactPollResults(this.serverUrl, compactFetchResults);
        }
        catch (e) {
            window?.log?.warn('Got error while compact fetch:', e.message);
        }
        finally {
            this.isPolling = false;
        }
    }
}
exports.OpenGroupServerPoller = OpenGroupServerPoller;
const handleDeletions = async (deleted, conversationId, convo) => {
    const allIdsRemoved = (deleted || []).map(d => d.deleted_message_id);
    const allRowIds = (deleted || []).map(d => d.id);
    const maxDeletedId = Math.max(...allRowIds);
    try {
        const messageIds = await (0, data_1.getMessageIdsFromServerIds)(allIdsRemoved, conversationId);
        await Promise.all((messageIds || []).map(async (id) => {
            if (convo) {
                await convo.removeMessage(id);
            }
            await (0, data_1.removeMessage)(id);
        }));
    }
    catch (e) {
        window?.log?.warn('handleDeletions failed:', e);
    }
    finally {
        try {
            const roomInfos = await (0, opengroups_1.getV2OpenGroupRoom)(conversationId);
            if (roomInfos && roomInfos.lastMessageDeletedServerID !== maxDeletedId) {
                roomInfos.lastMessageDeletedServerID = maxDeletedId;
                await (0, opengroups_1.saveV2OpenGroupRoom)(roomInfos);
            }
        }
        catch (e) {
            window?.log?.warn('handleDeletions updating roomInfos failed:', e);
        }
    }
};
const handleNewMessages = async (newMessages, conversationId, _convo) => {
    try {
        const roomInfos = await (0, opengroups_1.getV2OpenGroupRoom)(conversationId);
        if (!roomInfos || !roomInfos.serverUrl || !roomInfos.roomId) {
            throw new Error(`No room for convo ${conversationId}`);
        }
        if (!newMessages.length) {
            roomInfos.lastFetchTimestamp = Date.now();
            window?.log?.info(`No new messages for ${roomInfos.roomId}... just updating our last fetched timestamp`);
            await (0, opengroups_1.saveV2OpenGroupRoom)(roomInfos);
            return;
        }
        const incomingMessageIds = lodash_1.default.compact(newMessages.map(n => n.serverId));
        const maxNewMessageId = Math.max(...incomingMessageIds);
        const roomDetails = lodash_1.default.pick(roomInfos, 'serverUrl', 'roomId');
        const filteredDuplicates = await (0, SogsFilterDuplicate_1.filterDuplicatesFromDbAndIncoming)(newMessages);
        const startHandleOpengroupMessage = (0, lodash_1.now)();
        for (let index = 0; index < filteredDuplicates.length; index++) {
            const newMessage = filteredDuplicates[index];
            try {
                await (0, opengroup_1.handleOpenGroupV2Message)(newMessage, roomDetails);
            }
            catch (e) {
                window?.log?.warn('handleOpenGroupV2Message', e);
            }
        }
        window.log.debug(`[perf] handle ${filteredDuplicates.length} opengroupMessages took ${(0, lodash_1.now)() -
            startHandleOpengroupMessage}ms.`);
        if (roomInfos) {
            roomInfos.lastMessageFetchedServerID = maxNewMessageId;
            roomInfos.lastFetchTimestamp = Date.now();
            await (0, opengroups_1.saveV2OpenGroupRoom)(roomInfos);
        }
    }
    catch (e) {
        window?.log?.warn('handleNewMessages failed:', e);
    }
};
const handleCompactPollResults = async (serverUrl, results) => {
    await Promise.all(results.map(async (res) => {
        const convoId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(serverUrl, res.roomId);
        const convo = (0, conversations_1.getConversationController)().get(convoId);
        if (res.deletions.length) {
            await handleDeletions(res.deletions, convoId, convo);
        }
        await handleNewMessages(res.messages, convoId, convo);
        if (!convo) {
            window?.log?.warn('Could not find convo for compactPoll', convoId);
            return;
        }
        await convo.updateGroupAdmins(res.moderators);
    }));
};
const handleBase64AvatarUpdate = async (serverUrl, avatarResults) => {
    await Promise.all(avatarResults.map(async (res) => {
        const convoId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(serverUrl, res.roomId);
        const convo = (0, conversations_1.getConversationController)().get(convoId);
        if (!convo) {
            window?.log?.warn('Could not find convo for compactPoll', convoId);
            return;
        }
        if (!res.base64) {
            window?.log?.info('getPreview: no base64 data. skipping');
            return;
        }
        const existingHash = convo.get('avatarHash');
        const newHash = (0, crypto_1.sha256)(res.base64);
        if (newHash !== existingHash) {
            const upgradedAttachment = await (0, MessageAttachment_1.processNewAttachment)({
                isRaw: true,
                data: await (0, util_worker_interface_1.callUtilsWorker)('fromBase64ToArrayBuffer', res.base64),
                contentType: types_1.MIME.IMAGE_UNKNOWN,
            });
            await convo.setBchatProfile({
                displayName: convo.getName() || window.i18n('unknown'),
                avatar: upgradedAttachment.path,
                avatarHash: newHash,
            });
            convo.set({
                avatarHash: newHash,
            });
            await convo.commit();
        }
    }));
};
async function handleAllMemberCount(serverUrl, memberCountGotResults) {
    if (!memberCountGotResults.length) {
        return;
    }
    await Promise.all(memberCountGotResults.map(async (roomCount) => {
        const conversationId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(serverUrl, roomCount.roomId);
        const convo = (0, conversations_1.getConversationController)().get(conversationId);
        if (!convo) {
            window?.log?.warn('cannot update conversation memberCount as it does not exist');
            return;
        }
        if (convo.get('subscriberCount') !== roomCount.memberCount) {
            convo.set({ subscriberCount: roomCount.memberCount });
            await convo.commit();
        }
    }));
}
