"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMessageQueue = exports.MessageQueue = void 0;
const PendingMessageCache_1 = require("./PendingMessageCache");
const utils_1 = require("../utils");
const types_1 = require("../types");
const _1 = require(".");
const ClosedGroupMessage_1 = require("../messages/outgoing/controlMessage/group/ClosedGroupMessage");
const ConfigurationMessage_1 = require("../messages/outgoing/controlMessage/ConfigurationMessage");
const MessageSentHandler_1 = require("./MessageSentHandler");
const ExpirationTimerUpdateMessage_1 = require("../messages/outgoing/controlMessage/ExpirationTimerUpdateMessage");
const ClosedGroupNewMessage_1 = require("../messages/outgoing/controlMessage/group/ClosedGroupNewMessage");
const UnsendMessage_1 = require("../messages/outgoing/controlMessage/UnsendMessage");
class MessageQueue {
    jobQueues = new Map();
    pendingMessageCache;
    constructor(cache) {
        this.pendingMessageCache = cache ?? new PendingMessageCache_1.PendingMessageCache();
        void this.processAllPending();
    }
    async sendToPubKey(destinationPubKey, message, sentCb, isGroup = false) {
        if (message instanceof ConfigurationMessage_1.ConfigurationMessage || !!message.syncTarget) {
            throw new Error('SyncMessage needs to be sent with sendSyncMessage');
        }
        await this.process(destinationPubKey, message, sentCb, isGroup);
    }
    async sendToOpenGroupV2(message, roomInfos) {
        const error = new Error('Failed to send message to Social group.');
        try {
            const { sentTimestamp, serverId } = await _1.MessageSender.sendToOpenGroupV2(message, roomInfos);
            if (!serverId || serverId === -1) {
                throw new Error(`Invalid serverId returned by server: ${serverId}`);
            }
            if (message && message.dataProto().reaction) {
                return;
            }
            void MessageSentHandler_1.MessageSentHandler.handlePublicMessageSentSuccess(message, {
                serverId: serverId,
                serverTimestamp: sentTimestamp,
            });
        }
        catch (e) {
            window?.log?.warn(`Failed to send message to Social group: ${roomInfos}`, e);
            void MessageSentHandler_1.MessageSentHandler.handleMessageSentFailure(message, e || error);
        }
    }
    async sendToGroup(message, sentCb, groupPubKey) {
        let destinationPubKey = groupPubKey;
        if (message instanceof ExpirationTimerUpdateMessage_1.ExpirationTimerUpdateMessage || message instanceof ClosedGroupMessage_1.ClosedGroupMessage) {
            destinationPubKey = groupPubKey ? groupPubKey : message.groupId;
        }
        if (!destinationPubKey) {
            throw new Error('Invalid group message passed in sendToGroup.');
        }
        return this.sendToPubKey(types_1.PubKey.cast(destinationPubKey), message, sentCb, true);
    }
    async sendSyncMessage(message, sentCb) {
        if (!message) {
            return;
        }
        if (!(message instanceof ConfigurationMessage_1.ConfigurationMessage) &&
            !(message instanceof UnsendMessage_1.UnsendMessage) &&
            !message?.syncTarget) {
            throw new Error('Invalid message given to sendSyncMessage');
        }
        const ourPubKey = utils_1.UserUtils.getOurPubKeyStrFromCache();
        await this.process(types_1.PubKey.cast(ourPubKey), message, sentCb);
    }
    async sendToPubKeyNonDurably(user, message) {
        let rawMessage;
        try {
            rawMessage = await utils_1.MessageUtils.toRawMessage(user, message);
            const { wrappedEnvelope, effectiveTimestamp } = await _1.MessageSender.send(rawMessage);
            await MessageSentHandler_1.MessageSentHandler.handleMessageSentSuccess(rawMessage, effectiveTimestamp, wrappedEnvelope);
            return effectiveTimestamp;
        }
        catch (error) {
            if (rawMessage) {
                await MessageSentHandler_1.MessageSentHandler.handleMessageSentFailure(rawMessage, error);
            }
            return false;
        }
    }
    async processPending(device, isSyncMessage = false) {
        const messages = await this.pendingMessageCache.getForDevice(device);
        const jobQueue = this.getJobQueue(device);
        messages.forEach(async (message) => {
            const messageId = message.identifier;
            if (!jobQueue.has(messageId)) {
                const job = async () => {
                    try {
                        const { wrappedEnvelope, effectiveTimestamp } = await _1.MessageSender.send(message, undefined, undefined, isSyncMessage);
                        await MessageSentHandler_1.MessageSentHandler.handleMessageSentSuccess(message, effectiveTimestamp, wrappedEnvelope);
                        const cb = this.pendingMessageCache.callbacks.get(message.identifier);
                        if (cb) {
                            await cb(message);
                        }
                        this.pendingMessageCache.callbacks.delete(message.identifier);
                    }
                    catch (error) {
                        void MessageSentHandler_1.MessageSentHandler.handleMessageSentFailure(message, error);
                    }
                    finally {
                        void this.pendingMessageCache.remove(message);
                    }
                };
                await jobQueue.addWithId(messageId, job);
            }
        });
    }
    async processAllPending() {
        const devices = await this.pendingMessageCache.getDevices();
        const promises = devices.map(async (device) => this.processPending(device));
        return Promise.all(promises);
    }
    async process(destinationPk, message, sentCb, isGroup = false) {
        const currentDevice = utils_1.UserUtils.getOurPubKeyFromCache();
        let isSyncMessage = false;
        if (currentDevice && destinationPk.isEqual(currentDevice)) {
            if (message instanceof ConfigurationMessage_1.ConfigurationMessage ||
                message instanceof ClosedGroupNewMessage_1.ClosedGroupNewMessage ||
                message instanceof UnsendMessage_1.UnsendMessage ||
                message.syncTarget?.length > 0) {
                window?.log?.warn('Processing sync message');
                isSyncMessage = true;
            }
            else {
                window?.log?.warn('Dropping message in process() to be sent to ourself');
                return;
            }
        }
        await this.pendingMessageCache.add(destinationPk, message, sentCb, isGroup);
        void this.processPending(destinationPk, isSyncMessage);
    }
    getJobQueue(device) {
        let queue = this.jobQueues.get(device.key);
        if (!queue) {
            queue = new utils_1.JobQueue();
            this.jobQueues.set(device.key, queue);
        }
        return queue;
    }
}
exports.MessageQueue = MessageQueue;
let messageQueue;
function getMessageQueue() {
    if (!messageQueue) {
        messageQueue = new MessageQueue();
    }
    return messageQueue;
}
exports.getMessageQueue = getMessageQueue;
