"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BchatConversation = void 0;
const react_1 = __importDefault(require("react"));
const lodash_1 = __importDefault(require("lodash"));
const classnames_1 = __importDefault(require("classnames"));
const auto_bind_1 = __importDefault(require("auto-bind"));
const CompositionBox_1 = require("./composition/CompositionBox");
const Performance_1 = require("../../bchat/utils/Performance");
const DEFAULT_JPEG_QUALITY = 0.85;
const BchatMessagesListContainer_1 = require("./BchatMessagesListContainer");
const BchatFileDropzone_1 = require("./BchatFileDropzone");
const InConversationCallContainer_1 = require("../calling/InConversationCallContainer");
const LightboxGallery_1 = require("../lightbox/LightboxGallery");
const data_1 = require("../../data/data");
const conversations_1 = require("../../bchat/conversations");
const utils_1 = require("../../bchat/utils");
const conversations_2 = require("../../state/ducks/conversations");
const modalDialog_1 = require("../../state/ducks/modalDialog");
const BchatTheme_1 = require("../../state/ducks/BchatTheme");
const stagedAttachments_1 = require("../../state/ducks/stagedAttachments");
const types_1 = require("../../types");
const util_1 = require("../../util");
const BchatButton_1 = require("../basic/BchatButton");
const MainViewController_1 = require("../MainViewController");
const ConversationHeader_1 = require("./ConversationHeader");
const VisualAttachment_1 = require("../../types/attachments/VisualAttachment");
const blob_util_1 = require("blob-util");
const constants_1 = require("../../bchat/constants");
const ConversationRequestInfo_1 = require("./ConversationRequestInfo");
const storage_1 = require("../../util/storage");
const blueimp_load_image_1 = __importDefault(require("blueimp-load-image"));
const section_1 = require("../../state/ducks/section");
const BchatScrollButton_1 = require("../BchatScrollButton");
const Flex_1 = require("../basic/Flex");
const icon_1 = require("../icon");
const styled_components_1 = __importDefault(require("styled-components"));
const ReactListModal_1 = require("../dialog/ReactListModal");
const BchatProfileInfo_1 = require("../BchatProfileInfo");
class BchatConversation extends react_1.default.Component {
    messageContainerRef;
    dragCounter;
    publicMembersRefreshTimeout;
    updateMemberList;
    constructor(props) {
        super(props);
        this.state = {
            isDraggingFile: false,
        };
        this.messageContainerRef = react_1.default.createRef();
        this.dragCounter = 0;
        this.updateMemberList = lodash_1.default.debounce(this.updateMemberListBouncy.bind(this), 10000);
        (0, auto_bind_1.default)(this);
    }
    componentDidUpdate(prevProps, _prevState) {
        const { selectedConversationKey: newConversationKey, selectedConversation: newConversation, } = this.props;
        const { selectedConversationKey: oldConversationKey } = prevProps;
        if (newConversationKey && newConversation && newConversationKey !== oldConversationKey) {
            setTimeout(() => {
                const div = this.messageContainerRef.current;
                div?.addEventListener('dragenter', this.handleDragIn);
                div?.addEventListener('dragleave', this.handleDragOut);
                div?.addEventListener('dragover', this.handleDrag);
                div?.addEventListener('drop', this.handleDrop);
            }, 100);
            if (this.publicMembersRefreshTimeout) {
                global.clearInterval(this.publicMembersRefreshTimeout);
                this.publicMembersRefreshTimeout = undefined;
            }
            if (newConversation.isPublic) {
                void this.updateMemberListBouncy();
                this.publicMembersRefreshTimeout = global.setInterval(this.updateMemberList, 60000);
            }
        }
        if (!newConversation) {
            const div = this.messageContainerRef.current;
            div?.removeEventListener('dragenter', this.handleDragIn);
            div?.removeEventListener('dragleave', this.handleDragOut);
            div?.removeEventListener('dragover', this.handleDrag);
            div?.removeEventListener('drop', this.handleDrop);
            if (this.publicMembersRefreshTimeout) {
                global.clearInterval(this.publicMembersRefreshTimeout);
                this.publicMembersRefreshTimeout = undefined;
            }
        }
        if (newConversationKey !== oldConversationKey) {
            this.setState({
                isDraggingFile: false,
            });
        }
    }
    componentWillUnmount() {
        const div = this.messageContainerRef.current;
        div?.removeEventListener('dragenter', this.handleDragIn);
        div?.removeEventListener('dragleave', this.handleDragOut);
        div?.removeEventListener('dragover', this.handleDrag);
        div?.removeEventListener('drop', this.handleDrop);
        if (this.publicMembersRefreshTimeout) {
            global.clearInterval(this.publicMembersRefreshTimeout);
            this.publicMembersRefreshTimeout = undefined;
        }
    }
    sendMessageFn(msg) {
        const { selectedConversationKey } = this.props;
        const conversationModel = (0, conversations_1.getConversationController)().get(selectedConversationKey);
        if (!conversationModel) {
            return;
        }
        const sendAndScroll = async () => {
            void conversationModel.sendMessage(msg);
            await this.scrollToNow();
        };
        const recoveryPhrase = (0, storage_1.getCurrentRecoveryPhrase)();
        if (msg.body.replace(/\s/g, '').includes(recoveryPhrase.replace(/\s/g, ''))) {
            window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)({
                title: 'Warning',
                message: 'This is your recovery phrase. if you send it to someone they will have full access to your account.',
                okTheme: BchatButton_1.BchatButtonColor.Danger,
                okText: window.i18n('send'),
                iconShow: true,
                customIcon: (react_1.default.createElement(icon_1.BchatIcon, { iconType: "warningCircle", iconSize: 30, iconColor: "#F0AF13", clipRule: "evenodd", fillRule: "evenodd" })),
                onClickOk: () => {
                    void sendAndScroll();
                },
                onClickClose: () => {
                    window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)(null));
                },
            }));
        }
        else {
            void sendAndScroll();
        }
        window.inboxStore?.dispatch((0, conversations_2.quoteMessage)(undefined));
    }
    render() {
        const { isDraggingFile } = this.state;
        const { selectedConversation, messagesProps, selectedMessages, lightBoxOptions, convoList, focusedSection, reactListModalstate, } = this.props;
        const selectionMode = selectedMessages.length > 0;
        if (convoList?.conversations?.length == 0 &&
            (!selectedConversation || !messagesProps) &&
            focusedSection !== section_1.SectionType.Opengroup &&
            focusedSection !== section_1.SectionType.NewChat) {
            return react_1.default.createElement(MainViewController_1.AddNewContactInEmptyConvo, null);
        }
        if (!selectedConversation || !messagesProps) {
            return react_1.default.createElement(MainViewController_1.MessageView, null);
        }
        return (react_1.default.createElement(BchatTheme_1.BchatTheme, null,
            reactListModalstate && react_1.default.createElement(ReactListModal_1.ReactListModal, { ...reactListModalstate }),
            react_1.default.createElement("div", { className: "conversation-header" },
                react_1.default.createElement(ConversationHeader_1.ConversationHeaderWithDetails, null)),
            selectionMode && (react_1.default.createElement("div", { className: "conversation-header" },
                react_1.default.createElement(ConversationHeader_1.SelectionOverlay, null))),
            react_1.default.createElement("div", { className: (0, classnames_1.default)('conversation-content', selectionMode && 'selection-mode'), tabIndex: 0, onKeyDown: this.onKeyDown, role: "navigation" },
                selectedConversation?.isPublic && (react_1.default.createElement("div", { className: "pinned-msg" },
                    react_1.default.createElement(Flex_1.Flex, { container: true, alignItems: "center", justifyContent: "space-between" },
                        react_1.default.createElement(Flex_1.Flex, { container: true, alignItems: "center" },
                            react_1.default.createElement(VerticalLine, null),
                            react_1.default.createElement("div", null,
                                react_1.default.createElement("div", { className: "msg-title" }, "Pinned Message"),
                                react_1.default.createElement("div", { className: "msg-sub-title" }, "Community guidelines"))),
                        react_1.default.createElement(BchatButton_1.BchatButton, { buttonType: BchatButton_1.BchatButtonType.Brand, buttonColor: BchatButton_1.BchatButtonColor.Secondary, style: {
                                minWidth: '106px',
                                height: '31px',
                                fontSize: '14px',
                                padding: '0 0px',
                                borderRadius: '6px',
                            }, text: "Read More", onClick: () => window.inboxStore?.dispatch((0, modalDialog_1.updateCommunityGuidelinesModal)({})) })))),
                lightBoxOptions?.media && this.renderLightBox(lightBoxOptions),
                react_1.default.createElement("div", { className: "conversation-messages" },
                    this.props.hasOngoingCallWithFocusedConvo && (react_1.default.createElement(Flex_1.Flex, { container: true, justifyContent: "center", alignItems: "center" },
                        react_1.default.createElement(InConversationCallContainer_1.InConversationCallContainer, null))),
                    react_1.default.createElement(BchatMessagesListContainer_1.BchatMessagesListContainer, { messageContainerRef: this.messageContainerRef, scrollToNow: this.scrollToNow }),
                    isDraggingFile && react_1.default.createElement(BchatFileDropzone_1.BchatFileDropzone, null)),
                react_1.default.createElement(ConversationRequestInfo_1.ConversationRequestinfo, null),
                react_1.default.createElement(BchatScrollButton_1.BchatScrollButton, { onClickScrollBottom: this.scrollToNow, key: "scroll-down-button", unreadCount: selectedConversation.unreadCount }),
                react_1.default.createElement(CompositionBox_1.CompositionBox, { sendMessage: this.sendMessageFn, stagedAttachments: this.props.stagedAttachments, onChoseAttachments: this.onChoseAttachments })),
            react_1.default.createElement("div", { className: "profile-info" },
                react_1.default.createElement(BchatProfileInfo_1.ProfileInfo, { sendMessage: this.sendMessageFn }))));
    }
    async scrollToNow() {
        if (!this.props.selectedConversationKey) {
            return;
        }
        const mostNowMessage = await (0, data_1.getLastMessageInConversation)(this.props.selectedConversationKey);
        if (mostNowMessage) {
            await (0, conversations_2.openConversationToSpecificMessage)({
                conversationKey: this.props.selectedConversationKey,
                messageIdToNavigateTo: mostNowMessage.id,
                shouldHighlightMessage: false,
            });
            const messageContainer = this.messageContainerRef.current;
            if (!messageContainer) {
                return;
            }
            messageContainer.scrollTop = messageContainer.scrollHeight - messageContainer.clientHeight;
        }
    }
    onKeyDown(event) {
        const selectionMode = !!this.props.selectedMessages.length;
        if (event.target.classList.contains('conversation-content')) {
            switch (event.key) {
                case 'Escape':
                    if (selectionMode) {
                        window.inboxStore?.dispatch((0, conversations_2.resetSelectedMessageIds)());
                    }
                    break;
                default:
            }
        }
    }
    renderLightBox({ media, attachment }) {
        const selectedIndex = media.length > 1
            ? media.findIndex(mediaMessage => mediaMessage.attachment.path === attachment.path)
            : 0;
        return react_1.default.createElement(LightboxGallery_1.LightboxGallery, { media: media, selectedIndex: selectedIndex });
    }
    async onChoseAttachments(attachmentsFileList) {
        if (!attachmentsFileList || attachmentsFileList.length === 0) {
            return;
        }
        for (let i = 0; i < attachmentsFileList.length; i++) {
            await this.maybeAddAttachment(attachmentsFileList[i]);
        }
    }
    async maybeAddAttachment(file) {
        if (!file) {
            return;
        }
        const fileName = file.name;
        const contentType = file.type;
        const { stagedAttachments } = this.props;
        if (stagedAttachments.length >= 32) {
            utils_1.ToastUtils.pushMaximumAttachmentsError();
            return;
        }
        const haveNonImage = lodash_1.default.some(stagedAttachments, attachment => !types_1.MIME.isImage(attachment.contentType));
        if (haveNonImage) {
            utils_1.ToastUtils.pushMultipleNonImageError();
            return;
        }
        if (!types_1.MIME.isImage(contentType) && stagedAttachments.length > 0) {
            utils_1.ToastUtils.pushCannotMixError();
            return;
        }
        let blob = null;
        try {
            blob = await util_1.AttachmentUtil.autoScale({
                contentType,
                blob: file,
            });
            if (blob.blob.size >= constants_1.MAX_ATTACHMENT_FILESIZE_BYTES) {
                utils_1.ToastUtils.pushFileSizeErrorAsByte(constants_1.MAX_ATTACHMENT_FILESIZE_BYTES);
                return;
            }
        }
        catch (error) {
            window?.log?.error('Error ensuring that image is properly sized:', error && error.stack ? error.stack : error);
            utils_1.ToastUtils.pushLoadAttachmentFailure(error?.message);
            return;
        }
        try {
            if (util_1.GoogleChrome.isImageTypeSupported(contentType)) {
                const attachmentWithPreview = await renderImagePreview(contentType, file, fileName);
                this.addAttachments([attachmentWithPreview]);
            }
            else if (util_1.GoogleChrome.isVideoTypeSupported(contentType)) {
                const attachmentWithVideoPreview = await renderVideoPreview(contentType, file, fileName);
                this.addAttachments([attachmentWithVideoPreview]);
            }
            else {
                this.addAttachments([
                    {
                        file,
                        size: file.size,
                        contentType,
                        fileName,
                        url: '',
                        isVoiceMessage: false,
                        fileSize: null,
                        screenshot: null,
                        thumbnail: null,
                    },
                ]);
            }
        }
        catch (e) {
            window?.log?.error(`Was unable to generate thumbnail for file type ${contentType}`, e && e.stack ? e.stack : e);
            this.addAttachments([
                {
                    file,
                    size: file.size,
                    contentType,
                    fileName,
                    isVoiceMessage: false,
                    url: '',
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                },
            ]);
        }
    }
    addAttachments(newAttachments) {
        window.inboxStore?.dispatch((0, stagedAttachments_1.addStagedAttachmentsInConversation)({
            conversationKey: this.props.selectedConversationKey,
            newAttachments,
        }));
    }
    handleDrag(e) {
        e.preventDefault();
        e.stopPropagation();
    }
    handleDragIn(e) {
        e.preventDefault();
        e.stopPropagation();
        this.dragCounter++;
        if (e.dataTransfer.items && e.dataTransfer.items.length > 0) {
            this.setState({ isDraggingFile: true });
        }
    }
    handleDragOut(e) {
        e.preventDefault();
        e.stopPropagation();
        this.dragCounter--;
        if (this.dragCounter === 0) {
            this.setState({ isDraggingFile: false });
        }
    }
    handleDrop(e) {
        e.preventDefault();
        e.stopPropagation();
        if (e?.dataTransfer?.files && e.dataTransfer.files.length > 0) {
            void this.onChoseAttachments(Array.from(e.dataTransfer.files));
            e.dataTransfer.clearData();
            this.dragCounter = 0;
            this.setState({ isDraggingFile: false });
        }
    }
    async updateMemberListBouncy() {
        const start = Date.now();
        const allPubKeys = await (0, data_1.getPubkeysInPublicConversation)(this.props.selectedConversationKey);
        window?.log?.debug(`[perf] getPubkeysInPublicConversation returned '${allPubKeys?.length}' members in ${Date.now() - start}ms`);
        const allMembers = allPubKeys.map((pubKey) => {
            const conv = (0, conversations_1.getConversationController)().get(pubKey);
            const profileName = conv?.getProfileName() || 'Anonymous';
            return {
                id: pubKey,
                authorProfileName: profileName,
            };
        });
        window.inboxStore?.dispatch((0, conversations_2.updateMentionsMembers)(allMembers));
    }
}
exports.BchatConversation = BchatConversation;
const renderVideoPreview = async (contentType, file, fileName) => {
    const objectUrl = URL.createObjectURL(file);
    try {
        const type = VisualAttachment_1.THUMBNAIL_CONTENT_TYPE;
        const thumbnail = await (0, VisualAttachment_1.makeVideoScreenshot)({
            objectUrl,
            contentType: type,
        });
        const data = await (0, blob_util_1.blobToArrayBuffer)(thumbnail);
        const url = (0, util_1.arrayBufferToObjectURL)({
            data,
            type,
        });
        return {
            file,
            size: file.size,
            fileName,
            contentType,
            videoUrl: objectUrl,
            url,
            isVoiceMessage: false,
            fileSize: null,
            screenshot: null,
            thumbnail: null,
        };
    }
    catch (error) {
        URL.revokeObjectURL(objectUrl);
        throw error;
    }
};
const autoOrientJpegImage = async (fileOrBlobOrURL) => {
    (0, Performance_1.perfStart)('autoOrientJpegImage');
    const loadedImage = await (0, blueimp_load_image_1.default)(fileOrBlobOrURL, { orientation: true, canvas: true });
    (0, Performance_1.perfEnd)('autoOrientJpegImage', 'autoOrientJpegImage');
    const dataURL = loadedImage.image.toDataURL(types_1.MIME.IMAGE_JPEG, DEFAULT_JPEG_QUALITY);
    return dataURL;
};
const renderImagePreview = async (contentType, file, fileName) => {
    if (!types_1.MIME.isJPEG(contentType)) {
        const urlImage = URL.createObjectURL(file);
        if (!urlImage) {
            throw new Error('Failed to create object url for image!');
        }
        return {
            file,
            size: file.size,
            fileName,
            contentType,
            url: urlImage,
            isVoiceMessage: false,
            fileSize: null,
            screenshot: null,
            thumbnail: null,
        };
    }
    const orientedImageUrl = await autoOrientJpegImage(file);
    const thumbnailBuffer = await (0, VisualAttachment_1.makeImageThumbnailBuffer)({
        objectUrl: orientedImageUrl,
        contentType,
    });
    const url = (0, util_1.arrayBufferToObjectURL)({
        data: thumbnailBuffer,
        type: VisualAttachment_1.THUMBNAIL_CONTENT_TYPE,
    });
    return {
        file,
        size: file.size,
        fileName,
        contentType,
        url,
        isVoiceMessage: false,
        fileSize: null,
        screenshot: null,
        thumbnail: null,
    };
};
const VerticalLine = styled_components_1.default.div `
  width: 5px;
  background-color: var(--color-untrusted-vertical-bar);
  height: 38px;
  border-radius: 10px;
  margin-right: 10px;
`;
