"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdateGroupMembersDialog = void 0;
const react_1 = __importDefault(require("react"));
const utils_1 = require("../../bchat/utils");
const lodash_1 = __importDefault(require("lodash"));
const Text_1 = require("../basic/Text");
const modalDialog_1 = require("../../state/ducks/modalDialog");
const BchatButton_1 = require("../basic/BchatButton");
const MemberListItem_1 = require("../MemberListItem");
const BchatWrapperModal_1 = require("../BchatWrapperModal");
const react_redux_1 = require("react-redux");
const useParamSelector_1 = require("../../hooks/useParamSelector");
const useKey_1 = __importDefault(require("react-use/lib/useKey"));
const useSet_1 = require("../../hooks/useSet");
const conversations_1 = require("../../bchat/conversations");
const closed_group_1 = require("../../bchat/group/closed-group");
const ClassicMemberList = (props) => {
    const { onSelect, convoId, onUnselect, selectedMembers } = props;
    const weAreAdmin = (0, useParamSelector_1.useWeAreAdmin)(convoId);
    const convoProps = (0, useParamSelector_1.useConversationPropsById)(convoId);
    if (!convoProps) {
        throw new Error('MemberList needs convoProps');
    }
    let currentMembers = convoProps.members || [];
    const { groupAdmins } = convoProps;
    currentMembers = [...currentMembers].sort(m => (groupAdmins?.includes(m) ? -1 : 0));
    return (react_1.default.createElement(react_1.default.Fragment, null, currentMembers.map(member => {
        const isSelected = (weAreAdmin && selectedMembers.includes(member)) || false;
        const isAdmin = groupAdmins?.includes(member);
        return (react_1.default.createElement(MemberListItem_1.MemberListItem, { pubkey: member, isSelected: isSelected, onSelect: onSelect, onUnselect: onUnselect, key: member, isAdmin: isAdmin }));
    })));
};
const ZombiesList = ({ convoId }) => {
    const convoProps = (0, useParamSelector_1.useConversationPropsById)(convoId);
    function onZombieClicked() {
        if (!convoProps?.weAreAdmin) {
            utils_1.ToastUtils.pushOnlyAdminCanRemove();
        }
    }
    if (!convoProps || !convoProps.zombies?.length) {
        return null;
    }
    const { zombies, weAreAdmin } = convoProps;
    const zombieElements = zombies.map((zombie) => {
        const isSelected = weAreAdmin || false;
        return (react_1.default.createElement(MemberListItem_1.MemberListItem, { isSelected: isSelected, onSelect: onZombieClicked, onUnselect: onZombieClicked, isZombie: true, key: zombie, pubkey: zombie }));
    });
    return (react_1.default.createElement(react_1.default.Fragment, null,
        react_1.default.createElement(Text_1.SpacerLG, null),
        weAreAdmin && (react_1.default.createElement(Text_1.Text, { padding: "20px", text: window.i18n('removeResidueMembers'), subtle: true, maxWidth: "400px", textAlign: "center" })),
        zombieElements));
};
async function onSubmit(convoId, membersAfterUpdate) {
    const convoProps = (0, conversations_1.getConversationController)()
        .get(convoId)
        .getConversationModelProps();
    if (!convoProps || !convoProps.isGroup || convoProps.isPublic) {
        throw new Error('Invalid convo for updateGroupMembersDialog');
    }
    if (!convoProps.weAreAdmin) {
        window.log.warn('Skipping update of members, we are not the admin');
        return;
    }
    const ourPK = utils_1.UserUtils.getOurPubKeyStrFromCache();
    const allMembersAfterUpdate = lodash_1.default.uniq(lodash_1.default.concat(membersAfterUpdate, [ourPK]));
    const existingMembers = convoProps.members || [];
    const existingZombies = convoProps.zombies || [];
    const allExistingMembersWithZombies = lodash_1.default.uniq(existingMembers.concat(existingZombies));
    const notPresentInOld = allMembersAfterUpdate.filter(m => !allExistingMembersWithZombies.includes(m));
    const membersToRemove = allExistingMembersWithZombies.filter(m => !allMembersAfterUpdate.includes(m));
    const xor = lodash_1.default.xor(membersToRemove, notPresentInOld);
    if (xor.length === 0) {
        window.log.info('skipping group update: no detected changes in group member list');
        return;
    }
    const filteredMembers = allMembersAfterUpdate.filter(memberAfterUpdate => !lodash_1.default.includes(membersToRemove, memberAfterUpdate));
    void (0, closed_group_1.initiateClosedGroupUpdate)(convoId, convoProps.name || 'Unknown', filteredMembers);
}
const UpdateGroupMembersDialog = (props) => {
    const { conversationId } = props;
    const convoProps = (0, useParamSelector_1.useConversationPropsById)(conversationId);
    const existingMembers = convoProps?.members || [];
    const { addTo, removeFrom, uniqueValues: membersToKeepWithUpdate } = (0, useSet_1.useSet)(existingMembers);
    const dispatch = (0, react_redux_1.useDispatch)();
    if (!convoProps || !convoProps.isGroup || convoProps.isPublic) {
        throw new Error('UpdateGroupMembersDialog invalid convoProps');
    }
    const weAreAdmin = convoProps.weAreAdmin || false;
    const closeDialog = () => {
        dispatch((0, modalDialog_1.updateGroupMembersModal)(null));
    };
    const onClickOK = async () => {
        await onSubmit(conversationId, membersToKeepWithUpdate);
        closeDialog();
    };
    (0, useKey_1.default)((event) => {
        return event.key === 'Esc' || event.key === 'Escape';
    }, closeDialog);
    (0, useKey_1.default)((event) => {
        return event.key === 'Enter';
    }, onClickOK);
    const onAdd = (member) => {
        if (!weAreAdmin) {
            utils_1.ToastUtils.pushOnlyAdminCanRemove();
            return;
        }
        addTo(member);
    };
    const onRemove = (member) => {
        if (!weAreAdmin) {
            window?.log?.warn('Only group admin can remove members!');
            utils_1.ToastUtils.pushOnlyAdminCanRemove();
            return;
        }
        if (convoProps.groupAdmins?.includes(member)) {
            utils_1.ToastUtils.pushCannotRemoveCreatorFromGroup();
            window?.log?.warn(`User ${member} cannot be removed as they are the creator of the closed group.`);
            return;
        }
        removeFrom(member);
    };
    const showNoMembersMessage = existingMembers.length === 0;
    const okText = window.i18n('ok');
    const cancelText = window.i18n('cancel');
    const titleText = "Group Members";
    return (react_1.default.createElement(BchatWrapperModal_1.BchatWrapperModal, { title: titleText, onClose: closeDialog },
        react_1.default.createElement("div", { className: "group-member-list__selection", style: { paddingTop: "10px" } },
            react_1.default.createElement(ClassicMemberList, { convoId: conversationId, onSelect: onAdd, onUnselect: onRemove, selectedMembers: membersToKeepWithUpdate })),
        react_1.default.createElement(ZombiesList, { convoId: conversationId }),
        showNoMembersMessage && react_1.default.createElement("p", null, window.i18n('noMembersInThisGroup')),
        react_1.default.createElement(Text_1.SpacerLG, null),
        react_1.default.createElement("div", { className: "bchat-modal__button-group" },
            react_1.default.createElement(BchatButton_1.BchatButton, { text: cancelText, onClick: closeDialog, buttonColor: BchatButton_1.BchatButtonColor.White }),
            weAreAdmin && (react_1.default.createElement(BchatButton_1.BchatButton, { text: okText, onClick: onClickOK, buttonColor: BchatButton_1.BchatButtonColor.Green })))));
};
exports.UpdateGroupMembersDialog = UpdateGroupMembersDialog;
