"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateCache = exports.getAllFromCacheForSource = exports.getAllFromCache = exports.addToCache = exports.removeFromCache = void 0;
const utils_1 = require("../bchat/utils");
const lodash_1 = __importDefault(require("lodash"));
const data_1 = require("../data/data");
async function removeFromCache(envelope) {
    const { id } = envelope;
    return (0, data_1.removeUnprocessed)(id);
}
exports.removeFromCache = removeFromCache;
async function addToCache(envelope, plaintext, messageHash) {
    const { id } = envelope;
    const encodedEnvelope = utils_1.StringUtils.decode(plaintext, 'base64');
    const data = {
        id,
        version: 2,
        envelope: encodedEnvelope,
        messageHash,
        timestamp: Date.now(),
        attempts: 1,
    };
    if (envelope.senderIdentity) {
        data.senderIdentity = envelope.senderIdentity;
    }
    return (0, data_1.saveUnprocessed)(data);
}
exports.addToCache = addToCache;
async function fetchAllFromCache() {
    const count = await (0, data_1.getUnprocessedCount)();
    if (count > 1500) {
        await (0, data_1.removeAllUnprocessed)();
        window?.log?.warn(`There were ${count} messages in cache. Deleted all instead of reprocessing`);
        return [];
    }
    const items = await (0, data_1.getAllUnprocessed)();
    return items;
}
async function getAllFromCache() {
    window?.log?.info('getAllFromCache');
    const items = await fetchAllFromCache();
    window?.log?.info('getAllFromCache loaded', items.length, 'saved envelopes');
    return Promise.all(lodash_1.default.map(items, async (item) => {
        const attempts = lodash_1.default.toNumber(item.attempts || 0) + 1;
        try {
            if (attempts >= 10) {
                window?.log?.warn('getAllFromCache final attempt for envelope', item.id);
                await (0, data_1.removeUnprocessed)(item.id);
            }
            else {
                await (0, data_1.updateUnprocessedAttempts)(item.id, attempts);
            }
        }
        catch (error) {
            window?.log?.error('getAllFromCache error updating item after load:', error && error.stack ? error.stack : error);
        }
        return item;
    }));
}
exports.getAllFromCache = getAllFromCache;
async function getAllFromCacheForSource(source) {
    const items = await fetchAllFromCache();
    const itemsFromSource = items.filter(item => !!item.senderIdentity || item.senderIdentity === source);
    window?.log?.info('getAllFromCacheForSource loaded', itemsFromSource.length, 'saved envelopes');
    return Promise.all(lodash_1.default.map(items, async (item) => {
        const attempts = lodash_1.default.toNumber(item.attempts || 0) + 1;
        try {
            if (attempts >= 10) {
                window?.log?.warn('getAllFromCache final attempt for envelope', item.id);
                await (0, data_1.removeUnprocessed)(item.id);
            }
            else {
                await (0, data_1.updateUnprocessedAttempts)(item.id, attempts);
            }
        }
        catch (error) {
            window?.log?.error('getAllFromCache error updating item after load:', error && error.stack ? error.stack : error);
        }
        return item;
    }));
}
exports.getAllFromCacheForSource = getAllFromCacheForSource;
async function updateCache(envelope, plaintext) {
    const { id } = envelope;
    const item = await (0, data_1.getUnprocessedById)(id);
    if (!item) {
        window?.log?.error(`updateCache: Didn't find item ${id} in cache to update`);
        return;
    }
    item.source = envelope.source;
    if (envelope.senderIdentity) {
        item.senderIdentity = envelope.senderIdentity;
    }
    item.decrypted = utils_1.StringUtils.decode(plaintext, 'base64');
    return (0, data_1.updateUnprocessedWithData)(item.id, item);
}
exports.updateCache = updateCache;
