"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createClosedGroup = exports.markGroupAsLeftOrKicked = exports.handleNewSecretGroup = exports.handleClosedGroupControlMessage = exports.innerRemoveAllClosedGroupEncryptionKeyPairs = exports.addKeyPairToCacheAndDBIfNeeded = exports.getAllCachedECKeyPair = exports.distributingClosedGroupEncryptionKeyPairs = void 0;
const protobuf_1 = require("../protobuf");
const cache_1 = require("./cache");
const types_1 = require("../bchat/types");
const String_1 = require("../bchat/utils/String");
const conversations_1 = require("../bchat/conversations");
const ClosedGroup = __importStar(require("../bchat/group/closed-group"));
const util_1 = require("../util");
const crypto_1 = require("../bchat/crypto");
const bchat_1 = require("../bchat");
const contentMessage_1 = require("./contentMessage");
const data_1 = require("../../ts/data/data");
const ClosedGroupNewMessage_1 = require("../bchat/messages/outgoing/controlMessage/group/ClosedGroupNewMessage");
const keypairs_1 = require("./keypairs");
const utils_1 = require("../bchat/utils");
const conversation_1 = require("../models/conversation");
const lodash_1 = __importDefault(require("lodash"));
const syncUtils_1 = require("../bchat/utils/syncUtils");
const ClosedGroupEncryptionPairReplyMessage_1 = require("../bchat/messages/outgoing/controlMessage/group/ClosedGroupEncryptionPairReplyMessage");
const receiver_1 = require("./receiver");
const conversations_2 = require("../state/ducks/conversations");
const snode_api_1 = require("../bchat/apis/snode_api");
const modalDialog_1 = require("../state/ducks/modalDialog");
const Performance_1 = require("../bchat/utils/Performance");
exports.distributingClosedGroupEncryptionKeyPairs = new Map();
const cacheOfClosedGroupKeyPairs = new Map();
async function getAllCachedECKeyPair(groupPubKey) {
    let keyPairsFound = cacheOfClosedGroupKeyPairs.get(groupPubKey);
    if (!keyPairsFound || keyPairsFound.length === 0) {
        keyPairsFound = (await (0, data_1.getAllEncryptionKeyPairsForGroup)(groupPubKey)) || [];
        cacheOfClosedGroupKeyPairs.set(groupPubKey, keyPairsFound);
    }
    return keyPairsFound.slice();
}
exports.getAllCachedECKeyPair = getAllCachedECKeyPair;
async function addKeyPairToCacheAndDBIfNeeded(groupPubKey, keyPair) {
    const existingKeyPairs = await getAllCachedECKeyPair(groupPubKey);
    const alreadySaved = existingKeyPairs.some(k => {
        return k.privateHex === keyPair.privateHex && k.publicHex === keyPair.publicHex;
    });
    if (alreadySaved) {
        return false;
    }
    await (0, data_1.addClosedGroupEncryptionKeyPair)(groupPubKey, keyPair);
    if (!cacheOfClosedGroupKeyPairs.has(groupPubKey)) {
        cacheOfClosedGroupKeyPairs.set(groupPubKey, []);
    }
    cacheOfClosedGroupKeyPairs.get(groupPubKey)?.push(keyPair);
    return true;
}
exports.addKeyPairToCacheAndDBIfNeeded = addKeyPairToCacheAndDBIfNeeded;
async function innerRemoveAllClosedGroupEncryptionKeyPairs(groupPubKey) {
    cacheOfClosedGroupKeyPairs.set(groupPubKey, []);
    await (0, data_1.removeAllClosedGroupEncryptionKeyPairs)(groupPubKey);
}
exports.innerRemoveAllClosedGroupEncryptionKeyPairs = innerRemoveAllClosedGroupEncryptionKeyPairs;
async function handleClosedGroupControlMessage(envelope, groupUpdate) {
    const { type } = groupUpdate;
    const { Type } = protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage;
    window?.log?.info(` handle closed group update from ${envelope.senderIdentity || envelope.source} about group ${envelope.source}`);
    if (util_1.BlockedNumberController.isGroupBlocked(types_1.PubKey.cast(envelope.source))) {
        window?.log?.warn('Message ignored; destined for blocked group');
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    if (type === Type.ENCRYPTION_KEY_PAIR) {
        const isComingFromGroupPubkey = envelope.type === protobuf_1.SignalService.Envelope.Type.CLOSED_GROUP_MESSAGE;
        await handleClosedGroupEncryptionKeyPair(envelope, groupUpdate, isComingFromGroupPubkey);
        return;
    }
    if (type === Type.NEW) {
        if (!(0, conversations_1.getConversationController)()
            .get(envelope.senderIdentity || envelope.source)
            ?.isApproved()) {
            window?.log?.info('Received new closed group message from an unapproved sender -- dropping message.');
            return;
        }
        await handleNewSecretGroup(envelope, groupUpdate);
        return;
    }
    if (type === Type.NAME_CHANGE ||
        type === Type.MEMBERS_REMOVED ||
        type === Type.MEMBERS_ADDED ||
        type === Type.MEMBER_LEFT ||
        type === Type.ENCRYPTION_KEY_PAIR_REQUEST) {
        await performIfValid(envelope, groupUpdate);
        return;
    }
    window?.log?.error('Unknown group update type: ', type);
    await (0, cache_1.removeFromCache)(envelope);
}
exports.handleClosedGroupControlMessage = handleClosedGroupControlMessage;
function sanityCheckNewGroup(groupUpdate) {
    const { name, publicKey, members, admins, encryptionKeyPair } = groupUpdate;
    if (!name?.length) {
        window?.log?.warn('groupUpdate: name is empty');
        return false;
    }
    if (!name?.length) {
        window?.log?.warn('groupUpdate: name is empty');
        return false;
    }
    if (!publicKey?.length) {
        window?.log?.warn('groupUpdate: publicKey is empty');
        return false;
    }
    const hexGroupPublicKey = (0, String_1.toHex)(publicKey);
    if (!types_1.PubKey.from(hexGroupPublicKey)) {
        window?.log?.warn('groupUpdate: publicKey is not recognized as a valid pubkey', hexGroupPublicKey);
        return false;
    }
    if (!members?.length) {
        window?.log?.warn('groupUpdate: members is empty');
        return false;
    }
    if (members.some(m => m.length === 0)) {
        window?.log?.warn('groupUpdate: one of the member pubkey is empty');
        return false;
    }
    if (!admins?.length) {
        window?.log?.warn('groupUpdate: admins is empty');
        return false;
    }
    if (admins.some(a => a.length === 0)) {
        window?.log?.warn('groupUpdate: one of the admins pubkey is empty');
        return false;
    }
    if (!encryptionKeyPair?.publicKey?.length) {
        window?.log?.warn('groupUpdate: keypair publicKey is empty');
        return false;
    }
    if (!encryptionKeyPair?.privateKey?.length) {
        window?.log?.warn('groupUpdate: keypair privateKey is empty');
        return false;
    }
    return true;
}
async function handleNewSecretGroup(envelope, groupUpdate) {
    if (groupUpdate.type !== protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage.Type.NEW) {
        return;
    }
    if (!sanityCheckNewGroup(groupUpdate)) {
        window?.log?.warn('Sanity check for newGroup failed, dropping the message...');
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    const ourNumber = utils_1.UserUtils.getOurPubKeyFromCache();
    if (envelope.senderIdentity === ourNumber.key) {
        window?.log?.warn('Dropping new closed group updatemessage from our other device.');
        return (0, cache_1.removeFromCache)(envelope);
    }
    const { name, publicKey, members: membersAsData, admins: adminsAsData, encryptionKeyPair, } = groupUpdate;
    const groupId = (0, String_1.toHex)(publicKey);
    const members = membersAsData.map(String_1.toHex);
    const admins = adminsAsData.map(String_1.toHex);
    const envelopeTimestamp = lodash_1.default.toNumber(envelope.timestamp);
    const sender = envelope.source;
    if (!members.includes(ourNumber.key)) {
        window?.log?.info('Got a new group message but apparently we are not a member of it. Dropping it.');
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    const groupConvo = (0, conversations_1.getConversationController)().get(groupId);
    const expireTimer = groupUpdate.expireTimer;
    if (groupConvo) {
        if (!groupConvo.get('isKickedFromGroup') && !groupConvo.get('left')) {
            const ecKeyPairAlreadyExistingConvo = new keypairs_1.ECKeyPair(encryptionKeyPair.publicKey, encryptionKeyPair.privateKey);
            const isKeyPairAlreadyHere = await addKeyPairToCacheAndDBIfNeeded(groupId, ecKeyPairAlreadyExistingConvo.toHexKeyPair());
            await groupConvo.updateExpireTimer(expireTimer, sender, Date.now());
            if (isKeyPairAlreadyHere) {
                window.log.info('Dropping already saved keypair for group', groupId);
                await (0, cache_1.removeFromCache)(envelope);
                return;
            }
            window.log.info(`Received the encryptionKeyPair for new group ${groupId}`);
            await (0, cache_1.removeFromCache)(envelope);
            window.log.warn('Closed group message of type NEW: the conversation already exists, but we saved the new encryption keypair');
            return;
        }
        groupConvo.set({
            left: false,
            isKickedFromGroup: false,
            lastJoinedTimestamp: lodash_1.default.toNumber(envelope.timestamp),
            zombies: [],
        });
    }
    const convo = groupConvo ||
        (await (0, conversations_1.getConversationController)().getOrCreateAndWait(groupId, conversation_1.ConversationTypeEnum.GROUP));
    window?.log?.info('Received a new ClosedGroup of id:', groupId);
    await ClosedGroup.addUpdateMessage(convo, { newName: name, joiningMembers: members }, envelope.senderIdentity || envelope.source, envelopeTimestamp);
    const groupDetails = {
        id: groupId,
        name: name,
        members: members,
        admins,
        activeAt: envelopeTimestamp,
        weWereJustAdded: true,
    };
    await ClosedGroup.updateOrCreateClosedGroup(groupDetails);
    convo.set('lastJoinedTimestamp', envelopeTimestamp);
    await convo.updateExpireTimer(expireTimer, sender, envelopeTimestamp);
    convo.updateLastMessage();
    await convo.commit();
    const ecKeyPair = new keypairs_1.ECKeyPair(encryptionKeyPair.publicKey, encryptionKeyPair.privateKey);
    window?.log?.info(`Received the encryptionKeyPair for new group ${groupId}`);
    await addKeyPairToCacheAndDBIfNeeded(groupId, ecKeyPair.toHexKeyPair());
    (0, snode_api_1.getSwarmPollingInstance)().addGroupId(types_1.PubKey.cast(groupId));
    await (0, cache_1.removeFromCache)(envelope);
    await (0, receiver_1.queueAllCachedFromSource)(groupId);
}
exports.handleNewSecretGroup = handleNewSecretGroup;
async function markGroupAsLeftOrKicked(groupPublicKey, groupConvo, isKicked) {
    await innerRemoveAllClosedGroupEncryptionKeyPairs(groupPublicKey);
    if (isKicked) {
        groupConvo.set('isKickedFromGroup', true);
    }
    else {
        groupConvo.set('left', true);
    }
    (0, snode_api_1.getSwarmPollingInstance)().removePubkey(groupPublicKey);
}
exports.markGroupAsLeftOrKicked = markGroupAsLeftOrKicked;
async function handleClosedGroupEncryptionKeyPair(envelope, groupUpdate, isComingFromGroupPubkey) {
    if (groupUpdate.type !==
        protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage.Type.ENCRYPTION_KEY_PAIR) {
        return;
    }
    const ourNumber = utils_1.UserUtils.getOurPubKeyFromCache();
    const groupPublicKey = (0, String_1.toHex)(groupUpdate.publicKey) || envelope.source;
    const sender = isComingFromGroupPubkey ? envelope.senderIdentity : envelope.source;
    window?.log?.info(`Got a group update for group ${groupPublicKey}, type: ENCRYPTION_KEY_PAIR`);
    const ourKeyPair = await utils_1.UserUtils.getIdentityKeyPair();
    if (!ourKeyPair) {
        window?.log?.warn("Couldn't find user X25519 key pair.");
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    const groupConvo = (0, conversations_1.getConversationController)().get(groupPublicKey);
    if (!groupConvo) {
        window?.log?.warn(`Ignoring closed group encryption key pair for nonexistent group. ${groupPublicKey}`);
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    if (!groupConvo.isMediumGroup()) {
        window?.log?.warn(`Ignoring closed group encryption key pair for nonexistent medium group. ${groupPublicKey}`);
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    if (!groupConvo.get('groupAdmins')?.includes(sender)) {
        window?.log?.warn(`Ignoring closed group encryption key pair from non-admin. ${groupPublicKey}`);
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    const ourWrapper = groupUpdate.wrappers.find(w => (0, String_1.toHex)(w.publicKey) === ourNumber.key);
    if (!ourWrapper) {
        window?.log?.warn(`Couldn\'t find our wrapper in the encryption keypairs wrappers for group ${groupPublicKey}`);
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    let plaintext;
    try {
        (0, Performance_1.perfStart)(`encryptionKeyPair-${envelope.id}`);
        const buffer = await (0, contentMessage_1.decryptWithBchatProtocol)(envelope, ourWrapper.encryptedKeyPair, keypairs_1.ECKeyPair.fromKeyPair(ourKeyPair));
        (0, Performance_1.perfEnd)(`encryptionKeyPair-${envelope.id}`, 'encryptionKeyPair');
        if (!buffer || buffer.byteLength === 0) {
            throw new Error();
        }
        plaintext = new Uint8Array(buffer);
    }
    catch (e) {
        window?.log?.warn("Couldn't decrypt closed group encryption key pair.", e);
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    let proto;
    try {
        proto = protobuf_1.SignalService.KeyPair.decode(plaintext);
        if (!proto || proto.privateKey.length === 0 || proto.publicKey.length === 0) {
            throw new Error();
        }
    }
    catch (e) {
        window?.log?.warn("Couldn't parse closed group encryption key pair.");
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    let keyPair;
    try {
        keyPair = new keypairs_1.ECKeyPair(proto.publicKey, proto.privateKey);
    }
    catch (e) {
        window?.log?.warn("Couldn't parse closed group encryption key pair.");
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    window?.log?.info(`Received a new encryptionKeyPair for group ${groupPublicKey}`);
    const newKeyPairInHex = keyPair.toHexKeyPair();
    const isKeyPairAlreadyHere = await addKeyPairToCacheAndDBIfNeeded(groupPublicKey, newKeyPairInHex);
    if (isKeyPairAlreadyHere) {
        window?.log?.info('Dropping already saved keypair for group', groupPublicKey);
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    window?.log?.info('Got a new encryption keypair for group', groupPublicKey);
    await (0, cache_1.removeFromCache)(envelope);
    await (0, receiver_1.queueAllCachedFromSource)(groupPublicKey);
}
async function performIfValid(envelope, groupUpdate) {
    const { Type } = protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage;
    const groupPublicKey = envelope.source;
    const sender = envelope.senderIdentity;
    const convo = (0, conversations_1.getConversationController)().get(groupPublicKey);
    if (!convo) {
        window?.log?.warn('dropping message for nonexistent group');
        return (0, cache_1.removeFromCache)(envelope);
    }
    if (!convo) {
        window?.log?.warn('Ignoring a closed group update message (INFO) for a non-existing group');
        return (0, cache_1.removeFromCache)(envelope);
    }
    let lastJoinedTimestamp = convo.get('lastJoinedTimestamp');
    if (!lastJoinedTimestamp) {
        const aYearAgo = Date.now() - 1000 * 60 * 24 * 365;
        convo.set({
            lastJoinedTimestamp: aYearAgo,
        });
        lastJoinedTimestamp = aYearAgo;
    }
    const envelopeTimestamp = lodash_1.default.toNumber(envelope.timestamp);
    if (envelopeTimestamp <= lastJoinedTimestamp) {
        window?.log?.warn('Got a group update with an older timestamp than when we joined this group last time. Dropping it.');
        return (0, cache_1.removeFromCache)(envelope);
    }
    const oldMembers = convo.get('members') || [];
    if (!oldMembers.includes(sender)) {
        window?.log?.error(`Error: closed group: ignoring closed group update message from non-member. ${sender} is not a current member.`);
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    await (0, conversations_1.getConversationController)().getOrCreateAndWait(sender, conversation_1.ConversationTypeEnum.PRIVATE);
    if (groupUpdate.type === Type.NAME_CHANGE) {
        await handleClosedGroupNameChanged(envelope, groupUpdate, convo);
    }
    else if (groupUpdate.type === Type.MEMBERS_ADDED) {
        await handleClosedGroupMembersAdded(envelope, groupUpdate, convo);
    }
    else if (groupUpdate.type === Type.MEMBERS_REMOVED) {
        await handleClosedGroupMembersRemoved(envelope, groupUpdate, convo);
    }
    else if (groupUpdate.type === Type.MEMBER_LEFT) {
        await handleClosedGroupMemberLeft(envelope, convo);
    }
    else if (groupUpdate.type === Type.ENCRYPTION_KEY_PAIR_REQUEST) {
        window?.log?.warn('Received ENCRYPTION_KEY_PAIR_REQUEST message but it is not enabled for now.');
        await (0, cache_1.removeFromCache)(envelope);
    }
    return true;
}
async function handleClosedGroupNameChanged(envelope, groupUpdate, convo) {
    const newName = groupUpdate.name;
    window?.log?.info(`Got a group update for group ${envelope.source}, type: NAME_CHANGED`);
    if (newName !== convo.get('name')) {
        const groupDiff = {
            newName,
        };
        await ClosedGroup.addUpdateMessage(convo, groupDiff, envelope.senderIdentity, lodash_1.default.toNumber(envelope.timestamp));
        convo.set({ name: newName });
        convo.updateLastMessage();
        await convo.commit();
    }
    await (0, cache_1.removeFromCache)(envelope);
}
async function handleClosedGroupMembersAdded(envelope, groupUpdate, convo) {
    const { members: addedMembersBinary } = groupUpdate;
    const addedMembers = (addedMembersBinary || []).map(String_1.toHex);
    const oldMembers = convo.get('members') || [];
    const membersNotAlreadyPresent = addedMembers.filter(m => !oldMembers.includes(m));
    window?.log?.info(`Got a group update for group ${envelope.source}, type: MEMBERS_ADDED`);
    addedMembers.forEach(added => removeMemberFromZombies(envelope, types_1.PubKey.cast(added), convo));
    if (membersNotAlreadyPresent.length === 0) {
        window?.log?.info('no new members in this group update compared to what we have already. Skipping update');
        await convo.commit();
        await (0, cache_1.removeFromCache)(envelope);
        return;
    }
    if (await areWeAdmin(convo)) {
        await sendLatestKeyPairToUsers(convo, convo.id, membersNotAlreadyPresent);
    }
    const members = [...oldMembers, ...membersNotAlreadyPresent];
    await Promise.all(members.map(async (m) => (0, conversations_1.getConversationController)().getOrCreateAndWait(m, conversation_1.ConversationTypeEnum.PRIVATE)));
    const groupDiff = {
        joiningMembers: membersNotAlreadyPresent,
    };
    await ClosedGroup.addUpdateMessage(convo, groupDiff, envelope.senderIdentity, lodash_1.default.toNumber(envelope.timestamp));
    convo.set({ members });
    convo.updateLastMessage();
    await convo.commit();
    await (0, cache_1.removeFromCache)(envelope);
}
async function areWeAdmin(groupConvo) {
    if (!groupConvo) {
        throw new Error('areWeAdmin needs a convo');
    }
    const groupAdmins = groupConvo.get('groupAdmins');
    const ourNumber = utils_1.UserUtils.getOurPubKeyStrFromCache();
    return groupAdmins?.includes(ourNumber) || false;
}
async function handleClosedGroupMembersRemoved(envelope, groupUpdate, convo) {
    const currentMembers = convo.get('members');
    const removedMembers = groupUpdate.members.map(String_1.toHex);
    const effectivelyRemovedMembers = removedMembers.filter(m => currentMembers.includes(m));
    const groupPubKey = envelope.source;
    window?.log?.info(`Got a group update for group ${envelope.source}, type: MEMBERS_REMOVED`);
    const membersAfterUpdate = lodash_1.default.difference(currentMembers, removedMembers);
    const groupAdmins = convo.get('groupAdmins');
    if (!groupAdmins?.length) {
        throw new Error('No admins found for closed group member removed update.');
    }
    const firstAdmin = groupAdmins[0];
    if (removedMembers.includes(firstAdmin)) {
        window?.log?.warn('Ignoring invalid closed group update: trying to remove the admin.');
        await (0, cache_1.removeFromCache)(envelope);
        throw new Error('Admins cannot be removed. They can only leave');
    }
    if (!groupAdmins.includes(envelope.senderIdentity)) {
        window?.log?.warn('Ignoring invalid closed group update. Only admins can remove members.');
        await (0, cache_1.removeFromCache)(envelope);
        throw new Error('Only admins can remove members.');
    }
    const ourPubKey = utils_1.UserUtils.getOurPubKeyFromCache();
    const wasCurrentUserRemoved = !membersAfterUpdate.includes(ourPubKey.key);
    if (wasCurrentUserRemoved) {
        await markGroupAsLeftOrKicked(groupPubKey, convo, true);
    }
    if (membersAfterUpdate.length !== currentMembers.length) {
        const groupDiff = {
            kickedMembers: effectivelyRemovedMembers,
        };
        await ClosedGroup.addUpdateMessage(convo, groupDiff, envelope.senderIdentity, lodash_1.default.toNumber(envelope.timestamp));
        convo.updateLastMessage();
    }
    const zombies = convo.get('zombies').filter(z => membersAfterUpdate.includes(z));
    convo.set({ members: membersAfterUpdate });
    convo.set({ zombies });
    await convo.commit();
    await (0, cache_1.removeFromCache)(envelope);
}
function isUserAZombie(convo, user) {
    return convo.get('zombies').includes(user.key);
}
function addMemberToZombies(_envelope, userToAdd, convo) {
    const zombies = convo.get('zombies');
    const isAlreadyZombie = isUserAZombie(convo, userToAdd);
    if (isAlreadyZombie) {
        return false;
    }
    convo.set('zombies', [...zombies, userToAdd.key]);
    return true;
}
function removeMemberFromZombies(_envelope, userToAdd, convo) {
    const zombies = convo.get('zombies');
    const isAlreadyAZombie = isUserAZombie(convo, userToAdd);
    if (!isAlreadyAZombie) {
        return false;
    }
    convo.set('zombies', zombies.filter(z => z !== userToAdd.key));
    return true;
}
async function handleClosedGroupAdminMemberLeft(groupPublicKey, isCurrentUserAdmin, convo, envelope) {
    await markGroupAsLeftOrKicked(groupPublicKey, convo, !isCurrentUserAdmin);
    const groupDiff = {
        kickedMembers: convo.get('members'),
    };
    convo.set('members', []);
    await ClosedGroup.addUpdateMessage(convo, groupDiff, envelope.senderIdentity, lodash_1.default.toNumber(envelope.timestamp));
    convo.updateLastMessage();
    await convo.commit();
    await (0, cache_1.removeFromCache)(envelope);
}
async function handleClosedGroupLeftOurself(groupPublicKey, convo, envelope) {
    await markGroupAsLeftOrKicked(groupPublicKey, convo, false);
    const groupDiff = {
        leavingMembers: [envelope.senderIdentity],
    };
    await ClosedGroup.addUpdateMessage(convo, groupDiff, envelope.senderIdentity, lodash_1.default.toNumber(envelope.timestamp));
    convo.updateLastMessage();
    convo.set('members', convo.get('members').filter(m => !utils_1.UserUtils.isUsFromCache(m)));
    await convo.commit();
    await (0, cache_1.removeFromCache)(envelope);
}
async function handleClosedGroupMemberLeft(envelope, convo) {
    const sender = envelope.senderIdentity;
    const groupPublicKey = envelope.source;
    const didAdminLeave = convo.get('groupAdmins')?.includes(sender) || false;
    const oldMembers = convo.get('members') || [];
    const newMembers = oldMembers.filter(s => s !== sender);
    window?.log?.info(`Got a group update for group ${envelope.source}, type: MEMBER_LEFT`);
    if (utils_1.UserUtils.isUsFromCache(sender)) {
        window?.log?.info('Got self-sent group update member left...');
    }
    const ourPubkey = utils_1.UserUtils.getOurPubKeyStrFromCache();
    const isCurrentUserAdmin = convo.get('groupAdmins')?.includes(ourPubkey) || false;
    if (didAdminLeave) {
        await handleClosedGroupAdminMemberLeft(groupPublicKey, isCurrentUserAdmin, convo, envelope);
        return;
    }
    if (!newMembers.includes(ourPubkey)) {
        await handleClosedGroupLeftOurself(groupPublicKey, convo, envelope);
        return;
    }
    const groupDiff = {
        leavingMembers: [sender],
    };
    await ClosedGroup.addUpdateMessage(convo, groupDiff, envelope.senderIdentity, lodash_1.default.toNumber(envelope.timestamp));
    convo.updateLastMessage();
    if (oldMembers.includes(sender)) {
        addMemberToZombies(envelope, types_1.PubKey.cast(sender), convo);
    }
    convo.set('members', newMembers);
    await convo.commit();
    await (0, cache_1.removeFromCache)(envelope);
}
async function sendLatestKeyPairToUsers(_groupConvo, groupPubKey, targetUsers) {
    const inMemoryKeyPair = exports.distributingClosedGroupEncryptionKeyPairs.get(groupPubKey);
    const latestKeyPair = await (0, data_1.getLatestClosedGroupEncryptionKeyPair)(groupPubKey);
    if (!inMemoryKeyPair && !latestKeyPair) {
        window?.log?.info('We do not have the keypair ourself, so dropping this message.');
        return;
    }
    const keyPairToUse = inMemoryKeyPair || keypairs_1.ECKeyPair.fromHexKeyPair(latestKeyPair);
    await Promise.all(targetUsers.map(async (member) => {
        window?.log?.info(`Sending latest closed group encryption key pair to: ${member}`);
        await (0, conversations_1.getConversationController)().getOrCreateAndWait(member, conversation_1.ConversationTypeEnum.PRIVATE);
        const wrappers = await ClosedGroup.buildEncryptionKeyPairWrappers([member], keyPairToUse);
        const keypairsMessage = new ClosedGroupEncryptionPairReplyMessage_1.ClosedGroupEncryptionPairReplyMessage({
            groupId: groupPubKey,
            timestamp: Date.now(),
            encryptedKeyPairs: wrappers,
        });
        await (0, bchat_1.getMessageQueue)().sendToPubKey(types_1.PubKey.cast(member), keypairsMessage);
    }));
}
async function createClosedGroup(groupName, members) {
    const setOfMembers = new Set(members);
    const ourNumber = utils_1.UserUtils.getOurPubKeyFromCache();
    const groupPublicKey = await (0, crypto_1.generateClosedGroupPublicKey)();
    const encryptionKeyPair = await (0, crypto_1.generateCurve25519KeyPairWithoutPrefix)();
    if (!encryptionKeyPair) {
        throw new Error('Could not create encryption keypair for new closed group');
    }
    setOfMembers.add(ourNumber.key);
    const listOfMembers = [...setOfMembers];
    const convo = await (0, conversations_1.getConversationController)().getOrCreateAndWait(groupPublicKey, conversation_1.ConversationTypeEnum.GROUP);
    const admins = [ourNumber.key];
    const existingExpireTimer = 0;
    const groupDetails = {
        id: groupPublicKey,
        name: groupName,
        members: listOfMembers,
        admins,
        activeAt: Date.now(),
        expireTimer: existingExpireTimer,
    };
    const groupDiff = {
        newName: groupName,
        joiningMembers: listOfMembers,
    };
    const dbMessage = await ClosedGroup.addUpdateMessage(convo, groupDiff, utils_1.UserUtils.getOurPubKeyStrFromCache(), Date.now());
    await convo.setIsApproved(true, false);
    await ClosedGroup.updateOrCreateClosedGroup(groupDetails);
    await convo.commit();
    convo.updateLastMessage();
    const allInvitesSent = await sendToGroupMembers(listOfMembers, groupPublicKey, groupName, admins, encryptionKeyPair, dbMessage, existingExpireTimer);
    if (allInvitesSent) {
        const newHexKeypair = encryptionKeyPair.toHexKeyPair();
        const isHexKeyPairSaved = await addKeyPairToCacheAndDBIfNeeded(groupPublicKey, newHexKeypair);
        if (isHexKeyPairSaved) {
            window?.log?.info('Dropping already saved keypair for group', groupPublicKey);
        }
        (0, snode_api_1.getSwarmPollingInstance)().addGroupId(new types_1.PubKey(groupPublicKey));
    }
    await (0, syncUtils_1.forceSyncConfigurationNowIfNeeded)();
    await (0, conversations_2.openConversationWithMessages)({ conversationKey: groupPublicKey, messageId: null });
}
exports.createClosedGroup = createClosedGroup;
async function sendToGroupMembers(listOfMembers, groupPublicKey, groupName, admins, encryptionKeyPair, dbMessage, existingExpireTimer, isRetry = false) {
    const promises = createInvitePromises(listOfMembers, groupPublicKey, groupName, admins, encryptionKeyPair, dbMessage, existingExpireTimer);
    window?.log?.info(`Creating a new group and an encryptionKeyPair for group ${groupPublicKey}`);
    const inviteResults = await Promise.all(promises);
    const allInvitesSent = lodash_1.default.every(inviteResults, inviteResult => inviteResult !== false);
    if (allInvitesSent) {
        if (isRetry) {
            const invitesTitle = inviteResults.length > 1
                ? window.i18n('secretGroupInviteSuccessTitlePlural')
                : window.i18n('secretGroupInviteSuccessTitle');
            window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)({
                title: invitesTitle,
                message: window.i18n('secretGroupInviteSuccessMessage'),
                hideCancel: true,
            }));
        }
        return allInvitesSent;
    }
    else {
        window.inboxStore?.dispatch((0, modalDialog_1.updateConfirmModal)({
            title: inviteResults.length > 1
                ? window.i18n('secretGroupInviteFailTitlePlural')
                : window.i18n('secretGroupInviteFailTitle'),
            message: inviteResults.length > 1
                ? window.i18n('secretGroupInviteFailMessagePlural')
                : window.i18n('secretGroupInviteFailMessage'),
            okText: window.i18n('secretGroupInviteOkText'),
            onClickOk: async () => {
                const membersToResend = new Array();
                inviteResults.forEach((result, index) => {
                    const member = listOfMembers[index];
                    if (result !== true || admins.includes(member)) {
                        membersToResend.push(member);
                    }
                });
                if (membersToResend.length > 0) {
                    const isRetrySend = true;
                    await sendToGroupMembers(membersToResend, groupPublicKey, groupName, admins, encryptionKeyPair, dbMessage, existingExpireTimer, isRetrySend);
                }
            },
        }));
    }
    return allInvitesSent;
}
function createInvitePromises(listOfMembers, groupPublicKey, groupName, admins, encryptionKeyPair, dbMessage, existingExpireTimer) {
    return listOfMembers.map(async (m) => {
        const messageParams = {
            groupId: groupPublicKey,
            name: groupName,
            members: listOfMembers,
            admins,
            keypair: encryptionKeyPair,
            timestamp: Date.now(),
            identifier: dbMessage.id,
            expireTimer: existingExpireTimer,
        };
        const message = new ClosedGroupNewMessage_1.ClosedGroupNewMessage(messageParams);
        return (0, bchat_1.getMessageQueue)().sendToPubKeyNonDurably(types_1.PubKey.cast(m), message);
    });
}
