"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleConfigurationMessage = void 0;
const lodash_1 = __importDefault(require("lodash"));
const data_1 = require("../data/data");
const conversation_1 = require("../models/conversation");
const JoinOpenGroupV2_1 = require("../bchat/apis/open_group_api/opengroupV2/JoinOpenGroupV2");
const OpenGroupUtils_1 = require("../bchat/apis/open_group_api/utils/OpenGroupUtils");
const protobuf_1 = require("../protobuf");
const conversations_1 = require("../bchat/conversations");
const utils_1 = require("../bchat/utils");
const String_1 = require("../bchat/utils/String");
const events_1 = require("../shims/events");
const util_1 = require("../util");
const cache_1 = require("./cache");
const closedGroups_1 = require("./closedGroups");
const interactions_1 = require("../interactions");
const storage_1 = require("../util/storage");
const userProfileImageUpdates_1 = require("./userProfileImageUpdates");
async function handleOurProfileUpdate(sentAt, configMessage) {
    const latestProfileUpdateTimestamp = (0, storage_1.getLastProfileUpdateTimestamp)();
    if (!latestProfileUpdateTimestamp || sentAt > latestProfileUpdateTimestamp) {
        window?.log?.info(`Handling our profileUdpate ourLastUpdate:${latestProfileUpdateTimestamp}, envelope sent at: ${sentAt}`);
        const { profileKey, profilePicture, displayName } = configMessage;
        const bchatProfile = {
            displayName,
            profilePicture,
        };
        await (0, userProfileImageUpdates_1.updateOurProfileSync)(bchatProfile, profileKey);
        await (0, storage_1.setLastProfileUpdateTimestamp)(lodash_1.default.toNumber(sentAt));
        if (displayName) {
            (0, events_1.trigger)(events_1.configurationMessageReceived, displayName);
        }
        else {
            window?.log?.warn('Got a configuration message but the display name is empty');
        }
    }
}
async function handleGroupsAndContactsFromConfigMessage(envelope, configMessage) {
    const envelopeTimestamp = lodash_1.default.toNumber(envelope.timestamp);
    const lastConfigUpdate = await (0, data_1.getItemById)(data_1.hasSyncedInitialConfigurationItem);
    const lastConfigTimestamp = lastConfigUpdate?.timestamp;
    const isNewerConfig = !lastConfigTimestamp || (lastConfigTimestamp && lastConfigTimestamp < envelopeTimestamp);
    if (!isNewerConfig) {
        window?.log?.info('Received outdated configuration message... Dropping message.');
        return;
    }
    await (0, data_1.createOrUpdateItem)({
        id: 'hasSyncedInitialConfigurationItem',
        value: true,
        timestamp: envelopeTimestamp,
    });
    if (!lastConfigTimestamp) {
        await handleClosedGroupsFromConfig(configMessage.closedGroups, envelope);
    }
    handleOpenGroupsFromConfig(configMessage.openGroups);
    if (configMessage.contacts?.length) {
        await Promise.all(configMessage.contacts.map(async (c) => handleContactFromConfig(c, envelope)));
    }
}
const handleOpenGroupsFromConfig = (openGroups) => {
    const numberOpenGroup = openGroups?.length || 0;
    for (let i = 0; i < numberOpenGroup; i++) {
        const currentOpenGroupUrl = openGroups[i];
        const parsedRoom = (0, JoinOpenGroupV2_1.parseOpenGroupV2)(currentOpenGroupUrl);
        if (!parsedRoom) {
            continue;
        }
        const roomConvoId = (0, OpenGroupUtils_1.getOpenGroupV2ConversationId)(parsedRoom.serverUrl, parsedRoom.roomId);
        if (!(0, conversations_1.getConversationController)().get(roomConvoId)) {
            window?.log?.info(`triggering join of public chat '${currentOpenGroupUrl}' from ConfigurationMessage`);
            void (0, JoinOpenGroupV2_1.joinOpenGroupV2WithUIEvents)(currentOpenGroupUrl, false, true);
        }
    }
};
const handleClosedGroupsFromConfig = async (closedGroups, envelope) => {
    const numberClosedGroup = closedGroups?.length || 0;
    window?.log?.info(`Received ${numberClosedGroup} closed group on configuration. Creating them... `);
    await Promise.all(closedGroups.map(async (c) => {
        const groupUpdate = new protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage({
            type: protobuf_1.SignalService.DataMessage.ClosedGroupControlMessage.Type.NEW,
            encryptionKeyPair: c.encryptionKeyPair,
            name: c.name,
            admins: c.admins,
            members: c.members,
            publicKey: c.publicKey,
        });
        try {
            await (0, closedGroups_1.handleNewSecretGroup)(envelope, groupUpdate);
        }
        catch (e) {
            window?.log?.warn('failed to handle  a new closed group from configuration message');
        }
    }));
};
const handleContactFromConfig = async (contactReceived, envelope) => {
    try {
        if (!contactReceived.publicKey?.length) {
            return;
        }
        const contactConvo = await (0, conversations_1.getConversationController)().getOrCreateAndWait((0, String_1.toHex)(contactReceived.publicKey), conversation_1.ConversationTypeEnum.PRIVATE);
        const profile = {
            displayName: contactReceived.name,
            profilePictre: contactReceived.profilePicture,
        };
        const existingActiveAt = contactConvo.get('active_at');
        if (!existingActiveAt || existingActiveAt === 0) {
            contactConvo.set('active_at', lodash_1.default.toNumber(envelope.timestamp));
        }
        if (contactReceived.isApproved === true) {
            if (!contactConvo.isApproved()) {
                await contactConvo.setIsApproved(Boolean(contactReceived.isApproved));
                await contactConvo.addOutgoingApprovalMessage(lodash_1.default.toNumber(envelope.timestamp));
            }
            if (contactReceived.didApproveMe === true) {
                await contactConvo.setDidApproveMe(Boolean(contactReceived.didApproveMe));
            }
        }
        if (contactReceived.isBlocked === true) {
            if (contactConvo.isIncomingRequest()) {
                await interactions_1.ConversationInteraction.deleteAllMessagesByConvoIdNoConfirmation(contactConvo.id);
            }
            await util_1.BlockedNumberController.block(contactConvo.id);
        }
        else if (contactReceived.isBlocked === false) {
            await util_1.BlockedNumberController.unblock(contactConvo.id);
        }
        void (0, userProfileImageUpdates_1.appendFetchAvatarAndProfileJob)(contactConvo, profile, contactReceived.profileKey);
    }
    catch (e) {
        window?.log?.warn('failed to handle  a new closed group from configuration message');
    }
};
async function handleConfigurationMessage(envelope, configurationMessage) {
    window?.log?.info('Handling configuration message');
    const ourPubkey = utils_1.UserUtils.getOurPubKeyStrFromCache();
    if (!ourPubkey) {
        return;
    }
    if (envelope.source !== ourPubkey) {
        window?.log?.info('Dropping configuration change from someone else than us.');
        return (0, cache_1.removeFromCache)(envelope);
    }
    await handleOurProfileUpdate(envelope.timestamp, configurationMessage);
    await handleGroupsAndContactsFromConfigMessage(envelope, configurationMessage);
    await (0, cache_1.removeFromCache)(envelope);
}
exports.handleConfigurationMessage = handleConfigurationMessage;
