"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isSwarmMessageDuplicate = exports.handleSwarmDataMessage = exports.isMessageEmpty = void 0;
const protobuf_1 = require("./../protobuf");
const cache_1 = require("./cache");
const common_1 = require("./common");
const types_1 = require("../bchat/types");
const queuedJob_1 = require("./queuedJob");
const lodash_1 = __importDefault(require("lodash"));
const utils_1 = require("../bchat/utils");
const conversations_1 = require("../bchat/conversations");
const closedGroups_1 = require("./closedGroups");
const data_1 = require("../../ts/data/data");
const conversation_1 = require("../models/conversation");
const messageFactory_1 = require("../models/messageFactory");
const User_1 = require("../bchat/utils/User");
const userProfileImageUpdates_1 = require("./userProfileImageUpdates");
const Errors_1 = require("../types/attachments/Errors");
const reactions_1 = require("../util/reactions");
function cleanAttachment(attachment) {
    return {
        ...lodash_1.default.omit(attachment, 'thumbnail'),
        id: attachment.id.toString(),
        key: attachment.key ? utils_1.StringUtils.decode(attachment.key, 'base64') : null,
        digest: attachment.digest && attachment.digest.length > 0
            ? utils_1.StringUtils.decode(attachment.digest, 'base64')
            : null,
    };
}
function cleanAttachments(decrypted) {
    const { quote, group } = decrypted;
    if (group && group.type === protobuf_1.SignalService.GroupContext.Type.UPDATE) {
        if (group.avatar !== null) {
            group.avatar = cleanAttachment(group.avatar);
        }
    }
    decrypted.attachments = (decrypted.attachments || []).map(cleanAttachment);
    decrypted.preview = (decrypted.preview || []).map((item) => {
        const { image } = item;
        if (!image) {
            return item;
        }
        return {
            ...item,
            image: cleanAttachment(image),
        };
    });
    if (quote) {
        if (quote.id) {
            quote.id = lodash_1.default.toNumber(quote.id);
        }
        quote.attachments = (quote.attachments || []).map((item) => {
            const { thumbnail } = item;
            if (!thumbnail || thumbnail.length === 0) {
                return item;
            }
            return {
                ...item,
                thumbnail: cleanAttachment(item.thumbnail),
            };
        });
    }
}
function isMessageEmpty(message) {
    const { flags, body, attachments, group, quote, preview, openGroupInvitation, payment, sharedContact, reaction } = message;
    return (!flags &&
        isBodyEmpty(body) &&
        lodash_1.default.isEmpty(attachments) &&
        lodash_1.default.isEmpty(group) &&
        lodash_1.default.isEmpty(quote) &&
        lodash_1.default.isEmpty(preview) &&
        lodash_1.default.isEmpty(openGroupInvitation) &&
        lodash_1.default.isEmpty(payment) &&
        lodash_1.default.isEmpty(sharedContact) &&
        lodash_1.default.isEmpty(reaction));
}
exports.isMessageEmpty = isMessageEmpty;
function isBodyEmpty(body) {
    return lodash_1.default.isEmpty(body);
}
async function cleanIncomingDataMessage(envelope, rawDataMessage) {
    const FLAGS = protobuf_1.SignalService.DataMessage.Flags;
    if (rawDataMessage.flags == null) {
        rawDataMessage.flags = 0;
    }
    if (rawDataMessage.expireTimer == null) {
        rawDataMessage.expireTimer = 0;
    }
    if (rawDataMessage.flags & FLAGS.EXPIRATION_TIMER_UPDATE) {
        rawDataMessage.body = '';
        rawDataMessage.attachments = [];
    }
    else if (rawDataMessage.flags !== 0) {
        throw new Error('Unknown flags in message');
    }
    const attachmentCount = rawDataMessage?.attachments?.length || 0;
    const ATTACHMENT_MAX = 32;
    if (attachmentCount > ATTACHMENT_MAX) {
        await (0, cache_1.removeFromCache)(envelope);
        throw new Error(`Too many attachments: ${attachmentCount} included in one message, max is ${ATTACHMENT_MAX}`);
    }
    cleanAttachments(rawDataMessage);
    if (!lodash_1.default.isFinite(rawDataMessage?.timestamp)) {
        rawDataMessage.timestamp = envelope.timestamp;
    }
    return rawDataMessage;
}
async function handleSwarmDataMessage(envelope, sentAtTimestamp, rawDataMessage, messageHash, senderConversationModel) {
    window.log.info('handleSwarmDataMessage');
    const cleanDataMessage = await cleanIncomingDataMessage(envelope, rawDataMessage);
    if (cleanDataMessage.closedGroupControlMessage) {
        await (0, closedGroups_1.handleClosedGroupControlMessage)(envelope, cleanDataMessage.closedGroupControlMessage);
        return;
    }
    const isSyncedMessage = Boolean(cleanDataMessage.syncTarget?.length);
    const convoIdOfSender = envelope.senderIdentity || envelope.source;
    const isMe = utils_1.UserUtils.isUsFromCache(convoIdOfSender);
    if (isSyncedMessage && !isMe) {
        window?.log?.warn('Got a sync message from someone else than me. Dropping it.');
        return (0, cache_1.removeFromCache)(envelope);
    }
    else if (isSyncedMessage) {
    }
    const convoIdToAddTheMessageTo = types_1.PubKey.removeTextSecurePrefixIfNeeded(isSyncedMessage ? cleanDataMessage.syncTarget : envelope.source);
    const convoToAddMessageTo = await (0, conversations_1.getConversationController)().getOrCreateAndWait(convoIdToAddTheMessageTo, envelope.senderIdentity ? conversation_1.ConversationTypeEnum.GROUP : conversation_1.ConversationTypeEnum.PRIVATE);
    window?.log?.info(`Handle dataMessage about convo ${convoIdToAddTheMessageTo} from user: ${convoIdOfSender}`);
    if (!isMe &&
        senderConversationModel &&
        cleanDataMessage.profile &&
        cleanDataMessage.profileKey?.length) {
        void (0, userProfileImageUpdates_1.appendFetchAvatarAndProfileJob)(senderConversationModel, cleanDataMessage.profile, cleanDataMessage.profileKey);
    }
    if (isMessageEmpty(cleanDataMessage)) {
        window?.log?.warn(`Message ${(0, common_1.getEnvelopeId)(envelope)} ignored; it was empty`);
        return (0, cache_1.removeFromCache)(envelope);
    }
    if (!convoIdToAddTheMessageTo) {
        window?.log?.error('We cannot handle a message without a conversationId');
        confirm();
        return;
    }
    const msgModel = isSyncedMessage ||
        (envelope.senderIdentity && (0, User_1.isUsFromCache)(envelope.senderIdentity)) ||
        (envelope.source && (0, User_1.isUsFromCache)(envelope.source))
        ? (0, messageFactory_1.createSwarmMessageSentFromUs)({
            conversationId: convoIdToAddTheMessageTo,
            messageHash,
            sentAt: sentAtTimestamp,
        })
        : (0, messageFactory_1.createSwarmMessageSentFromNotUs)({
            conversationId: convoIdToAddTheMessageTo,
            messageHash,
            sender: senderConversationModel.id,
            sentAt: sentAtTimestamp,
        });
    await handleSwarmMessage(msgModel, messageHash, sentAtTimestamp, cleanDataMessage, convoToAddMessageTo, () => (0, cache_1.removeFromCache)(envelope));
}
exports.handleSwarmDataMessage = handleSwarmDataMessage;
async function isSwarmMessageDuplicate({ source, sentAt, }) {
    try {
        const result = await (0, data_1.getMessageBySenderAndSentAt)({
            source,
            sentAt,
        });
        return Boolean(result);
    }
    catch (error) {
        window?.log?.error('isSwarmMessageDuplicate error:', (0, Errors_1.toLogFormat)(error));
        return false;
    }
}
exports.isSwarmMessageDuplicate = isSwarmMessageDuplicate;
async function handleSwarmMessage(msgModel, messageHash, sentAt, rawDataMessage, convoToAddMessageTo, confirm) {
    if (!rawDataMessage || !msgModel) {
        window?.log?.warn('Invalid data passed to handleSwarmMessage.');
        confirm();
        return;
    }
    void convoToAddMessageTo.queueJob(async () => {
        if (!msgModel.get('isPublic') && rawDataMessage.reaction && rawDataMessage.syncTarget) {
            await (0, reactions_1.handleMessageReaction)(rawDataMessage.reaction, msgModel.get('source'), false, messageHash);
            confirm();
            return;
        }
        const isDuplicate = await isSwarmMessageDuplicate({
            source: msgModel.get('source'),
            sentAt,
        });
        if (isDuplicate) {
            window?.log?.info('Received duplicate message. Dropping it.');
            confirm();
            return;
        }
        await (0, queuedJob_1.handleMessageJob)(msgModel, convoToAddMessageTo, (0, queuedJob_1.toRegularMessage)(rawDataMessage), confirm, msgModel.get('source'), messageHash);
    });
}
