"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleMessageJob = exports.toRegularMessage = void 0;
const attachments_1 = require("./attachments");
const lodash_1 = __importDefault(require("lodash"));
const conversations_1 = require("../bchat/conversations");
const conversation_1 = require("../models/conversation");
const message_1 = require("../models/message");
const data_1 = require("../../ts/data/data");
const utils_1 = require("../bchat/utils");
const linkPreviews_1 = require("../util/linkPreviews");
const util_1 = require("../util");
const userProfileImageUpdates_1 = require("./userProfileImageUpdates");
const reactions_1 = require("../util/reactions");
function contentTypeSupported(type) {
    const Chrome = util_1.GoogleChrome;
    return Chrome.isImageTypeSupported(type) || Chrome.isVideoTypeSupported(type);
}
async function copyFromQuotedMessage(msg, quote) {
    if (!quote) {
        return;
    }
    const { attachments, id: quoteId, author } = quote;
    const quoteLocal = {
        attachments: attachments || null,
        author: author,
        id: lodash_1.default.toNumber(quoteId),
        text: null,
        referencedMessageNotFound: false,
    };
    const firstAttachment = attachments?.[0] || undefined;
    const id = lodash_1.default.toNumber(quoteId);
    const collection = await (0, data_1.getMessagesBySentAt)(id);
    const found = collection.find(message => {
        return Boolean(author === message.getSource());
    });
    if (!found) {
        window?.log?.warn(`We did not found quoted message ${id}.`);
        quoteLocal.referencedMessageNotFound = true;
        msg.set({ quote: quoteLocal });
        return;
    }
    window?.log?.info(`Found quoted message id: ${id}`);
    quoteLocal.referencedMessageNotFound = false;
    quoteLocal.text = (0, message_1.sliceQuoteText)(found.get('body') || '');
    if (!firstAttachment ||
        !firstAttachment.contentType ||
        !contentTypeSupported(firstAttachment.contentType)) {
        msg.set({ quote: quoteLocal });
        return;
    }
    firstAttachment.thumbnail = null;
    const queryAttachments = found.get('attachments') || [];
    if (queryAttachments.length > 0) {
        const queryFirst = queryAttachments[0];
        const { thumbnail } = queryFirst;
        if (thumbnail && thumbnail.path) {
            firstAttachment.thumbnail = {
                ...thumbnail,
                copied: true,
            };
        }
    }
    const queryPreview = found.get('preview') || [];
    if (queryPreview.length > 0) {
        const queryFirst = queryPreview[0];
        const { image } = queryFirst;
        if (image && image.path) {
            firstAttachment.thumbnail = {
                ...image,
                copied: true,
            };
        }
    }
    quoteLocal.attachments = [firstAttachment];
    msg.set({ quote: quoteLocal });
}
function handleLinkPreviews(messageBody, messagePreview, message) {
    const urls = linkPreviews_1.LinkPreviews.findLinks(messageBody);
    const incomingPreview = messagePreview || [];
    const preview = incomingPreview.filter((item) => (item.image || item.title) && urls.includes(item.url));
    if (preview.length < incomingPreview.length) {
        window?.log?.info(`${message.idForLogging()}: Eliminated ${preview.length -
            incomingPreview.length} previews with invalid urls'`);
    }
    message.set({ preview });
}
async function processProfileKeyNoCommit(conversation, sendingDeviceConversation, profileKeyBuffer) {
    if (conversation.isPrivate()) {
        await conversation.setProfileKey(profileKeyBuffer, false);
    }
    else {
        await sendingDeviceConversation.setProfileKey(profileKeyBuffer, false);
    }
}
function handleMentions(message, conversation, ourPrimaryNumber) {
    const body = message.get('body');
    if (body && body.indexOf(`@${ourPrimaryNumber.key}`) !== -1) {
        conversation.set({ mentionedUs: true });
    }
}
function updateReadStatus(message) {
    if (message.isExpirationTimerUpdate()) {
        message.set({ unread: 0 });
    }
}
function handleSyncedReceiptsNoCommit(message, conversation) {
    const sentTimestamp = message.get('sent_at');
    if (sentTimestamp) {
        conversation.markRead(sentTimestamp);
    }
}
function toRegularMessage(rawDataMessage) {
    return {
        ...lodash_1.default.pick(rawDataMessage, [
            'attachments',
            'preview',
            'body',
            'flags',
            'profileKey',
            'openGroupInvitation',
            'quote',
            'profile',
            'expireTimer',
            'payment',
            'sharedContact',
            'reaction',
        ]),
        isRegularMessage: true,
    };
}
exports.toRegularMessage = toRegularMessage;
async function handleRegularMessage(conversation, sendingDeviceConversation, message, rawDataMessage, source, messageHash) {
    const type = message.get('type');
    await copyFromQuotedMessage(message, rawDataMessage.quote);
    if (rawDataMessage.payment) {
        message.set({ payment: rawDataMessage.payment });
    }
    if (rawDataMessage.openGroupInvitation) {
        message.set({ groupInvitation: rawDataMessage.openGroupInvitation });
    }
    if (rawDataMessage.sharedContact) {
        message.set({ sharedContact: rawDataMessage.sharedContact });
    }
    handleLinkPreviews(rawDataMessage.body, rawDataMessage.preview, message);
    const existingExpireTimer = conversation.get('expireTimer');
    message.set({
        flags: rawDataMessage.flags,
        quote: rawDataMessage.quote,
        attachments: rawDataMessage.attachments,
        body: rawDataMessage.body,
        conversationId: conversation.id,
        messageHash,
        errors: [],
    });
    if (existingExpireTimer) {
        message.set({ expireTimer: existingExpireTimer });
    }
    const ourPubKey = utils_1.UserUtils.getOurPubKeyFromCache();
    handleMentions(message, conversation, ourPubKey);
    if (type === 'incoming') {
        if (conversation.isPrivate()) {
            updateReadStatus(message);
            if (conversation.isOutgoingRequest()) {
                await conversation.addIncomingApprovalMessage(lodash_1.default.toNumber(message.get('sent_at')) - 1, source);
            }
            await conversation.setDidApproveMe(true);
        }
    }
    else if (type === 'outgoing') {
        handleSyncedReceiptsNoCommit(message, conversation);
        if (conversation.isPrivate()) {
            await conversation.setIsApproved(true);
        }
    }
    const conversationActiveAt = conversation.get('active_at');
    if (!conversationActiveAt || (message.get('sent_at') || 0) > conversationActiveAt) {
        conversation.set({
            active_at: message.get('sent_at'),
            lastMessage: message.getNotificationText(),
        });
    }
    if (rawDataMessage.profileKey) {
        await processProfileKeyNoCommit(conversation, sendingDeviceConversation, rawDataMessage.profileKey);
    }
    await conversation.notifyTypingNoCommit({
        isTyping: false,
        sender: source,
    });
}
async function handleExpirationTimerUpdateNoCommit(conversation, message, source, expireTimer) {
    message.set({
        expirationTimerUpdate: {
            source,
            expireTimer,
        },
        unread: 0,
    });
    conversation.set({ expireTimer });
    await conversation.updateExpireTimer(expireTimer, source, message.get('received_at'), {}, false);
}
async function handleMessageJob(messageModel, conversation, regularDataMessage, confirm, source, messageHash) {
    window?.log?.info(`Starting handleMessageJob for message ${messageModel.idForLogging()}, ${messageModel.get('serverTimestamp') || messageModel.get('timestamp')} in conversation ${conversation.idForLogging()}`);
    if (!messageModel.get('isPublic') && regularDataMessage.reaction) {
        await (0, reactions_1.handleMessageReaction)(regularDataMessage.reaction, source, false, messageHash);
        if (regularDataMessage.reaction.action === 0 &&
            conversation.isPrivate() &&
            messageModel.get('unread')) {
            messageModel.set('reaction', regularDataMessage.reaction);
            conversation.throttledNotify(messageModel);
        }
        confirm?.();
    }
    else {
        const sendingDeviceConversation = await (0, conversations_1.getConversationController)().getOrCreateAndWait(source, conversation_1.ConversationTypeEnum.PRIVATE);
        try {
            messageModel.set({ flags: regularDataMessage.flags });
            if (messageModel.isExpirationTimerUpdate()) {
                const { expireTimer } = regularDataMessage;
                const oldValue = conversation.get('expireTimer');
                if (expireTimer === oldValue) {
                    confirm?.();
                    window?.log?.info('Dropping ExpireTimerUpdate message as we already have the same one set.');
                    return;
                }
                await handleExpirationTimerUpdateNoCommit(conversation, messageModel, source, expireTimer);
            }
            else {
                await handleRegularMessage(conversation, sendingDeviceConversation, messageModel, regularDataMessage, source, messageHash);
            }
            const id = await messageModel.commit();
            messageModel.set({ id });
            const unreadCount = await conversation.getUnreadCount();
            conversation.set({ unreadCount });
            conversation.updateLastMessage();
            await conversation.commit();
            if (conversation.id !== sendingDeviceConversation.id) {
                await sendingDeviceConversation.commit();
            }
            void (0, attachments_1.queueAttachmentDownloads)(messageModel, conversation);
            if (messageModel.isIncoming() && regularDataMessage.profile) {
                void (0, userProfileImageUpdates_1.appendFetchAvatarAndProfileJob)(sendingDeviceConversation, regularDataMessage.profile, regularDataMessage.profileKey);
            }
            if (messageModel.get('unread')) {
                conversation.throttledNotify(messageModel);
            }
            confirm?.();
        }
        catch (error) {
            const errorForLog = error && error.stack ? error.stack : error;
            window?.log?.error('handleMessageJob', messageModel.idForLogging(), 'error:', errorForLog);
        }
    }
}
exports.handleMessageJob = handleMessageJob;
