"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.reducer = exports.initialSearchState = exports.updateSearchTerm = exports.clearSearch = exports.search = exports.actions = void 0;
const cleanSearchTerm_1 = require("../../util/cleanSearchTerm");
const data_1 = require("../../../ts/data/data");
const types_1 = require("../../bchat/types");
const conversation_1 = require("../../models/conversation");
const lodash_1 = __importDefault(require("lodash"));
const conversations_1 = require("../../bchat/conversations");
const utils_1 = require("../../bchat/utils");
exports.actions = {
    search,
    clearSearch,
    updateSearchTerm,
};
function search(query) {
    return {
        type: 'SEARCH_RESULTS',
        payload: doSearch(query),
    };
}
exports.search = search;
async function doSearch(query) {
    const options = {
        noteToSelf: window.i18n('noteToSelf').toLowerCase(),
        ourNumber: utils_1.UserUtils.getOurPubKeyStrFromCache(),
    };
    const advancedSearchOptions = getAdvancedSearchOptionsFromQuery(query);
    const processedQuery = advancedSearchOptions.query;
    const [discussions, messages] = await Promise.all([
        queryConversationsAndContacts(processedQuery, options),
        queryMessages(processedQuery),
    ]);
    const { conversations, contacts } = discussions;
    const contactsAndGroups = lodash_1.default.uniq([...conversations, ...contacts]);
    const filteredMessages = lodash_1.default.compact(messages);
    return {
        query,
        normalizedPhoneNumber: types_1.PubKey.normalize(query),
        contactsAndGroups,
        messages: filteredMessages,
    };
}
function clearSearch() {
    return {
        type: 'SEARCH_CLEAR',
        payload: null,
    };
}
exports.clearSearch = clearSearch;
function updateSearchTerm(query) {
    return {
        type: 'SEARCH_UPDATE',
        payload: {
            query,
        },
    };
}
exports.updateSearchTerm = updateSearchTerm;
function getUnixMillisecondsTimestamp(timestamp) {
    const timestampInt = parseInt(timestamp, 10);
    if (!isNaN(timestampInt)) {
        try {
            if (timestampInt > 10000) {
                return new Date(timestampInt).getTime();
            }
            return new Date(timestamp).getTime();
        }
        catch (error) {
            window?.log?.warn('Advanced Search: ', error);
            return 0;
        }
    }
    return 0;
}
function getAdvancedSearchOptionsFromQuery(query) {
    const filterSeperator = ':';
    const filters = {
        query: null,
        from: null,
        before: null,
        after: null,
    };
    let newQuery = query;
    const splitQuery = query.toLowerCase().split(' ');
    const filtersList = Object.keys(filters);
    for (const queryPart of splitQuery) {
        for (const filter of filtersList) {
            const filterMatcher = filter + filterSeperator;
            if (queryPart.startsWith(filterMatcher)) {
                filters[filter] = queryPart.replace(filterMatcher, '');
                newQuery = newQuery.replace(queryPart, '').trim();
            }
        }
    }
    filters.before = getUnixMillisecondsTimestamp(filters.before);
    filters.after = getUnixMillisecondsTimestamp(filters.after);
    filters.query = newQuery;
    return filters;
}
async function queryMessages(query) {
    try {
        const trimmedQuery = query.trim();
        const normalized = (0, cleanSearchTerm_1.cleanSearchTerm)(trimmedQuery);
        const limit = Math.min((trimmedQuery.length || 2) * 50, 200);
        return (0, data_1.searchMessages)(normalized, limit);
    }
    catch (e) {
        window.log.warn('queryMessages failed with', e.message);
        return [];
    }
}
async function queryConversationsAndContacts(providedQuery, options) {
    const { ourNumber, noteToSelf } = options;
    const query = providedQuery.replace(/[+-.()]*/g, '');
    const searchResults = await (0, data_1.searchConversations)(query);
    let conversations = [];
    let contacts = [];
    const max = searchResults.length;
    for (let i = 0; i < max; i += 1) {
        const conversation = searchResults[i];
        if (conversation.id && conversation.activeAt) {
            if (conversation.id === ourNumber) {
                conversations.push(ourNumber);
            }
            else {
                conversations.push(conversation.id);
            }
        }
        else if (conversation.type === conversation_1.ConversationTypeEnum.PRIVATE) {
            contacts.push(conversation.id);
        }
        else if (conversation.type !== conversation_1.ConversationTypeEnum.GROUP) {
            contacts.push(conversation.id);
        }
        else {
            conversations.push(conversation.id);
        }
    }
    if (noteToSelf.indexOf(providedQuery.toLowerCase()) !== -1) {
        const ourConvo = (0, conversations_1.getConversationController)().get(ourNumber);
        if (ourConvo && ourConvo.isActive()) {
            contacts = contacts.filter(id => id !== ourNumber);
            conversations = conversations.filter(id => id !== ourNumber);
            contacts.unshift(ourNumber);
        }
    }
    return { conversations, contacts };
}
exports.initialSearchState = {
    query: '',
    contactsAndGroups: [],
    messages: [],
};
function getEmptyState() {
    return exports.initialSearchState;
}
function reducer(state, action) {
    if (!state) {
        return getEmptyState();
    }
    if (action.type === 'SEARCH_CLEAR') {
        return getEmptyState();
    }
    if (action.type === 'SEARCH_UPDATE') {
        const { payload } = action;
        const { query } = payload;
        return {
            ...state,
            query,
        };
    }
    if (action.type === 'SEARCH_RESULTS_FULFILLED') {
        const { payload } = action;
        const { query, normalizedPhoneNumber, contactsAndGroups, messages } = payload;
        if (state.query !== query) {
            return state;
        }
        return {
            ...state,
            query,
            normalizedPhoneNumber,
            contactsAndGroups,
            messages,
        };
    }
    return state;
}
exports.reducer = reducer;
