"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const protobuf_1 = require("../../../../protobuf");
const util_1 = require("util");
const lodash_1 = require("lodash");
const bchat_1 = require("../../../../bchat");
const VisibleMessage_1 = require("../../../../bchat/messages/outgoing/visibleMessage/VisibleMessage");
describe('VisibleMessage', () => {
    it('can create empty message with just a timestamp', () => {
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        (0, chai_1.expect)(decoded).to.have.not.property('dataMessage', null);
        (0, chai_1.expect)(decoded).to.have.not.property('dataMessage', undefined);
    });
    it('can create message with a body', () => {
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            body: 'body',
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        (0, chai_1.expect)(decoded.dataMessage).to.have.deep.property('body', 'body');
    });
    it('can create message with a expire timer', () => {
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            expireTimer: 3600,
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        (0, chai_1.expect)(decoded.dataMessage).to.have.deep.property('expireTimer', 3600);
    });
    it('can create message with a full bchat profile', () => {
        const profileKey = new util_1.TextEncoder().encode('profileKey');
        const bchatProfile = {
            displayName: 'displayName',
            avatarPointer: 'avatarPointer',
            profileKey,
        };
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            lokiProfile: bchatProfile,
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        (0, chai_1.expect)(decoded.dataMessage).to.have.deep.property('profile');
        (0, chai_1.expect)(decoded.dataMessage)
            .to.have.property('profile')
            .to.have.deep.property('displayName', 'displayName');
        (0, chai_1.expect)(decoded.dataMessage)
            .to.have.property('profile')
            .to.have.deep.property('profilePicture', 'avatarPointer');
        (0, chai_1.expect)(decoded.dataMessage).to.have.deep.property('profileKey', profileKey);
    });
    it('can create message with a quote without attachments', () => {
        let quote;
        quote = { id: 1234, author: 'author', text: 'text' };
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            quote,
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        const decodedID = (0, lodash_1.toNumber)(decoded.dataMessage?.quote?.id);
        (0, chai_1.expect)(decodedID).to.be.equal(1234);
        (0, chai_1.expect)(decoded.dataMessage?.quote).to.have.deep.property('author', 'author');
        (0, chai_1.expect)(decoded.dataMessage?.quote).to.have.deep.property('text', 'text');
    });
    it('can create message with a preview', () => {
        let preview;
        preview = { url: 'url', title: 'title' };
        const previews = new Array();
        previews.push(preview);
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            preview: previews,
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        (0, chai_1.expect)(decoded.dataMessage?.preview).to.have.lengthOf(1);
        (0, chai_1.expect)(decoded.dataMessage)
            .to.have.nested.property('preview[0].url')
            .to.be.deep.equal('url');
        (0, chai_1.expect)(decoded.dataMessage)
            .to.have.nested.property('preview[0].title')
            .to.be.deep.equal('title');
    });
    it('can create message with an AttachmentPointer', () => {
        let attachment;
        attachment = { url: 'url', contentType: 'contentType', id: 1234 };
        const attachments = new Array();
        attachments.push(attachment);
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
            attachments: attachments,
        });
        const plainText = message.plainTextBuffer();
        const decoded = protobuf_1.SignalService.Content.decode(plainText);
        (0, chai_1.expect)(decoded.dataMessage?.attachments).to.have.lengthOf(1);
        const firstAttachment = decoded?.dataMessage?.attachments?.[0];
        const decodedID = (0, lodash_1.toNumber)(firstAttachment?.id);
        (0, chai_1.expect)(decodedID).to.be.equal(1234);
        (0, chai_1.expect)(firstAttachment?.contentType).to.be.deep.equal('contentType');
        (0, chai_1.expect)(firstAttachment?.url).to.be.deep.equal('url');
    });
    it('correct ttl', () => {
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
        });
        (0, chai_1.expect)(message.ttl()).to.equal(bchat_1.Constants.TTL_DEFAULT.TTL_MAX);
    });
    it('has an identifier', () => {
        const message = new VisibleMessage_1.VisibleMessage({
            timestamp: Date.now(),
        });
        (0, chai_1.expect)(message.identifier).to.not.equal(null, 'identifier cannot be null');
        (0, chai_1.expect)(message.identifier).to.not.equal(undefined, 'identifier cannot be undefined');
    });
});
