"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const keypairs_1 = require("../../../../receiver/keypairs");
const constants_1 = require("../../../../bchat/constants");
const ConfigurationMessage_1 = require("../../../../bchat/messages/outgoing/controlMessage/ConfigurationMessage");
const test_utils_1 = require("../../../test-utils");
describe('ConfigurationMessage', () => {
    it('throw if closed group is not set', () => {
        const activeClosedGroups = null;
        const params = {
            activeClosedGroups,
            activeOpenGroups: [],
            timestamp: Date.now(),
            displayName: 'displayName',
            contacts: [],
        };
        (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessage(params)).to.throw('closed group must be set');
    });
    it('throw if Social group is not set', () => {
        const activeOpenGroups = null;
        const params = {
            activeClosedGroups: [],
            activeOpenGroups,
            timestamp: Date.now(),
            displayName: 'displayName',
            contacts: [],
        };
        (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessage(params)).to.throw('Social group must be set');
    });
    it('throw if display name is not set', () => {
        const params = {
            activeClosedGroups: [],
            activeOpenGroups: [],
            timestamp: Date.now(),
            displayName: undefined,
            contacts: [],
        };
        (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessage(params)).to.throw('displayName must be set');
    });
    it('throw if display name is set but empty', () => {
        const params = {
            activeClosedGroups: [],
            activeOpenGroups: [],
            timestamp: Date.now(),
            displayName: undefined,
            contacts: [],
        };
        (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessage(params)).to.throw('displayName must be set');
    });
    it('ttl is 4 days', () => {
        const params = {
            activeClosedGroups: [],
            activeOpenGroups: [],
            timestamp: Date.now(),
            displayName: 'displayName',
            contacts: [],
        };
        const configMessage = new ConfigurationMessage_1.ConfigurationMessage(params);
        (0, chai_1.expect)(configMessage.ttl()).to.be.equal(constants_1.TTL_DEFAULT.TTL_MAX);
    });
    describe('ConfigurationMessageClosedGroup', () => {
        it('throw if closed group has no encryptionkeypair', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                name: 'groupname',
                members: [member],
                admins: [member],
                encryptionKeyPair: undefined,
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw('Encryption key pair looks invalid');
        });
        it('throw if closed group has invalid encryptionkeypair', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                name: 'groupname',
                members: [member],
                admins: [member],
                encryptionKeyPair: new keypairs_1.ECKeyPair(new Uint8Array(), new Uint8Array()),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw('Encryption key pair looks invalid');
        });
        it('throw if closed group has invalid pubkey', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: 'invalidpubkey',
                name: 'groupname',
                members: [member],
                admins: [member],
                encryptionKeyPair: test_utils_1.TestUtils.generateFakeECKeyPair(),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw();
        });
        it('throw if closed group has invalid name', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                name: '',
                members: [member],
                admins: [member],
                encryptionKeyPair: test_utils_1.TestUtils.generateFakeECKeyPair(),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw('name must be set');
        });
        it('throw if members is empty', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                name: 'groupname',
                members: [],
                admins: [member],
                encryptionKeyPair: test_utils_1.TestUtils.generateFakeECKeyPair(),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw('members must be set');
        });
        it('throw if admins is empty', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                name: 'groupname',
                members: [member],
                admins: [],
                encryptionKeyPair: test_utils_1.TestUtils.generateFakeECKeyPair(),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw('admins must be set');
        });
        it('throw if some admins are not members', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const admin = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                name: 'groupname',
                members: [member],
                admins: [admin],
                encryptionKeyPair: test_utils_1.TestUtils.generateFakeECKeyPair(),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw('some admins are not members');
        });
    });
    describe('ConfigurationMessageContact', () => {
        it('throws if contacts is not set', () => {
            const params = {
                activeClosedGroups: [],
                activeOpenGroups: [],
                timestamp: Date.now(),
                displayName: 'displayName',
                contacts: undefined,
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessage(params)).to.throw('contacts must be set');
        });
        it('throw if some admins are not members', () => {
            const member = test_utils_1.TestUtils.generateFakePubKey().key;
            const admin = test_utils_1.TestUtils.generateFakePubKey().key;
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                name: 'groupname',
                members: [member],
                admins: [admin],
                encryptionKeyPair: test_utils_1.TestUtils.generateFakeECKeyPair(),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageClosedGroup(params)).to.throw('some admins are not members');
        });
        it('throw if the contact has not a valid pubkey', () => {
            const params = {
                publicKey: 'bd',
                displayName: 'contactDisplayName',
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageContact(params)).to.throw();
            const params2 = {
                publicKey: undefined,
                displayName: 'contactDisplayName',
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageContact(params2)).to.throw();
        });
        it('throw if the contact has an empty display name', () => {
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageContact(params2)).to.throw();
            const params2 = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                displayName: '',
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageContact(params2)).to.throw();
        });
        it('throw if the contact has a profileAvatar set but empty', () => {
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                displayName: 'contactDisplayName',
                profilePictureURL: '',
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageContact(params)).to.throw('profilePictureURL must either undefined or not empty');
        });
        it('throw if the contact has a profileKey set but empty', () => {
            const params = {
                publicKey: test_utils_1.TestUtils.generateFakePubKey().key,
                displayName: 'contactDisplayName',
                profileKey: new Uint8Array(),
            };
            (0, chai_1.expect)(() => new ConfigurationMessage_1.ConfigurationMessageContact(params)).to.throw('profileKey must either undefined or not empty');
        });
    });
});
