"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const crypto = __importStar(require("crypto"));
const sinon_1 = __importStar(require("sinon")), sinon = sinon_1;
const sending_1 = require("../../../../bchat/sending");
const test_utils_1 = require("../../../test-utils");
const crypto_1 = require("../../../../bchat/crypto");
const protobuf_1 = require("../../../../protobuf");
const utils_1 = require("../../../../bchat/utils");
const opengroupV2_1 = require("../../../../bchat/apis/open_group_api/opengroupV2");
const Data = __importStar(require("../../../../data/data"));
const snode_api_1 = require("../../../../bchat/apis/snode_api");
const lodash_1 = __importDefault(require("lodash"));
describe('MessageSender', () => {
    afterEach(() => {
        sinon.restore();
    });
    beforeEach(() => {
        test_utils_1.TestUtils.stubWindowLog();
    });
    describe('send', () => {
        const ourNumber = '0123456789abcdef';
        let bchatMessageAPISendStub;
        let encryptStub;
        beforeEach(() => {
            bchatMessageAPISendStub = sinon_1.default.stub(sending_1.MessageSender, 'sendMessageToSnode').resolves();
            sinon_1.default.stub(Data, 'getMessageById').resolves();
            encryptStub = sinon_1.default.stub(crypto_1.MessageEncrypter, 'encrypt').resolves({
                envelopeType: protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE,
                cipherText: crypto.randomBytes(10),
            });
            sinon_1.default.stub(utils_1.UserUtils, 'getOurPubKeyStrFromCache').returns(ourNumber);
        });
        describe('retry', () => {
            let rawMessage;
            beforeEach(async () => {
                rawMessage = await utils_1.MessageUtils.toRawMessage(test_utils_1.TestUtils.generateFakePubKey(), test_utils_1.TestUtils.generateVisibleMessage());
            });
            it('should not retry if an error occurred during encryption', async () => {
                encryptStub.throws(new Error('Failed to encrypt.'));
                const promise = sending_1.MessageSender.send(rawMessage, 3, 10);
                await (0, chai_1.expect)(promise).is.rejectedWith('Failed to encrypt.');
                (0, chai_1.expect)(bchatMessageAPISendStub.callCount).to.equal(0);
            });
            it('should only call bchatMessageAPI once if no errors occured', async () => {
                await sending_1.MessageSender.send(rawMessage, 3, 10);
                (0, chai_1.expect)(bchatMessageAPISendStub.callCount).to.equal(1);
            });
            it('should only retry the specified amount of times before throwing', async () => {
                bchatMessageAPISendStub.throws(new Error('API error'));
                const attempts = 2;
                const promise = sending_1.MessageSender.send(rawMessage, attempts, 10);
                await (0, chai_1.expect)(promise).is.rejectedWith('API error');
                (0, chai_1.expect)(bchatMessageAPISendStub.callCount).to.equal(attempts);
            });
            it('should not throw error if successful send occurs within the retry limit', async () => {
                bchatMessageAPISendStub.onFirstCall().throws(new Error('API error'));
                await sending_1.MessageSender.send(rawMessage, 3, 10);
                (0, chai_1.expect)(bchatMessageAPISendStub.callCount).to.equal(2);
            });
        });
        describe('logic', () => {
            let messageEncyrptReturnEnvelopeType = protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE;
            beforeEach(() => {
                encryptStub.callsFake(async (_device, plainTextBuffer, _type) => ({
                    envelopeType: messageEncyrptReturnEnvelopeType,
                    cipherText: plainTextBuffer,
                }));
            });
            it('should pass the correct values to bchatMessageAPI', async () => {
                const device = test_utils_1.TestUtils.generateFakePubKey();
                const visibleMessage = test_utils_1.TestUtils.generateVisibleMessage();
                const rawMessage = await utils_1.MessageUtils.toRawMessage(device, visibleMessage);
                await sending_1.MessageSender.send(rawMessage, 3, 10);
                const args = bchatMessageAPISendStub.getCall(0).args;
                (0, chai_1.expect)(args[0]).to.equal(device.key);
                (0, chai_1.expect)(args[2]).to.equal(visibleMessage.ttl());
            });
            it('should correctly build the envelope and override the timestamp', async () => {
                messageEncyrptReturnEnvelopeType = protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE;
                const device = test_utils_1.TestUtils.generateFakePubKey();
                const visibleMessage = test_utils_1.TestUtils.generateVisibleMessage();
                const rawMessage = await utils_1.MessageUtils.toRawMessage(device, visibleMessage);
                const offset = 200000;
                sinon_1.default.stub(snode_api_1.SNodeAPI, 'getLatestTimestampOffset').returns(offset);
                await sending_1.MessageSender.send(rawMessage, 3, 10);
                const data = bchatMessageAPISendStub.getCall(0).args[1];
                const webSocketMessage = protobuf_1.SignalService.WebSocketMessage.decode(data);
                (0, chai_1.expect)(webSocketMessage.request?.body).to.not.equal(undefined, 'Request body should not be undefined');
                (0, chai_1.expect)(webSocketMessage.request?.body).to.not.equal(null, 'Request body should not be null');
                const envelope = protobuf_1.SignalService.Envelope.decode(webSocketMessage.request?.body);
                (0, chai_1.expect)(envelope.type).to.equal(protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
                (0, chai_1.expect)(envelope.source).to.equal('');
                const expectedTimestamp = Date.now() - offset;
                const decodedTimestampFromSending = lodash_1.default.toNumber(envelope.timestamp);
                (0, chai_1.expect)(decodedTimestampFromSending).to.be.above(expectedTimestamp - 10);
                (0, chai_1.expect)(decodedTimestampFromSending).to.be.below(expectedTimestamp + 10);
                const visibleMessageExpected = test_utils_1.TestUtils.generateVisibleMessage({
                    timestamp: decodedTimestampFromSending,
                });
                const rawMessageExpected = await utils_1.MessageUtils.toRawMessage(device, visibleMessageExpected);
                (0, chai_1.expect)(envelope.content).to.deep.equal(rawMessageExpected.plainTextBuffer);
            });
            describe('BCHAT_MESSAGE', () => {
                it('should set the envelope source to be empty', async () => {
                    messageEncyrptReturnEnvelopeType = protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE;
                    const device = test_utils_1.TestUtils.generateFakePubKey();
                    const visibleMessage = test_utils_1.TestUtils.generateVisibleMessage();
                    const rawMessage = await utils_1.MessageUtils.toRawMessage(device, visibleMessage);
                    await sending_1.MessageSender.send(rawMessage, 3, 10);
                    const data = bchatMessageAPISendStub.getCall(0).args[1];
                    const webSocketMessage = protobuf_1.SignalService.WebSocketMessage.decode(data);
                    (0, chai_1.expect)(webSocketMessage.request?.body).to.not.equal(undefined, 'Request body should not be undefined');
                    (0, chai_1.expect)(webSocketMessage.request?.body).to.not.equal(null, 'Request body should not be null');
                    const envelope = protobuf_1.SignalService.Envelope.decode(webSocketMessage.request?.body);
                    (0, chai_1.expect)(envelope.type).to.equal(protobuf_1.SignalService.Envelope.Type.BCHAT_MESSAGE);
                    (0, chai_1.expect)(envelope.source).to.equal('', 'envelope source should be empty in BCHAT_MESSAGE');
                });
            });
        });
    });
    describe('sendToOpenGroupV2', () => {
        let postMessageRetryableStub;
        beforeEach(() => {
            sinon_1.default.stub(utils_1.UserUtils, 'getOurPubKeyStrFromCache').resolves(test_utils_1.TestUtils.generateFakePubKey().key);
            postMessageRetryableStub = sinon_1.default.stub(opengroupV2_1.ApiV2, 'postMessageRetryable').resolves(test_utils_1.TestUtils.generateOpenGroupMessageV2());
        });
        afterEach(() => {
            sinon_1.default.restore();
        });
        it('should call postMessageRetryableStub', async () => {
            const message = test_utils_1.TestUtils.generateOpenGroupVisibleMessage();
            const roomInfos = test_utils_1.TestUtils.generateOpenGroupV2RoomInfos();
            await sending_1.MessageSender.sendToOpenGroupV2(message, roomInfos);
            (0, chai_1.expect)(postMessageRetryableStub.callCount).to.eq(1);
        });
        it('should retry postMessageRetryableStub ', async () => {
            const message = test_utils_1.TestUtils.generateOpenGroupVisibleMessage();
            const roomInfos = test_utils_1.TestUtils.generateOpenGroupV2RoomInfos();
            postMessageRetryableStub.throws('whate');
            sinon_1.default.stub(opengroupV2_1.ApiV2, 'getMinTimeout').returns(2);
            postMessageRetryableStub.onThirdCall().resolves();
            await sending_1.MessageSender.sendToOpenGroupV2(message, roomInfos);
            (0, chai_1.expect)(postMessageRetryableStub.callCount).to.eq(3);
        });
        it('should not retry more than 3 postMessageRetryableStub ', async () => {
            const message = test_utils_1.TestUtils.generateOpenGroupVisibleMessage();
            const roomInfos = test_utils_1.TestUtils.generateOpenGroupV2RoomInfos();
            sinon_1.default.stub(opengroupV2_1.ApiV2, 'getMinTimeout').returns(2);
            postMessageRetryableStub.throws('fake error');
            postMessageRetryableStub.onCall(4).resolves();
            try {
                await sending_1.MessageSender.sendToOpenGroupV2(message, roomInfos);
                throw new Error('Error expected');
            }
            catch (e) {
                (0, chai_1.expect)(e.name).to.eq('fake error');
            }
            (0, chai_1.expect)(postMessageRetryableStub.calledThrice);
        });
    });
});
