"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const sinon_1 = __importDefault(require("sinon"));
const _ = __importStar(require("lodash"));
const utils_1 = require("../../../../bchat/utils");
const test_utils_1 = require("../../../test-utils");
const PendingMessageCache_1 = require("../../../../bchat/sending/PendingMessageCache");
describe('PendingMessageCache', () => {
    let data;
    let pendingMessageCacheStub;
    beforeEach(() => {
        const storageID = 'pendingMessages';
        data = {
            id: storageID,
            value: '[]',
        };
        test_utils_1.TestUtils.stubDataItem('getItemById')
            .withArgs('pendingMessages')
            .callsFake(async () => {
            return data;
        });
        test_utils_1.TestUtils.stubDataItem('createOrUpdateItem').callsFake((item) => {
            if (item.id === storageID) {
                data = item;
            }
        });
        pendingMessageCacheStub = new PendingMessageCache_1.PendingMessageCache();
    });
    afterEach(() => {
        sinon_1.default.restore();
    });
    it('can initialize cache', async () => {
        const cache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(cache).to.be.instanceOf(Array);
        (0, chai_1.expect)(cache).to.have.length(0);
    });
    it('can add to cache', async () => {
        const device = test_utils_1.TestUtils.generateFakePubKey();
        const message = test_utils_1.TestUtils.generateVisibleMessage();
        const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
        await pendingMessageCacheStub.add(device, message);
        const finalCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(finalCache).to.have.length(1);
        const addedMessage = finalCache[0];
        (0, chai_1.expect)(addedMessage.device).to.deep.equal(rawMessage.device);
    });
    it('can add multiple messages belonging to the same user', async () => {
        const device = test_utils_1.TestUtils.generateFakePubKey();
        await pendingMessageCacheStub.add(device, test_utils_1.TestUtils.generateVisibleMessage());
        await test_utils_1.TestUtils.timeout(5);
        await pendingMessageCacheStub.add(device, test_utils_1.TestUtils.generateVisibleMessage());
        await test_utils_1.TestUtils.timeout(5);
        await pendingMessageCacheStub.add(device, test_utils_1.TestUtils.generateVisibleMessage());
        const finalCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(finalCache).to.have.length(3);
    });
    it('can remove from cache', async () => {
        const device = test_utils_1.TestUtils.generateFakePubKey();
        const message = test_utils_1.TestUtils.generateVisibleMessage();
        const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
        await pendingMessageCacheStub.add(device, message);
        const initialCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(initialCache).to.have.length(1);
        await pendingMessageCacheStub.remove(rawMessage);
        const finalCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(finalCache).to.have.length(0);
    });
    it('should only remove messages with different identifier and device', async () => {
        const device = test_utils_1.TestUtils.generateFakePubKey();
        const message = test_utils_1.TestUtils.generateVisibleMessage();
        const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
        await pendingMessageCacheStub.add(device, message);
        await test_utils_1.TestUtils.timeout(5);
        const one = await pendingMessageCacheStub.add(device, test_utils_1.TestUtils.generateVisibleMessage());
        const two = await pendingMessageCacheStub.add(test_utils_1.TestUtils.generateFakePubKey(), message);
        const initialCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(initialCache).to.have.length(3);
        await pendingMessageCacheStub.remove(rawMessage);
        const finalCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(finalCache).to.have.length(2);
        (0, chai_1.expect)(finalCache).to.have.deep.members([one, two]);
    });
    it('can get devices', async () => {
        const cacheItems = [
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
        ];
        for (const item of cacheItems) {
            await pendingMessageCacheStub.add(item.device, item.message);
        }
        const cache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(cache).to.have.length(cacheItems.length);
        const devicesKeys = cacheItems.map(item => item.device.key);
        const pulledDevices = await pendingMessageCacheStub.getDevices();
        const pulledDevicesKeys = pulledDevices.map(d => d.key);
        (0, chai_1.expect)(pulledDevicesKeys).to.have.members(devicesKeys);
    });
    it('can get pending for device', async () => {
        const cacheItems = [
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
        ];
        for (const item of cacheItems) {
            await pendingMessageCacheStub.add(item.device, item.message);
        }
        const initialCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(initialCache).to.have.length(cacheItems.length);
        for (const item of cacheItems) {
            const pendingForDevice = await pendingMessageCacheStub.getForDevice(item.device);
            (0, chai_1.expect)(pendingForDevice).to.have.length(1);
            (0, chai_1.expect)(pendingForDevice[0].device).to.equal(item.device.key);
        }
    });
    it('can find nothing when empty', async () => {
        const device = test_utils_1.TestUtils.generateFakePubKey();
        const message = test_utils_1.TestUtils.generateVisibleMessage();
        const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
        const foundMessage = pendingMessageCacheStub.find(rawMessage);
        (0, chai_1.expect)(foundMessage, 'a message was found in empty cache').to.be.undefined;
    });
    it('can find message in cache', async () => {
        const device = test_utils_1.TestUtils.generateFakePubKey();
        const message = test_utils_1.TestUtils.generateVisibleMessage();
        const rawMessage = await utils_1.MessageUtils.toRawMessage(device, message);
        await pendingMessageCacheStub.add(device, message);
        const finalCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(finalCache).to.have.length(1);
        const foundMessage = pendingMessageCacheStub.find(rawMessage);
        (0, chai_1.expect)(foundMessage, 'message not found in cache').to.be.ok;
        foundMessage && (0, chai_1.expect)(foundMessage.device).to.equal(device.key);
    });
    it('can clear cache', async () => {
        const cacheItems = [
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
        ];
        for (const item of cacheItems) {
            await pendingMessageCacheStub.add(item.device, item.message);
        }
        const initialCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(initialCache).to.have.length(cacheItems.length);
        await pendingMessageCacheStub.clear();
        const finalCache = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(finalCache).to.have.length(0);
    });
    it('can restore from db', async () => {
        const cacheItems = [
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
            {
                device: test_utils_1.TestUtils.generateFakePubKey(),
                message: test_utils_1.TestUtils.generateVisibleMessage(),
            },
        ];
        for (const item of cacheItems) {
            await pendingMessageCacheStub.add(item.device, item.message);
        }
        const addedMessages = await pendingMessageCacheStub.getAllPending();
        (0, chai_1.expect)(addedMessages).to.have.length(cacheItems.length);
        const freshCache = new PendingMessageCache_1.PendingMessageCache();
        const rebuiltMessages = await freshCache.getAllPending();
        for (const [index, message] of rebuiltMessages.entries()) {
            const addedMessage = addedMessages[index];
            const buffersCompare = Buffer.compare(message.plainTextBuffer, addedMessage.plainTextBuffer) === 0;
            (0, chai_1.expect)(buffersCompare).to.equal(true, 'buffers were not loaded properly from database');
            const trimmedAdded = _.omit(addedMessage, ['plainTextBuffer']);
            const trimmedRebuilt = _.omit(message, ['plainTextBuffer']);
            (0, chai_1.expect)(_.isEqual(trimmedAdded, trimmedRebuilt)).to.equal(true, 'cached messages were not rebuilt properly');
        }
    });
});
