"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const moment_1 = __importDefault(require("moment"));
const Attachment = __importStar(require("../../types/Attachment"));
const MIME = __importStar(require("../../types/MIME"));
const protobuf_1 = require("../../protobuf");
const stringToArrayBuffer = (str) => {
    if (typeof str !== 'string') {
        throw new TypeError("'string' must be a string");
    }
    const array = new Uint8Array(str.length);
    for (let i = 0; i < str.length; i += 1) {
        array[i] = str.charCodeAt(i);
    }
    return array.buffer;
};
describe('Attachment', () => {
    describe('getFileExtension', () => {
        it('should return file extension from content type', () => {
            const input = {
                fileName: 'funny-cat.mov',
                url: 'funny-cat.mov',
                contentType: MIME.IMAGE_GIF,
                fileSize: null,
                screenshot: null,
                thumbnail: null,
            };
            chai_1.assert.strictEqual(Attachment.getFileExtension(input), 'gif');
        });
        it('should return file extension for QuickTime videos', () => {
            const input = {
                fileName: 'funny-cat.mov',
                url: 'funny-cat.mov',
                contentType: MIME.VIDEO_QUICKTIME,
                fileSize: null,
                screenshot: null,
                thumbnail: null,
            };
            chai_1.assert.strictEqual(Attachment.getFileExtension(input), 'mov');
        });
        it('should return file extension for application files', () => {
            const input = {
                fileName: 'funny-cat.odt',
                url: 'funny-cat.odt',
                contentType: MIME.ODT,
                fileSize: null,
                screenshot: null,
                thumbnail: null,
            };
            chai_1.assert.strictEqual(Attachment.getFileExtension(input), 'odt');
        });
    });
    describe('getSuggestedFilename', () => {
        context('for attachment with filename', () => {
            it('should generate a filename without timestamp', () => {
                const attachment = {
                    fileName: 'funny-cat.mov',
                    url: 'funny-cat.mov',
                    contentType: MIME.VIDEO_QUICKTIME,
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const actual = Attachment.getSuggestedFilename({ attachment });
                const expected = 'funny-cat.mov';
                chai_1.assert.strictEqual(actual, expected);
            });
            it('should generate a filename without timestamp but with an index', () => {
                const attachment = {
                    fileName: 'funny-cat.mov',
                    url: 'funny-cat.mov',
                    contentType: MIME.VIDEO_QUICKTIME,
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const actual = Attachment.getSuggestedFilename({
                    attachment,
                    index: 3,
                });
                const expected = 'funny-cat.mov';
                chai_1.assert.strictEqual(actual, expected);
            });
            it('should generate a filename with an extension if contentType is not setup', () => {
                const attachment = {
                    fileName: 'funny-cat.ini',
                    url: 'funny-cat.ini',
                    contentType: '',
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const actual = Attachment.getSuggestedFilename({
                    attachment,
                    index: 3,
                });
                const expected = 'funny-cat.ini';
                chai_1.assert.strictEqual(actual, expected);
            });
            it('should generate a filename with an extension if contentType is text/plain', () => {
                const attachment = {
                    fileName: 'funny-cat.txt',
                    url: 'funny-cat.txt',
                    contentType: 'text/plain',
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const actual = Attachment.getSuggestedFilename({
                    attachment,
                    index: 3,
                });
                const expected = 'funny-cat.txt';
                chai_1.assert.strictEqual(actual, expected);
            });
            it('should generate a filename with an extension if contentType is json', () => {
                const attachment = {
                    fileName: 'funny-cat.json',
                    url: 'funny-cat.json',
                    contentType: '',
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const actual = Attachment.getSuggestedFilename({
                    attachment,
                    index: 3,
                });
                const expected = 'funny-cat.json';
                chai_1.assert.strictEqual(actual, expected);
            });
        });
        context('for attachment without filename', () => {
            it('should generate a filename based on timestamp', () => {
                const attachment = {
                    contentType: MIME.VIDEO_QUICKTIME,
                    url: 'funny-cat.mov',
                    fileName: 'funny-cat.mov',
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const timestamp = (0, moment_1.default)('2000-01-01').toDate();
                const actual = Attachment.getSuggestedFilename({
                    attachment,
                    timestamp,
                });
                const expected = 'funny-cat.mov';
                chai_1.assert.strictEqual(actual, expected);
            });
        });
        context('for attachment with index', () => {
            it('should generate a filename based on timestamp if filename is not set', () => {
                const attachment = {
                    fileName: '',
                    url: 'funny-cat.mov',
                    contentType: MIME.VIDEO_QUICKTIME,
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const timestamp = new Date(new Date(0).getTimezoneOffset() * 60 * 1000);
                const actual = Attachment.getSuggestedFilename({
                    attachment,
                    timestamp,
                    index: 3,
                });
                const expected = 'bchat-attachment-1970-01-01-000000_003.mov';
                chai_1.assert.strictEqual(actual, expected);
            });
            it('should generate a filename based on filename if present', () => {
                const attachment = {
                    fileName: 'funny-cat.mov',
                    url: 'funny-cat.mov',
                    contentType: MIME.VIDEO_QUICKTIME,
                    fileSize: null,
                    screenshot: null,
                    thumbnail: null,
                };
                const timestamp = new Date(new Date(0).getTimezoneOffset() * 60 * 1000);
                const actual = Attachment.getSuggestedFilename({
                    attachment,
                    timestamp,
                    index: 3,
                });
                const expected = 'funny-cat.mov';
                chai_1.assert.strictEqual(actual, expected);
            });
        });
    });
    describe('isVisualMedia', () => {
        it('should return true for images', () => {
            const attachment = {
                fileName: 'meme.gif',
                data: stringToArrayBuffer('gif'),
                contentType: MIME.IMAGE_GIF,
            };
            chai_1.assert.isTrue(Attachment.isVisualMedia(attachment));
        });
        it('should return true for videos', () => {
            const attachment = {
                fileName: 'meme.mp4',
                data: stringToArrayBuffer('mp4'),
                contentType: MIME.VIDEO_MP4,
            };
            chai_1.assert.isTrue(Attachment.isVisualMedia(attachment));
        });
        it('should return false for voice message attachment', () => {
            const attachment = {
                fileName: 'Voice Message.aac',
                flags: protobuf_1.SignalService.AttachmentPointer.Flags.VOICE_MESSAGE,
                data: stringToArrayBuffer('voice message'),
                contentType: MIME.AUDIO_AAC,
            };
            chai_1.assert.isFalse(Attachment.isVisualMedia(attachment));
        });
        it('should return false for other attachments', () => {
            const attachment = {
                fileName: 'foo.json',
                data: stringToArrayBuffer('{"foo": "bar"}'),
                contentType: MIME.APPLICATION_JSON,
            };
            chai_1.assert.isFalse(Attachment.isVisualMedia(attachment));
        });
    });
    describe('isFile', () => {
        it('should return true for JSON', () => {
            const attachment = {
                fileName: 'foo.json',
                data: stringToArrayBuffer('{"foo": "bar"}'),
                contentType: MIME.APPLICATION_JSON,
            };
            chai_1.assert.isTrue(Attachment.isFile(attachment));
        });
        it('should return false for images', () => {
            const attachment = {
                fileName: 'meme.gif',
                data: stringToArrayBuffer('gif'),
                contentType: MIME.IMAGE_GIF,
            };
            chai_1.assert.isFalse(Attachment.isFile(attachment));
        });
        it('should return false for videos', () => {
            const attachment = {
                fileName: 'meme.mp4',
                data: stringToArrayBuffer('mp4'),
                contentType: MIME.VIDEO_MP4,
            };
            chai_1.assert.isFalse(Attachment.isFile(attachment));
        });
        it('should return false for voice message attachment', () => {
            const attachment = {
                fileName: 'Voice Message.aac',
                flags: protobuf_1.SignalService.AttachmentPointer.Flags.VOICE_MESSAGE,
                data: stringToArrayBuffer('voice message'),
                contentType: MIME.AUDIO_AAC,
            };
            chai_1.assert.isFalse(Attachment.isFile(attachment));
        });
    });
    describe('isVoiceMessage', () => {
        it('should return true for voice message attachment', () => {
            const attachment = {
                fileName: 'Voice Message.aac',
                flags: protobuf_1.SignalService.AttachmentPointer.Flags.VOICE_MESSAGE,
                data: stringToArrayBuffer('voice message'),
                contentType: MIME.AUDIO_AAC,
            };
            chai_1.assert.isTrue(Attachment.isVoiceMessage(attachment));
        });
        it('should return true for legacy Android voice message attachment', () => {
            const attachment = {
                data: stringToArrayBuffer('voice message'),
                contentType: MIME.AUDIO_MP3,
            };
            chai_1.assert.isTrue(Attachment.isVoiceMessage(attachment));
        });
        it('should return false for other attachments', () => {
            const attachment = {
                fileName: 'foo.gif',
                data: stringToArrayBuffer('foo'),
                contentType: MIME.IMAGE_GIF,
            };
            chai_1.assert.isFalse(Attachment.isVoiceMessage(attachment));
        });
    });
});
