"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.redactAll = void 0;
const fp_1 = require("lodash/fp");
const lodash_1 = require("lodash");
const getRootPath_1 = require("../node/getRootPath");
const APP_ROOT_PATH = (0, getRootPath_1.getAppRootPath)();
const BCHAT_ID_PATTERN = /\b((bd)?[0-9a-f]{64})\b/gi;
const SNODE_PATTERN = /(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)/g;
const GROUP_ID_PATTERN = /(group\()([^)]+)(\))/g;
const SERVER_URL_PATTERN = /https?:\/\/[-a-zA-Z0-9@:%._+~#=]{1,256}\.[a-zA-Z0-9()]{1,6}\b([-a-zA-Z0-9()@:%_+.~#?&//=]*)/g;
const REDACTION_PLACEHOLDER = '[REDACTED]';
const redactPath = (filePath) => {
    if (!filePath) {
        throw new TypeError("'filePath' must be a string");
    }
    const filePathPattern = _pathToRegExp(filePath);
    return (text) => {
        if (!(0, lodash_1.isString)(text)) {
            throw new TypeError("'text' must be a string");
        }
        if (!(0, lodash_1.isRegExp)(filePathPattern)) {
            return text;
        }
        return text.replace(filePathPattern, REDACTION_PLACEHOLDER);
    };
};
const _pathToRegExp = (filePath) => {
    try {
        const pathWithNormalizedSlashes = filePath.replace(/\//g, '\\');
        const pathWithEscapedSlashes = filePath.replace(/\\/g, '\\\\');
        const urlEncodedPath = encodeURI(filePath);
        const patternString = [
            filePath,
            pathWithNormalizedSlashes,
            pathWithEscapedSlashes,
            urlEncodedPath,
        ]
            .map(lodash_1.escapeRegExp)
            .join('|');
        return new RegExp(patternString, 'g');
    }
    catch (error) {
        return null;
    }
};
const redactBchatID = (text) => {
    if (!(0, lodash_1.isString)(text)) {
        throw new TypeError("'text' must be a string");
    }
    return text.replaceAll(BCHAT_ID_PATTERN, REDACTION_PLACEHOLDER);
};
const redactSnodeIP = (text) => {
    if (!(0, lodash_1.isString)(text)) {
        throw new TypeError("'text' must be a string");
    }
    return text.replaceAll(SNODE_PATTERN, REDACTION_PLACEHOLDER);
};
const redactServerUrl = (text) => {
    if (!(0, lodash_1.isString)(text)) {
        throw new TypeError("'text' must be a string");
    }
    return text.replaceAll(SERVER_URL_PATTERN, REDACTION_PLACEHOLDER);
};
const redactGroupIds = (text) => {
    if (!(0, lodash_1.isString)(text)) {
        throw new TypeError("'text' must be a string");
    }
    return text.replaceAll(GROUP_ID_PATTERN, (_match, before, id, after) => `${before}${REDACTION_PLACEHOLDER}${removeNewlines(id).slice(-3)}${after}`);
};
const removeNewlines = (text) => text.replace(/\r?\n|\r/g, '');
const redactSensitivePaths = redactPath(APP_ROOT_PATH);
const isDev = (process.env.NODE_APP_INSTANCE || '').startsWith('devprod');
exports.redactAll = !isDev
    ? (0, fp_1.compose)(redactSensitivePaths, redactGroupIds, redactBchatID, redactSnodeIP, redactServerUrl)
    : (text) => text;
