"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isUsAnySogsFromCache = exports.updateRecentReactions = exports.handleOpenGroupMessageReactions = exports.handleMessageReaction = exports.sendMessageReaction = void 0;
const utils_1 = require("../bchat/utils");
const storage_1 = require("./storage");
const protobuf_1 = require("../protobuf");
const data_1 = require("../data/data");
const lodash_1 = require("lodash");
const Reaction_1 = require("../types/Reaction");
const PubKey_1 = require("../bchat/types/PubKey");
let cachedKnownMapping = null;
const rateCountLimit = 20;
const rateTimeLimit = 60 * 1000;
const latestReactionTimestamps = [];
const getMessageByReaction = async (reaction, isOpenGroup) => {
    let originalMessage = null;
    const originalMessageId = Number(reaction.id);
    if (isOpenGroup) {
        originalMessage = await data_1.Data.getMessageByServerId(originalMessageId);
    }
    else {
        const collection = await data_1.Data.getMessagesBySentAt(originalMessageId);
        originalMessage = collection.find((item) => {
            const messageTimestamp = item.get('sent_at');
            return Boolean(messageTimestamp &&
                messageTimestamp === originalMessageId);
        });
    }
    if (!originalMessage) {
        window?.log?.warn(`Cannot find the original reacted message ${originalMessageId}.`);
        return null;
    }
    return originalMessage;
};
const sendMessageReaction = async (messageId, emoji) => {
    const found = await (0, data_1.getMessageById)(messageId);
    if (found) {
        const conversationModel = found?.getConversation();
        if (!conversationModel) {
            window.log.warn(`Conversation for ${messageId} not found in db`);
            return;
        }
        if (!conversationModel.hasReactions) {
            window.log.warn("This conversation doesn't have reaction support");
            return;
        }
        const timestamp = Date.now();
        latestReactionTimestamps.push(timestamp);
        if (latestReactionTimestamps.length > rateCountLimit) {
            const firstTimestamp = latestReactionTimestamps[0];
            if (timestamp - firstTimestamp < rateTimeLimit) {
                latestReactionTimestamps.pop();
                return;
            }
            else {
                latestReactionTimestamps.shift();
            }
        }
        const isOpenGroup = Boolean(found?.get('isPublic'));
        const id = (isOpenGroup && found.get('serverId')) || Number(found.get('sent_at'));
        const me = isOpenGroup || utils_1.UserUtils.getOurPubKeyStrFromCache();
        let action = 0;
        const reacts = found.get('reacts');
        const findEmojiForSender = (data, targetKey) => {
            for (const emoji in data) {
                const { senders } = data[emoji];
                if (targetKey in senders) {
                    return emoji;
                }
            }
            return null;
        };
        const result = reacts ? findEmojiForSender(reacts, me) : null;
        if (reacts &&
            Object.keys(reacts).includes(emoji) &&
            Object.keys(reacts[emoji].senders).includes(me)) {
            window.log.info('found matching reaction removing it');
            action = 1;
        }
        else if (result) {
            let reaction = {
                id,
                author: utils_1.UserUtils.getOurPubKeyStrFromCache(),
                emoji: result,
                action: 1,
            };
            window.log.info('found matching reaction removing it, limit user has react one reaction');
            await conversationModel.sendReaction(messageId, reaction);
        }
        const reactions = (0, storage_1.getRecentReactions)();
        if (reactions) {
            await (0, exports.updateRecentReactions)(reactions, emoji);
        }
        const reaction = {
            id,
            author: utils_1.UserUtils.getOurPubKeyStrFromCache(),
            emoji,
            action,
        };
        await conversationModel.sendReaction(messageId, reaction);
        window.log.info(`${action === 0 ? 'added' : 'removed'} `, emoji, 'reaction at', id);
        return reaction;
    }
    else {
        window.log.warn(`Message ${messageId} not found in db`);
        return;
    }
};
exports.sendMessageReaction = sendMessageReaction;
const handleMessageReaction = async (reaction, sender, isOpenGroup, messageId) => {
    window?.log?.warn(`reaction: DataMessage ID: ${messageId}.`);
    if (!reaction.emoji) {
        window?.log?.warn(`There is no emoji for the reaction ${messageId}.`);
        return;
    }
    const originalMessage = await getMessageByReaction(reaction, isOpenGroup);
    if (!originalMessage) {
        return;
    }
    if (originalMessage.get('isPublic')) {
        return;
    }
    const reacts = originalMessage.get('reacts') ?? {};
    reacts[reaction.emoji] = reacts[reaction.emoji] || { count: null, senders: {} };
    const details = reacts[reaction.emoji] ?? {};
    const senders = Object.keys(details.senders);
    window.log.info(`${sender} ${reaction.action === 0 ? 'added' : 'removed'} a ${reaction.emoji} reaction`);
    switch (reaction.action) {
        case protobuf_1.SignalService.DataMessage.Reaction.Action.REACT:
            if (senders.includes(sender) && details.senders[sender] !== '') {
                window?.log?.info('Received duplicate message reaction. Dropping it. id:', details.senders[sender]);
                return;
            }
            details.senders[sender] = messageId ?? '';
            break;
        case protobuf_1.SignalService.DataMessage.Reaction.Action.REMOVE:
        default:
            if (senders.length > 0) {
                if (senders.indexOf(sender) >= 0) {
                    delete details.senders[sender];
                }
            }
    }
    const count = Object.keys(details.senders).length;
    if (count > 0) {
        reacts[reaction.emoji].count = count;
        reacts[reaction.emoji].senders = details.senders;
        if (details && details.index === undefined) {
            reacts[reaction.emoji].index = originalMessage.get('reactsIndex') ?? 0;
            originalMessage.set('reactsIndex', (originalMessage.get('reactsIndex') ?? 0) + 1);
        }
    }
    else {
        delete reacts[reaction.emoji];
    }
    originalMessage.set({
        reacts: !(0, lodash_1.isEmpty)(reacts) ? reacts : undefined,
    });
    await originalMessage.commit();
    return originalMessage;
};
exports.handleMessageReaction = handleMessageReaction;
const handleOpenGroupMessageReactions = async (reactions, serverId) => {
    const originalMessage = await data_1.Data.getMessageByServerId(serverId);
    if (!originalMessage) {
        window?.log?.warn(`Cannot find the original reacted message ${serverId}.`);
        return;
    }
    if ((0, lodash_1.isEmpty)(reactions)) {
        if (originalMessage.get('reacts')) {
            originalMessage.set({
                reacts: undefined,
            });
        }
    }
    else {
        const reacts = {};
        Object.keys(reactions).forEach(key => {
            const emoji = decodeURI(key);
            const senders = {};
            reactions[key].reactors.forEach(reactor => {
                senders[reactor] = String(serverId);
            });
            reacts[emoji] = { count: reactions[key].count, index: reactions[key].index, senders };
        });
        originalMessage.set({
            reacts,
        });
    }
    await originalMessage.commit();
    return originalMessage;
};
exports.handleOpenGroupMessageReactions = handleOpenGroupMessageReactions;
const updateRecentReactions = async (reactions, newReaction) => {
    window?.log?.info('updating recent reactions with', newReaction);
    const recentReactions = new Reaction_1.RecentReactions(reactions);
    const foundIndex = recentReactions.items.indexOf(newReaction);
    if (foundIndex >= 0) {
        if (foundIndex === 0) {
            return;
        }
        recentReactions.swap(foundIndex);
    }
    else {
        recentReactions.push(newReaction);
    }
    await (0, storage_1.saveRecentReations)(recentReactions.items);
};
exports.updateRecentReactions = updateRecentReactions;
function assertLoaded() {
    if (cachedKnownMapping === null) {
        throw new Error('loadKnownBlindedKeys must be called on app start');
    }
    return cachedKnownMapping;
}
function isUsAnySogsFromCache(blindedOrNakedId) {
    const usUnblinded = utils_1.UserUtils.getOurPubKeyStrFromCache();
    if (!PubKey_1.PubKey?.isBlinded(blindedOrNakedId)) {
        return blindedOrNakedId === usUnblinded;
    }
    const found = assertLoaded().find(m => m.blindedId === blindedOrNakedId && m.realSessionId === usUnblinded);
    return Boolean(found);
}
exports.isUsAnySogsFromCache = isUsAnySogsFromCache;
