"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.wallet = void 0;
const child_process_1 = __importDefault(require("child_process"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const path_1 = __importDefault(require("path"));
const os_1 = __importDefault(require("os"));
const node_fetch_1 = __importDefault(require("node-fetch"));
const errors_1 = require("../bchat/utils/errors");
const request_promise_1 = __importDefault(require("request-promise"));
const http_1 = __importDefault(require("http"));
const portscanner_1 = __importDefault(require("portscanner"));
const cross_port_killer_1 = require("cross-port-killer");
const crypto = require('crypto');
const daemon_rpc_1 = require("./daemon-rpc");
const utils_1 = require("../bchat/utils");
const wallet_1 = require("../state/ducks/wallet");
const SCEE_1 = require("./SCEE");
const walletConfig_1 = require("../state/ducks/walletConfig");
const BchatWalletHelper_1 = require("./BchatWalletHelper");
const settings_key_1 = require("../data/settings-key");
const promise_queue_1 = __importDefault(require("promise-queue"));
class Wallet {
    heartbeat;
    wss;
    wallet_dir;
    auth;
    wallet_state;
    scee;
    id;
    agent;
    queue;
    backend;
    last_height_send_time;
    height_regexes;
    constructor() {
        this.heartbeat = null;
        this.wss = null;
        this.wallet_dir = '';
        this.auth = [];
        this.wallet_state = {
            open: false,
            name: '',
            balance: 0,
            unlocked_balance: 0,
            fiatCurrency: '',
            tx_list: [],
            password_hash: '',
        };
        this.id = 0;
        this.scee = new SCEE_1.SCEE();
        this.agent = new http_1.default.Agent({ keepAlive: true, maxSockets: 10 });
        this.queue = new promise_queue_1.default(1, Infinity);
        this.last_height_send_time = Date.now();
        this.height_regexes = [
            {
                string: /Processed block: <([a-f0-9]+)>, height (\d+)/,
                height: (match) => match[2],
            },
            {
                string: /Skipped block by height: (\d+)/,
                height: (match) => match[1],
            },
            {
                string: /Skipped block by timestamp, height: (\d+)/,
                height: (match) => match[1],
            },
            {
                string: /Blockchain sync progress: <([a-f0-9]+)>, height (\d+)/,
                height: (match) => match[2],
            },
        ];
    }
    startWallet = async (type) => {
        try {
            let getFiatCurrency = window.getSettingValue(settings_key_1.walletSettingsKey.settingsFiatCurrency);
            if (!getFiatCurrency) {
                window.setSettingValue(settings_key_1.walletSettingsKey.settingsFiatCurrency, 'USD');
            }
            let walletDir = await this.findDir();
            const rpcExecutable = process.platform === 'linux'
                ? '/beldex-wallet-rpc-ubuntu'
                : process.platform === 'win32'
                    ? '/beldex-wallet-rpc-windows'
                    : '/beldex-wallet-rpc-darwin';
            let __ryo_bin;
            if (process.env.NODE_ENV == 'production') {
                __ryo_bin = path_1.default.join(__dirname, '../../../bin');
            }
            else {
                __ryo_bin = path_1.default.join(__dirname, '../../bin');
            }
            const rpcPath = await path_1.default.join(__ryo_bin, rpcExecutable);
            if (!fs_extra_1.default.existsSync(rpcPath)) {
            }
            else {
            }
            if (!fs_extra_1.default.existsSync(walletDir)) {
                fs_extra_1.default.mkdirpSync(walletDir);
            }
            else {
            }
            const status = await this.runningStatus(64371);
            if (status == true) {
                if (type == 'settings') {
                    return;
                }
                (0, cross_port_killer_1.kill)(64371)
                    .then()
                    .catch((err) => {
                    throw new errors_1.HTTPError('beldex_rpc_port', err);
                });
                await this.walletRpc(rpcPath, walletDir);
            }
            else {
                await this.walletRpc(rpcPath, walletDir);
            }
        }
        catch (e) {
            console.log('exception during wallet-rpc:', e);
        }
    };
    runningStatus = (port) => {
        return portscanner_1.default
            .checkPortStatus(port, '127.0.0.1')
            .catch(() => 'closed')
            .then(async (status) => {
            if (status === 'closed') {
                return false;
            }
            else {
                return true;
            }
        });
    };
    walletRpc = async (rpcPath, walletDir) => {
        try {
            let currentDeamonLoc = window.getSettingValue('current-deamon');
            const currentDaemon = currentDeamonLoc ? currentDeamonLoc : window.currentDaemon;
            window.setSettingValue('syncStatus', false);
            let rescanStatusUpdate = false;
            window.inboxStore?.dispatch((0, walletConfig_1.updateWalletRescaning)(rescanStatusUpdate));
            if (!window.getSettingValue('balancevisibility')) {
                window.setSettingValue('balancevisibility', true);
            }
            const generateCredentials = await crypto.randomBytes(64 + 64 + 32);
            const auth = generateCredentials.toString('hex');
            this.auth = [
                auth.substr(0, 64),
                auth.substr(64, 64),
                auth.substr(128, 32),
            ];
            this.wallet_dir = `${walletDir}/bchat`;
            const option = [
                '--rpc-login',
                this.auth[0] + ':' + this.auth[1],
                '--rpc-bind-port',
                '64371',
                '--daemon-address',
                `${currentDaemon.host}:${currentDaemon.port}`,
                '--rpc-bind-ip',
                '127.0.0.1',
                '--log-level',
                '0',
                '--wallet-dir',
                `${walletDir}/bchat`,
                '--log-file',
                `${walletDir}/bchat/logs/wallet-rpc.log`,
                '--trusted-daemon'
            ];
            if (window.networkType == 'testnet') {
                option.push('--testnet');
            }
            const wallet = await child_process_1.default.spawn(rpcPath, option, { detached: true });
            wallet.stdout.on('data', data => {
                process.stdout.write(`Wallet: ${data}`);
                let lines = data.toString().split('\n');
                let match, height = null;
                for (const line of lines) {
                    for (const regex of this.height_regexes) {
                        match = line.match(regex.string);
                        if (match) {
                            height = regex.height(match);
                            break;
                        }
                    }
                }
                if (height && Date.now() - this.last_height_send_time > 1000) {
                    this.last_height_send_time = Date.now();
                    window.inboxStore?.dispatch((0, walletConfig_1.updateWalletHeight)(height));
                }
            });
            wallet.stdout.on('error', err => {
                process.stderr.write(`Wallet: ${err}`);
            });
            wallet.stdout.on('close', (code) => {
                process.stderr.write(`Wallet: exited with code ${code} \n`);
            });
        }
        catch (error) {
            console.log('failed to start wallet rpc', error);
        }
    };
    heartRpc = async (method, params = {}, timeout = 0) => {
        try {
            const options = {
                uri: `http://localhost:64371/json_rpc`,
                method: 'POST',
                json: {
                    jsonrpc: '2.0',
                    id: '0',
                    method: method,
                    params,
                },
                auth: {
                    user: this.auth[0],
                    pass: this.auth[1],
                    sendImmediately: false,
                },
                timeout,
            };
            let requestData = await (0, request_promise_1.default)(options);
            if (requestData.hasOwnProperty('error')) {
                if (requestData.error.code === -21) {
                    let walletDir = os_1.default.platform() === 'win32' ? `${this.findDir()}\\bchat` : `${this.findDir()}//bchat`;
                    fs_extra_1.default.emptyDirSync(walletDir);
                    requestData = await (0, request_promise_1.default)(options);
                }
            }
            if (requestData.hasOwnProperty('error')) {
                return {
                    method: method,
                    params: params,
                    error: requestData.error,
                };
            }
            return requestData;
        }
        catch (err) {
            console.log('ERR:', err);
        }
    };
    generateMnemonic = async (props) => {
        try {
            await this.heartRpc('create_wallet', {
                filename: props.displayName,
                language: 'English',
                password: props.password,
            });
            const getAddress = await this.heartRpc('get_address');
            const mnemonic = await this.heartRpc('query_key', { key_type: 'mnemonic' });
            if (!getAddress.hasOwnProperty('error') && !mnemonic.hasOwnProperty('error')) {
                localStorage.setItem('userAddress', getAddress.result.address);
                return mnemonic.result.key;
            }
        }
        catch (e) {
            console.log('exception during wallet-rpc:', e);
        }
    };
    async getHeigthFromDateAndUserInput(refreshDetails) {
        let restore_height;
        if (refreshDetails.refresh_type == 'date') {
            restore_height = await daemon_rpc_1.daemon.timestampToHeight(refreshDetails.refresh_start_timestamp_or_height);
            if (restore_height === false) {
                return utils_1.ToastUtils.pushToastError('invalidRestoreDate', window.i18n('invalidRestoreDate'));
            }
        }
        else {
            restore_height = Number.parseInt(refreshDetails.refresh_start_timestamp_or_height);
            if (!restore_height) {
                restore_height = 0;
            }
        }
        return restore_height;
    }
    restoreWallet = async (displayName, password, userRecoveryPhrase, refreshDetails, type) => {
        let restoreWallet;
        let restore_height = await this.getHeigthFromDateAndUserInput(refreshDetails);
        try {
            let walletDir = os_1.default.platform() === 'win32' ? `${this.findDir()}\\bchat` : `${this.findDir()}//bchat`;
            fs_extra_1.default.emptyDirSync(walletDir);
            restoreWallet = await this.heartRpc('restore_deterministic_wallet', {
                restore_height: restore_height,
                filename: displayName,
                password: password,
                seed: userRecoveryPhrase,
            });
            if (restoreWallet.hasOwnProperty('error')) {
                restoreWallet = await this.deleteWallet(displayName, password, userRecoveryPhrase, refreshDetails, type);
            }
            if (restoreWallet.hasOwnProperty('result')) {
                this.wallet_state.password_hash = this.passwordEncrypt(password);
                if (!type) {
                    console.log('killed...............:', type);
                    (0, cross_port_killer_1.kill)(64371)
                        .then(() => console.log('port kill successFull'))
                        .catch(err => {
                        throw new errors_1.HTTPError('beldex_rpc_port', err);
                    });
                }
            }
            return restoreWallet;
        }
        catch (error) {
            return { message: 'exception during wallet-rpc', error: error };
        }
    };
    async saveWallet() {
        await this.sendRPC('store');
    }
    async closeWallet() {
        await this.saveWallet();
        this.wallet_state.open = false;
        await this.sendRPC('close_wallet');
    }
    getLatestHeight = async () => {
        try {
            let url;
            if (window.networkType === 'mainnet') {
                url = 'http://publicnode1.rpcnode.stream:29095';
            }
            else {
                url = 'http://149.102.156.174:19095';
            }
            const response = await (0, node_fetch_1.default)(`${url}/get_height`, {
                method: 'POST',
                body: JSON.stringify({}),
            });
            if (!response.ok) {
                throw new errors_1.HTTPError('Beldex_rpc error', response);
            }
            const result = await response.json();
            return result.height;
        }
        catch (e) {
            throw new errors_1.HTTPError('exception during wallet-rpc:', e);
        }
    };
    validateAddres = async (address) => {
        const validateAddress = await this.sendRPC('validate_address', { address });
        if (validateAddress.hasOwnProperty('error')) {
            return false;
        }
        return validateAddress.result.valid;
    };
    deleteWallet = async (displayName, password, userRecoveryPhrase, refreshDetails, type) => {
        let walletDir = os_1.default.platform() === 'win32' ? `${this.findDir()}\\bchat` : `${this.findDir()}//bchat`;
        fs_extra_1.default.emptyDirSync(walletDir);
        return await this.restoreWallet(displayName, password, userRecoveryPhrase, refreshDetails, type);
    };
    findDir = () => {
        let walletDir;
        if (os_1.default.platform() === 'win32') {
            walletDir = `${os_1.default.homedir()}\\Documents\\Beldex`;
        }
        else {
            walletDir = path_1.default.join(os_1.default.homedir(), 'Beldex');
        }
        return walletDir;
    };
    startHeartbeat(type) {
        clearInterval(this.heartbeat);
        this.heartbeat = setInterval(async () => {
            this.heartbeatAction(type);
        }, 8000);
    }
    async heartbeatAction(type) {
        Promise.all([
            this.sendRPC('getheight', {}, 5000),
            this.sendRPC('getbalance', { account_index: 0 }),
        ]).then(async (data) => {
            let wallet = {
                info: {
                    height: 0,
                    balance: 0,
                    unlocked_balance: 0,
                    balanceConvert: 0,
                },
                transacations: {
                    tx_list: [],
                },
            };
            for (let n of data) {
                if (n.hasOwnProperty('error') || !n.hasOwnProperty('result')) {
                    if (n.error && n.error.code === -13) {
                    }
                    continue;
                }
                let response = n;
                if (n.method == 'getheight') {
                    wallet.info.height = response.result.height;
                    window.inboxStore?.dispatch((0, walletConfig_1.updateWalletHeight)(response.result.height));
                }
                else if (n.method == 'getbalance') {
                    let transacationsHistory = [];
                    transacationsHistory = await this.getTransactions();
                    transacationsHistory = transacationsHistory.transactions.tx_list;
                    if (this.wallet_state.balance == response.result.balance &&
                        this.wallet_state.unlocked_balance == response.result.unlocked_balance &&
                        this.wallet_state.tx_list == transacationsHistory) {
                        continue;
                    }
                    this.wallet_state.balance = wallet.info.balance = response.result.balance;
                    this.wallet_state.unlocked_balance = wallet.info.unlocked_balance =
                        response.result.unlocked_balance;
                    this.wallet_state.tx_list = transacationsHistory;
                    if (type == 'wallet') {
                        await this.getFiatBalance();
                    }
                    window.inboxStore?.dispatch(await (0, wallet_1.updateBalance)({
                        balance: this.wallet_state.balance,
                        unlocked_balance: this.wallet_state.unlocked_balance,
                        transacations: transacationsHistory,
                    }));
                }
            }
        });
    }
    getTransactions() {
        return new Promise(resolve => {
            this.sendRPC('get_transfers', {
                in: true,
                out: true,
                pending: true,
                failed: true,
                pool: true,
            }).then(async (data) => {
                if (data.hasOwnProperty('error') || !data.hasOwnProperty('result')) {
                    resolve({});
                    return;
                }
                await this.saveWallet();
                let wallet = {
                    transactions: {
                        tx_list: [],
                    },
                };
                const types = ['in', 'out', 'pending', 'failed', 'bns'];
                types.forEach(type => {
                    if (data.result.hasOwnProperty(type)) {
                        wallet.transactions.tx_list = wallet.transactions.tx_list.concat(data.result[type]);
                    }
                });
                wallet.transactions.tx_list.sort(function (a, b) {
                    if (a.timestamp < b.timestamp)
                        return 1;
                    if (a.timestamp > b.timestamp)
                        return -1;
                    return 0;
                });
                resolve(wallet);
            });
        });
    }
    getFiatBalance = async (currency) => {
        try {
            const fiatCurrency = currency
                ? currency.toLocaleLowerCase()
                : window.getSettingValue(settings_key_1.walletSettingsKey.settingsFiatCurrency)?.toLocaleLowerCase();
            const balance = this.wallet_state.balance;
            const response = await (0, node_fetch_1.default)(`https://api.coingecko.com/api/v3/simple/price?ids=beldex&vs_currencies=${fiatCurrency}`);
            const currencyValue = await response.json();
            const FiatBalance = response.ok && currencyValue.beldex[fiatCurrency]
                ? balance * currencyValue.beldex[fiatCurrency]
                : 0;
            window.inboxStore?.dispatch((0, walletConfig_1.updateFiatBalance)(FiatBalance));
        }
        catch (error) {
            console.log("Uncaught error Fiat Balance", error);
        }
    };
    transfer = async (address, amount, priority, isSweepAll) => {
        const rpc_endpoint = isSweepAll ? 'sweep_all' : 'transfer_split';
        const rpcSpecificParams = isSweepAll
            ? {
                address,
                account_index: 0,
                subaddr_indices_all: true,
            }
            : {
                destinations: [{ amount: amount, address: address }],
            };
        const params = {
            ...rpcSpecificParams,
            account_index: 0,
            priority: priority,
            ring_size: 7,
            get_tx_key: true,
        };
        const data = await this.sendRPC(rpc_endpoint, params);
        if (data.result) {
            return data;
        }
        else {
            return data;
        }
    };
    passwordEncrypt = (password) => {
        return crypto.pbkdf2Sync(password, this.auth[2], 1000, 64, 'sha512').toString('hex');
    };
    openWallet = async (filename, password) => {
        const openWallet = await this.heartRpc('open_wallet', {
            filename,
            password,
        });
        if (openWallet.hasOwnProperty('error')) {
            return openWallet;
        }
        let address_txt_path = path_1.default.join(this.wallet_dir, filename + '.address.txt');
        if (!fs_extra_1.default.existsSync(address_txt_path)) {
            this.sendRPC('get_address', { account_index: 0 }).then((data) => {
                if (data.hasOwnProperty('error') || !data.hasOwnProperty('result')) {
                    return;
                }
                fs_extra_1.default.writeFile(address_txt_path, data.result.address, 'utf8', () => { });
            });
        }
        this.wallet_state.name = filename;
        this.wallet_state.open = true;
        this.wallet_state.password_hash = this.passwordEncrypt(password);
        return openWallet;
    };
    chooseDaemon = async () => {
        let data = window.getSettingValue(settings_key_1.walletSettingsKey.settingsDeamonList);
        for (let i = 0; i < data.length; i++) {
            if (data[i].type == 'Remote') {
                const deamonStatus = await (0, BchatWalletHelper_1.workingStatusForDeamon)(data[i], 'daemonValidation');
                if (deamonStatus.status === 'OK') {
                    return data[i];
                }
            }
        }
    };
    sendRPC(method, params = {}, timeout = 0) {
        let id = this.id++;
        let options = {
            uri: `http://localhost:64371/json_rpc`,
            method: 'POST',
            json: {
                jsonrpc: '2.0',
                id: `${id}`,
                method: method,
            },
            auth: {
                user: this.auth[0],
                pass: this.auth[1],
                sendImmediately: false,
            },
            agent: this.agent,
        };
        if (Object.keys(params).length !== 0) {
            options.json.params = params;
        }
        if (timeout > 0) {
            options.timeout = timeout;
        }
        return this.queue.add(() => {
            return (0, request_promise_1.default)(options)
                .then((response) => {
                if (response.hasOwnProperty('error')) {
                    return {
                        method: method,
                        params: params,
                        error: response.error,
                    };
                }
                return {
                    method: method,
                    params: params,
                    result: response.result,
                };
            })
                .catch((error) => {
                return {
                    method: method,
                    params: params,
                    error: {
                        code: -1,
                        message: 'Cannot connect to wallet-rpc',
                        cause: error.cause,
                    },
                };
            });
        });
    }
    rescanBlockchain() {
        this.sendRPC('rescan_blockchain');
    }
    changeWalletPassword = async (old_password, new_password) => {
        const changePassword = await this.sendRPC('change_wallet_password', {
            old_password,
            new_password,
        });
        return changePassword;
    };
}
exports.wallet = new Wallet();
